/*
 * tourview.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// ----------------------------------------------------------------------------
// headers
// ----------------------------------------------------------------------------

#include "tourview.h"

#include <wx/toolbar.h>
#include <wx/statline.h>

#include "robotop.h"
#include "inirepdata.h"
#include "dialogs.h"
#include "insertview.h"
#include "detview.h"
#include "matrixview.h"
#include "wxini.h"
#include "wxext.h"

// Include the XPM resources
#include "bitmaps/addbot.xpm"
#include "bitmaps/rembot.xpm"
#include "bitmaps/details.xpm"
#include "bitmaps/refresh.xpm"
#include "bitmaps/options.xpm"
#include "bitmaps/web.xpm"
#include "bitmaps/matrix.xpm"


// ----------------------------------------------------------------------------
// event tables and other macros for wxWindows
// ----------------------------------------------------------------------------
// my event types
#if wxCHECK_VERSION(2,3,0)
DEFINE_EVENT_TYPE(EVT_TOUR_UPDATED)
DEFINE_EVENT_TYPE(EVT_TOUR_CLOSED)
#endif

// the event tables connect the wxWindows events with the functions (event
// handlers) which process them. It can be also done at run-time, but for the
// simple menu events like this the static method is much simpler.
BEGIN_EVENT_TABLE(TournamentFrame, wxxMDIChildFrame)
  EVT_MENU(wxID_CLOSE,          TournamentFrame::OnCloseCommand)
  EVT_MENU(TourFrame_AddBot,    TournamentFrame::OnAddBot)
  EVT_MENU(TourFrame_DelBot,    TournamentFrame::OnDelBot)
  EVT_MENU(TourFrame_Details,   TournamentFrame::OnDetails)
  EVT_MENU(TourFrame_Browser,   TournamentFrame::OnBrowser)
  EVT_MENU(TourFrame_Matrix,    TournamentFrame::OnMatrix)
  EVT_MENU(TourFrame_Redisplay, TournamentFrame::OnRedisplay)
  EVT_MENU(TourFrame_Options, TournamentFrame::OnOptions)
  EVT_LIST_ITEM_SELECTED(TourFrame_List, TournamentFrame::OnShowRobotInStatus) 
  EVT_LIST_ITEM_ACTIVATED(TourFrame_List, TournamentFrame::OnDetailsList) 
  EVT_CLOSE(TournamentFrame::OnCloseWindow)
END_EVENT_TABLE()

TournamentFrame::TournamentFrame( wxMDIParentFrame *parent, TourData* data)
  : wxxMDIChildFrame(parent, wxT("[Temporary Title]"), wxT("TournamentFrame")), 
    configData(data), insertFrame(0), parent(parent), tourEventHandlers()
{
	UpdateTitle();

	wxBoxSizer* sizer = new wxBoxSizer( wxVERTICAL );
	wxPanel* panel = new wxPanel(this); 

	wxToolBar* bar = new wxToolBar(panel, -1, wxDefaultPosition, wxDefaultSize, 
		wxNO_BORDER | wxTB_FLAT | wxTB_HORIZONTAL | wxTB_NODIVIDER);
	InitToolBar( bar );

	wxString myChartsXXX = GetMainTourFolder() + configData->tourFolder +
		wxFILE_SEP_PATH + wxT("results") + wxFILE_SEP_PATH + wxT("charts.xxx");
	chartsData = new ChartsReportData(myChartsXXX);
	chartsList = new wxxReportCtrl(panel, TourFrame_List, chartsData, 0, wxT("/Controls/Charts"));
	chartsData->SetSortColumn(0); // sort after rank (first column)
	chartsData->SortItems();

	wxxAddCloseBar(panel, this, sizer, bar, (wxCommandEventFunction) &TournamentFrame::OnCloseCommand);
	sizer->Add( chartsList,	1, wxEXPAND | wxALL, 2 );

	panel->SetSizer(sizer); 

	// size myself
	wxBoxSizer* mySizer = new wxBoxSizer(wxVERTICAL); 
	mySizer->Add(panel, 1, wxEXPAND | wxALL, 0); 
	SetSizer(mySizer);
	SetSizeHints(mySizer->GetMinSize()); // set size hints to honour mininum size

	SetMenuBar(CreateMenu());

	// check if we shall open the insertFrame automatically because there are still bots to simulate
	InsertFrame* ifr = AccessInsertFrame(false); 
	ifr->ScanForFiles(); 
	if(ifr->GetNumWaitingFiles()) {
		SetNeedMakehtml(); 
		ifr->Show(true); 
	}
}

TournamentFrame::~TournamentFrame()
{
	delete configData;
}

wxMenuBar* TournamentFrame::CreateMenu()
{
    // create a menu bar
    wxMenu *menuFile = new wxMenu;
    menuFile->Append(RoboTop_New,  _("&New Competition...\tCtrl-N"), _("Create a new competition"));
    menuFile->Append(RoboTop_Open, _("&Open Competition...\tCtrl-O"), _("Open a competition"));
	menuFile->Append(wxID_CLOSE,   _("&Close Competition\tCtrl-F4"), _("Close this competition"));
    menuFile->AppendSeparator();
    menuFile->Append(RoboTop_Quit, _("E&xit\tAlt-X"), _("Quit this program"));

	// mirrors the toolbar
	wxMenu* menuTour = new wxMenu;
	menuTour->Append(TourFrame_AddBot,		_("&Add Robots...\tCtrl-A"), _("Add robots to this competition"));
	menuTour->Append(TourFrame_DelBot,		_("&Remove Robot\tCtrl-X"), _("Remove the selected robot from this competition"));
	menuTour->Append(TourFrame_Details,		_("View &Details\tCtrl-D"), _("View the selected robot's detailed results"));
	menuTour->Append(TourFrame_Browser,		_("View in &Browser\tCtrl-B"), _("View the charts page in a web browser"));
	menuTour->Append(TourFrame_Matrix,		_("View &Matrix\tCtrl-M"), _("View the result matrix for this competition"));
	menuTour->Append(TourFrame_Redisplay,	_("Re&fresh\tCtrl-R"), _("Reload the charts from disk"));
	menuTour->Append(TourFrame_Options,		_("&Settings..."), _("Change simulation settings for this competition"));

	wxMenu* menuExtras = new wxMenu;
	menuExtras->Append(RoboTop_Settings, _("&Settings..."), _("Configure RoboTop settings"));
#ifdef __WXMSW__
	menuExtras->Append(RoboTop_Fullscreen, _("&Fullscreen\tF11"), _("Show RoboTop fullscreen"), true); // checkable
#endif

    // the "About" item should be in the help menu
    wxMenu *helpMenu = new wxMenu;
    helpMenu->Append(RoboTop_Help, _("Using RoboTop\tF1"), _("Show help for RoboTop"));
    helpMenu->Append(RoboTop_About, _("&About..."), _("Show about dialog"));

    // now append the freshly created menu to the menu bar...
    wxMenuBar *menuBar = new wxMenuBar();
    menuBar->Append(menuFile, _("&File"));
    menuBar->Append(menuTour, _("&Competition"));
	menuBar->Append(menuExtras, _("&Extras"));
    menuBar->Append(helpMenu, _("&Help"));

	return menuBar;
}

void TournamentFrame::InitToolBar(wxToolBar* toolBar)
{
	const int numTools = 7;
    wxBitmap* bitmaps[numTools];

    bitmaps[0] = new wxBitmap( addbot_xpm );
    bitmaps[1] = new wxBitmap( rembot_xpm );
    bitmaps[2] = new wxBitmap( refresh_xpm );
    bitmaps[3] = new wxBitmap( details_xpm );
    bitmaps[4] = new wxBitmap( options_xpm );
    bitmaps[5] = new wxBitmap( web_xpm );
    bitmaps[6] = new wxBitmap( matrix_xpm );

	wxSize bitmapSize(bitmaps[0]->GetWidth(), bitmaps[0]->GetHeight());
	toolBar->SetToolBitmapSize(bitmapSize);

    toolBar->AddTool(TourFrame_AddBot,    *(bitmaps[0]), _("Add Bot"), _("Add a robot to the competition"));
    toolBar->AddTool(TourFrame_DelBot,    *(bitmaps[1]), 
    	      _("Delete Bot"), _("Remove the selected robot from the competition"));
    toolBar->AddTool(TourFrame_Details, *(bitmaps[3]), 
		      _("View Details"), _("View Detailed Results for the selected robot"));
    toolBar->AddTool(TourFrame_Browser, *(bitmaps[5]), 
		      _("View in Browser"), _("View the charts in your web browser"));
    toolBar->AddTool(TourFrame_Matrix, *(bitmaps[6]), 
		      _("View Matrix"), _("View the result matrix of the charts"));
    toolBar->AddTool(TourFrame_Redisplay, *(bitmaps[2]),
		      _("Update Display"), _("Update the display with the newest simulation results"));
    toolBar->AddTool(TourFrame_Options, *(bitmaps[4]), 
		      _("Configure Competition"), _("Change simulation options for this competition"));

    toolBar->Realize();

    for (int i = 0; i < numTools; i++)
        delete bitmaps[i];
}

void TournamentFrame::SetNeedMakehtml()
{
	needMakeHtml = true; 
}

bool TournamentFrame::NeedsMakehtml()
{
	return needMakeHtml; 
}

const TourData& TournamentFrame::GetConfigData()
{
	return *configData;
}

InsertFrame* TournamentFrame::GetInsertFrame()
{
	return insertFrame;
}

InsertFrame* TournamentFrame::AccessInsertFrame(bool show)
{
	if(insertFrame) {
		if(show)
			insertFrame->Show(true); 
		return insertFrame;
	}
	else {
		insertFrame = new InsertFrame(this);
		if(show) 
			insertFrame->Show(true); 
		return insertFrame; 
	}
}

void TournamentFrame::UpdateTitle()
{
	SetTitle(configData->tourName + wxT(" - [") + configData->tourFile + wxT("]"));
}

void TournamentFrame::Redisplay(bool runMakeHtml)
{
    if(runMakeHtml)
        RunMakeHtml(true, true);
	chartsData->Reload();
}

void TournamentFrame::RunMakeHtml(bool recalc, bool all)
{
	wxString command = GetProgramFileName(wxT("makehtml"));
	
	wxString tourFolder = GetMainTourFolder() + configData->tourFolder + wxFILE_SEP_PATH;
	wxString cf = tourFolder + 	wxT("results") + wxFILE_SEP_PATH;
	wxString of = tourFolder + 	wxT("html") + wxFILE_SEP_PATH;
#ifdef __WXMSW__ // avoids the Windows "feature" which interpretes \" in command lines as " instead of \"
	cf += '\\';
	of += '\\';
#endif
	wxString recalcArgs = (recalc ? wxT("-recalc") : wxT(""));
	wxString allArgs = (all ? wxT("*.rob *.rbi") : wxT(""));
	wxString args = wxString::Format(wxT(" -batch -cf \"%s\" -of \"%s\" %s -name \"%s\" %s"), 
		cf.c_str(), of.c_str(), recalcArgs.c_str(), configData->tourName.c_str(), allArgs.c_str());
//	wxLogMessage("MakeHTML startup: %s%s", command.c_str(), args.c_str());
    // NOTE: They recommend to enclose wxExecute's in these wxEnableTopLevelWindows(XXX) functions
	//       to make it safer, but under wxGTK it seems actually to be LESS safe with them...
#ifdef __WXMSW__ 
	wxEnableTopLevelWindows(false);
	long ret = wxExecute(command + args, wxEXEC_SYNC);
	wxEnableTopLevelWindows(true);
#else
	long ret = wxExecute(command + args, wxEXEC_SYNC);
#endif
	if(!ret) {
		wxLogStatus(_("MakeHTML run was successful."));
		needMakeHtml = false; 
	}
	else {
		wxLogError(_("MakeHTML failed, exit code %ld."), ret);
	}
}

// my event handling
void TournamentFrame::AddTournamentListener(wxEvtHandler* handler)
{
	tourEventHandlers.Add(handler);
}

void TournamentFrame::RemoveTournamentListener(wxEvtHandler* handler)
{
	int index = tourEventHandlers.Index(handler);
	if(index != wxNOT_FOUND)
		tourEventHandlers.RemoveAt(index);
}

void TournamentFrame::InformListeners(int eventType, bool force)
{
	TournamentEvent evt(eventType);
	evt.force = force;
	evt.SetEventObject(this);
	for(size_t pos = 0; pos < tourEventHandlers.GetCount(); pos++)
		wxPostEvent((wxEvtHandler*) tourEventHandlers[pos], evt);
}

// event handlers (these functions should _not_ be virtual)
void TournamentFrame::OnAddBot(wxCommandEvent& event)
{
	InsertFrame* myInsert = AccessInsertFrame();
	if(myInsert->IsRunning())
	{
		wxMessageBox(_("RoboTour is already simulating for this competition.\nPlease wait until it is finished."), _("Sorry"));
		return; 
	}
	// have choose some bots...
	wxFileDialog myFd(this, _("Choose robot(s)"), wxxReadString(wxT(""), wxT("/InsertFolder")), wxT(""), 
		_("All Robot files|*.rob;*.rbi|Robot source files|*.rob|Binary robot files|*.rbi"),
		wxOPEN | wxFILE_MUST_EXIST | wxMULTIPLE);
	bool res = (myFd.ShowModal() == wxID_OK);
	if(!res) { 
		myInsert->Layout(); 
		return; 
	}
	wxxStoreString(myFd.GetPath(), wxT("/InsertFolder")); 
	wxArrayString paths;
	myFd.GetPaths(paths);
	if(paths.GetCount() == 0) {	wxMessageBox(wxT("Interesting: zero robots selected.")); return; }
	if(!myInsert->CopyFiles(paths))
		wxMessageBox(_("There was a problem while copying the files.\nCheck your free disk space and file permissions."),
			_("Warning"));
	myInsert->Layout(); 
}


void TournamentFrame::OnDelBot(wxCommandEvent& event)
{
	long item = chartsList->GetSelectedRow();
	if(item < 0) {
		wxLogError(_("Nothing to remove. Select a bot first."));
		return;
	}
//	wxMessageBox(wxString::Format(wxT("Did you select item %ld (%s)?"), item, chartsData->GetItemText(item, 1).c_str()));

	wxString command = GetProgramFileName(wxT("kickbot"));
	
	wxString tourFolder = GetMainTourFolder() + configData->tourFolder + wxFILE_SEP_PATH;
	wxString cf = tourFolder + 	wxT("results") + wxFILE_SEP_PATH;
	wxString prog = cf + chartsData->GetRowName(item);
#ifdef __WXMSW__ // avoids the Windows "feature" which interpretes \" in command lines as " instead of \"
	cf += '\\';
#endif
	wxString args = wxString::Format(wxT(" -batch -cf \"%s\" \"%s\""),
		cf.c_str(), prog.c_str());
	//wxLogMessage("KickBot startup: %s%s", command.c_str(), args.c_str());
	wxEnableTopLevelWindows(false);
	long ret = wxExecute(command + args, true);
	wxEnableTopLevelWindows(true);
	if(!ret) {
		RunMakeHtml(true, false);
		chartsData->Reload();
		wxLogStatus(_("Robot removed successfully."));
	}
	else {
		wxLogError(_("KickBot failed, exit code %ld."), ret);
	}
}

void TournamentFrame::OnShowRobotInStatus(wxListEvent& event)
{
	long sel = chartsList->GetSelectedRow();
	if(sel < 0) {
		return;
	}
	wxString botname = chartsData->GetRowName(sel);
	wxLogStatus(_("Robot file name: %s"), botname.c_str());
}

void TournamentFrame::OnRedisplay(wxCommandEvent& event)
{
    Redisplay(true);
}
void TournamentFrame::OnDetails(wxCommandEvent& event)
{
	long sel = chartsList->GetSelectedRow();
	if(sel < 0) {
		wxMessageBox(_("Select a robot first."), _("Which detailed results?"), wxICON_INFORMATION, this);
		return;
	}
	wxString botname = chartsData->GetRowName(sel);
	/* wxMDIParentFrame *parent, const wxString& botname, const TourData& data,
		   const wxPoint& pos = wxDefaultPosition, const long style = wxDEFAULT_FRAME_STYLE);
	*/
	DetailsFrame* df = new DetailsFrame(this, parent, botname);
	df->Show(); 
}

void TournamentFrame::OnBrowser(wxCommandEvent& event)
{
	if(needMakeHtml) 
		RunMakeHtml(true, true); 

	wxString myHtml = GetMainTourFolder() + configData->tourFolder + wxFILE_SEP_PATH
		+ wxT("html") + wxFILE_SEP_PATH + wxT("charts.html");
	if(!wxxShowInBrowser(myHtml))
	{
#ifdef __WXMSW__
		const wxChar* msg = _("Couldn't start your default browser.");
#else
		const wxChar* msg = _("Couldn't start your web browser. Please check the configuration at Extras -> Settings...");
#endif
		wxLogError(msg);
	}
}

void TournamentFrame::OnMatrix(wxCommandEvent& event)
{
	MatrixFrame* mf = new MatrixFrame(this, parent);
	mf->Show(); 
}

void TournamentFrame::OnOptions(wxCommandEvent& event)
{
	TourPrefsDialog* dlg = new TourPrefsDialog(this, tourPrefsModeChange, configData);
	bool ret = dlg->Go();
	dlg->Close();
	if(!ret) return; // cancel pressed in dlg
	UpdateTitle();
}

void TournamentFrame::OnCloseCommand(wxCommandEvent&)
{
	wxCloseEvent pseudoEvent;
	pseudoEvent.SetCanVeto(true);
	OnCloseWindow(pseudoEvent);
}

void TournamentFrame::OnCloseWindow(wxCloseEvent& event)
{
	// it is no longer necessary to ask if we want to stop RoboTour (if running)
	// - that is done by the insert frame. 
/*
	if(GetInsertFrame())
	{
		if(event.CanVeto() && GetInsertFrame()->IsRunning()) // still simulates! User probably doesn't want to close the tour. 
		{
			int res = wxMessageBox(wxT("RoboTour is still running!\n") 
				wxT("Do you really want to close the competition?"), wxT("Confirm"), wxYES_NO, this);
			if(res == wxNO)
			{
				event.Veto();
				return;
			}
		}
	}
*/
	bool cascade = false;

	if(event.CanVeto())
	{
		int res = wxMessageBox(_("Do you want to close all other views \n(e.g. details, matrix) for this competition, too?"), 
			_("Confirm Close"), wxYES_NO | wxCANCEL, this);
		if(res == wxCANCEL)
		{
			event.Veto();
			return;
		}
		else if(res == wxYES)
			cascade = true;
	}
	InformListeners(EVT_TOUR_CLOSED, cascade);

	if(GetInsertFrame())
	{
		GetInsertFrame()->chartsFrame = 0;
		if(cascade)
			GetInsertFrame()->OnCloseWindow(event);
	}

	Destroy();
}

/************* my event system **************/
IMPLEMENT_DYNAMIC_CLASS(TournamentEvent, wxEvent)
TournamentEvent::TournamentEvent(int eventType) : force(false) 
{ 
	SetId(-1); 
	SetEventType(eventType);
}

#if wxCHECK_VERSION(2,3,0)
TournamentEvent::TournamentEvent(const TournamentEvent& evt)
    : wxEvent(evt), force(evt.force)
{
}

wxEvent* TournamentEvent::Clone() const
{
    return new TournamentEvent(*this);
}
#else
void TournamentEvent::CopyObject(wxObject& destObject) const
{
	wxEvent::CopyObject(destObject);
	TournamentEvent& obj = (TournamentEvent&) destObject;
	obj.force = force;
}
#endif


/*********** ********////////// //////////////************* ***********////////////

static wxxIniColumnInfo* chartsColInfo[] = {
	new wxxIniColumnInfo(wxTRANSLATE("#"),      wxT("NONE"), 30, wxREPORT_RIGHT, wxREPORT_SORTNUMERIC),
	new wxxIniColumnInfo(wxTRANSLATE("Name"),   wxT("programname"), wxREPORT_AUTOSIZE),
	new wxxIniColumnInfo(wxTRANSLATE("Author"), wxT("author"), wxREPORT_AUTOSIZE),
	new wxxIniColumnInfo(wxTRANSLATE("Country"),wxT("country"), wxREPORT_AUTOSIZE),
	new wxxIniColumnInfo(wxTRANSLATE("Wins"),   wxT("wins"), 40, wxREPORT_RIGHT, wxREPORT_SORTNUMERIC),
	new wxxIniColumnInfo(wxTRANSLATE("Losses"), wxT("looses"), 40, wxREPORT_RIGHT, wxREPORT_SORTNUMERIC),
	new wxxIniColumnInfo(wxTRANSLATE("Ties"),   wxT("ties"), 40, wxREPORT_RIGHT, wxREPORT_SORTNUMERIC),
	new wxxIniColumnInfo(wxTRANSLATE("Points"), wxT("points"), 40, wxREPORT_RIGHT, wxREPORT_SORTNUMERIC),
};
const static int chartsColCount = 8;
static bool chartsColInit = false; 

ChartsOrderData::ChartsOrderData(long row, long points) : rank(0), row(row), points(points)
{
}

#include <wx/arrimpl.cpp> // this is a magic incantation which must be done!
WX_DEFINE_OBJARRAY(ArrayOfCODs);

int coDataSortByPoints(ChartsOrderData** d1, ChartsOrderData** d2)
{
	return (*d2)->points - (*d1)->points;
}

int coDataSortByRow(ChartsOrderData** d1, ChartsOrderData** d2)
{
	return (*d1)->row - (*d2)->row;
}


ChartsReportData::ChartsReportData(const wxString& chartsIni)
		: wxxIniReportData(chartsIni, chartsColCount, InitColumnInfo(chartsColInfo, chartsColCount, chartsColInit), 
		                  wxREPORT_DELNOTHING), coData()
{
	ReloadArray();
}

wxString ChartsReportData::GetItemText(long row, int col)
{
	if(col == 0) {
//		if(row < coData.GetCount())
			return wxString::Format(wxT("%ld"), coData[row].rank);
/*		else {
			wxLogStatus(wxT("not enough in coData (%ld >= %d)"), row, coData.GetCount());
			return wxT("0");
		} */
	}
	else
		return wxxIniReportData::GetItemText(row, col);
}

int ChartsReportData::CompareItems(long row1, long row2)
{
	if(GetSortColumn() == 0) {
		return (coData[row1].rank - coData[row2].rank) * GetSortOrder();
	}
	else
		return wxxIniReportData::CompareItems(row1, row2);

}

void ChartsReportData::ReloadArray()
{
	coData.Empty();
	for(int r = 0; r < numSections; r++)
		coData.Add(ChartsOrderData(r, sectionCache[r]->GetLong(wxT("points"))));
	coData.Sort(&coDataSortByPoints);
	for(int s = 0; s < numSections; s++)
		coData[s].rank = s+1;
	coData.Sort(&coDataSortByRow);
}

bool ChartsReportData::DoSectionCache()
{
	bool ret = wxxIniReportData::DoSectionCache();

	ReloadArray();

	return ret;
}
