/*
 * robotop.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */


// ----------------------------------------------------------------------------
// headers
// ----------------------------------------------------------------------------

#include "robotop.h"

#include "tourview.h"
#include "dialogs.h"
#include "wxext.h"
#include "package.h"


// ----------------------------------------------------------------------------
// resources
// ----------------------------------------------------------------------------
// the application icon
#if defined(__WXGTK__) || defined(__WXMOTIF__)
    #include "bitmaps/robotop.xpm"
#endif

// the version string
const wxChar* RoboTopVersion = wxT("3.21");

// ----------------------------------------------------------------------------
// event tables and other macros for wxWindows
// ----------------------------------------------------------------------------

// the event tables connect the wxWindows events with the functions (event
// handlers) which process them. It can be also done at run-time, but for the
// simple menu events like this the static method is much simpler.
BEGIN_EVENT_TABLE(MainFrame, wxMDIParentFrame)
    EVT_MENU(RoboTop_New,   MainFrame::OnNew)
    EVT_MENU(RoboTop_Open,  MainFrame::OnOpen)
    EVT_MENU(RoboTop_Quit,  MainFrame::OnQuit)
	EVT_MENU(RoboTop_Settings, MainFrame::OnSettings)
	EVT_MENU(RoboTop_Fullscreen, MainFrame::OnFullscreen)
    EVT_MENU(RoboTop_About, MainFrame::OnAbout)
    EVT_MENU(RoboTop_Help,  MainFrame::OnHelp)
	EVT_CLOSE(MainFrame::OnClose)
END_EVENT_TABLE()

// Create a new application object: this macro will allow wxWindows to create
// the application object during program execution (it's better than using a
// static object for many reasons) and also declares the accessor function
// wxGetApp() which will return the reference of the right type (i.e. RoboTopApp and
// not wxApp)
IMPLEMENT_APP(RoboTopApp)

// ============================================================================
// implementation
// ============================================================================

// ----------------------------------------------------------------------------
// the application class
// ----------------------------------------------------------------------------

// 'Main program' equivalent: the program execution "starts" here
bool RoboTopApp::OnInit()
{
	SetAppName(wxT("RoboTop"));

	// init translation
	InitTranslation(); 


	globalConfig = new wxConfig(GetAppName(), wxT("RoboComTeam"));
	wxConfig::Set(globalConfig);

    // create the main application window
    MainFrame* frame = new MainFrame(wxT("RoboTop"), wxDEFAULT_FRAME_STYLE | wxHSCROLL | wxVSCROLL );

    // and show it (the frames, unlike simple controls, are not shown when
    // created initially)
    frame->Show(TRUE);

	SetTopWindow(frame);
	SetVendorName(wxT("RoboCom Team"));
	SetExitOnFrameDelete(true);

	CheckConfig(); // asks for configuration if not yet done

    // success: wxApp::OnRun() will be called which will enter the main message
    // loop and the application will run. If we returned FALSE here, the
    // application would exit immediately.
    return TRUE;
}

void RoboTopApp::InitTranslation()
{
	bool locOk = myLocale.Init();

#ifdef EXTRA_DATA_DIR
	myLocale.AddCatalogLookupPathPrefix(wxString(wxT(EXTRA_DATA_DIR)) + wxFILE_SEP_PATH + wxT("locale")); 
#endif
	// look near the executable file also
	myLocale.AddCatalogLookupPathPrefix(wxxGetExecutableFolder() + wxT("locale"));

	
	locOk &= myLocale.AddCatalog(wxT("robotour"));
	// in some configurations, we bring out own version of the wxstd catalog
	myLocale.AddCatalog(wxT("robotour-wxstd")); 
}


bool RoboTopApp::CheckConfig()
{
	bool isOk = true;
	isOk &= wxFileExists(globalConfig->Read(wxT("RoboTourExecutable"), wxT("")));
	isOk &= wxDirExists(globalConfig->Read(wxT("TournamentFolder"), wxT("")));
	if(!isOk)
		return GlobalPrefsDialog::Get()->Go();
	return isOk;
}

RoboTopApp::~RoboTopApp()
{
	// doesn't work? seems to be already deleted
	//delete globalConfig; // writes things back
}

// ----------------------------------------------------------------------------
// main frame
// ----------------------------------------------------------------------------

// frame constructor
MainFrame::MainFrame( const wxString& title, const long style )
  : wxMDIParentFrame( /*parent*/0, /*id*/-1, title, wxDefaultPosition, wxDefaultSize, style)
{
    // set the frame icon
    SetIcon(wxICON(robotop));

    // create the menu
    SetMenuBar(CreateMenu());

	wxxRestoreRect(this, wxT("/Rect/MainFrame")); 

#if wxUSE_STATUSBAR
    CreateStatusBar(1);
    SetStatusText(_("Welcome to RoboTop!"));
#endif // wxUSE_STATUSBAR
}

wxMenuBar* MainFrame::CreateMenu()
{
    // create a menu bar
    wxMenu *menuFile = new wxMenu;
    menuFile->Append(RoboTop_New,  _("&New Competition...\tCtrl-N"), _("Create a new competition"));
    menuFile->Append(RoboTop_Open, _("&Open Competition...\tCtrl-O"), _("Open a competition"));
    menuFile->AppendSeparator();
    menuFile->Append(RoboTop_Quit, _("E&xit\tAlt-X"), _("Quit this program"));

	wxMenu* menuExtras = new wxMenu;
	menuExtras->Append(RoboTop_Settings, _("&Settings..."), _("Configure RoboTop settings"));
#ifdef __WXMSW__
	menuExtras->Append(RoboTop_Fullscreen, _("&Fullscreen\tF11"), _("Show RoboTop fullscreen"), true); // checkable
#endif

    // the _("About") item should be in the help menu
    wxMenu *helpMenu = new wxMenu;
    helpMenu->Append(RoboTop_Help, _("Using RoboTop\tF1"), _("Show help for RoboTop"));
    helpMenu->Append(RoboTop_About, _("&About...\tCtrl-A"), _("Show about dialog"));

    // now append the freshly created menu to the menu bar...
    wxMenuBar *menuBar = new wxMenuBar();
    menuBar->Append(menuFile, _("&File"));
	menuBar->Append(menuExtras, _("&Extras"));
    menuBar->Append(helpMenu, _("&Help"));

	return menuBar;
}

// event handlers

void MainFrame::OnNew(wxCommandEvent& WXUNUSED(event))
{
  TourData* data = new TourData();
  TourPrefsDialog* tpDialog = new TourPrefsDialog(this, tourPrefsModeNew, data);
  bool ret = tpDialog->Go();
  tpDialog->Close();
  if(!ret) { delete data; return; }
  ret &= data->Create();
  if(!ret) { delete data; return; }

  TournamentFrame *subframe = new TournamentFrame( this, data);
  subframe->Show(TRUE);
}
void MainFrame::OnOpen(wxCommandEvent& WXUNUSED(event))
{
	wxString tourFile = wxFileSelector(_("Choose competition file"), GetMainTourFolder(), wxT(""), wxT(".top"), 
		_("Competition files (*.top)|*.top"), wxOPEN, this);
	if(!tourFile) // nothing selected
		return;
	tourFile = tourFile.AfterLast(wxFILE_SEP_PATH);
	TourData* data = new TourData();
	data->tourFile = tourFile;
	if(!data->Read()) {
		wxLogError(_("Could not read competition file %s."), tourFile.c_str());
		delete data;
		return;
	}
	TournamentFrame* subframe = new TournamentFrame( this, data);
	subframe->Show(true);

}

void MainFrame::OnQuit(wxCommandEvent& WXUNUSED(event))
{
    // TRUE is to force the frame to close
    Close(TRUE);
}

void MainFrame::OnSettings(wxCommandEvent& WXUNUSED(event))
{
	GlobalPrefsDialog::Get()->Go();
}

void MainFrame::OnFullscreen(wxCommandEvent& event)
{
	bool fs = (event.IsChecked() ? true : false);
	ShowFullScreen(fs, wxFULLSCREEN_NOTOOLBAR | wxFULLSCREEN_NOBORDER | wxFULLSCREEN_NOCAPTION);
}

void MainFrame::OnHelp(wxCommandEvent& WXUNUSED(event))
{
	wxString myHelp = wxxFindHelpFile(wxT("robotop.html")); 
	if(!myHelp.size()) {
		wxMessageBox(_("The RoboTop help was not found!"), _("RoboTop Help"), wxICON_ERROR); 
	}
	else
		wxxShowInBrowser(myHelp);
}

void MainFrame::OnAbout(wxCommandEvent& WXUNUSED(event))
{
    wxString msg;
    msg.Printf( _("Welcome to RoboTop v%s\nThe competition manager for RoboTour.\nCopyright (c) Florian Fischer, 2002-2005.\nBuilt using %s"), RoboTopVersion, wxVERSION_STRING);

    wxMessageBox(msg, _("About RoboTop"), wxOK | wxICON_INFORMATION, this);
}

void MainFrame::OnClose(wxCloseEvent& event)
{
	wxxStoreRect(this, wxT("/Rect/MainFrame")); 
	Destroy();
}
