/*
 * rtfile.unix.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

#ifndef __LRT_FILE_UNIX__
#define __LRT_FILE_UNIX__

#include "rtfile.h"

#include <unistd.h>
#include <sys/stat.h>
#include <dirent.h>
#include <stdlib.h>
#include "rtcollect.h"
#include "rtstring.h"
#include "rtchar.h"

namespace lrt {

const int File::minNameCount = 0;
const char File::separatorChar = '/';
const String File::separator = "/";
const bool File::caseSensitive = true;

void File::initialize(const String &executableFileName)
{
	char *buf = new char[1000];
	getcwd(buf, 1000);
	String cwdFile(buf);
	if(!cwdFile.endsWith(separator))
	  cwdFile += separatorChar;
	currentFolder = new File(cwdFile);
	delete[] buf;
	executableFile = new File(executableFileName);

    // get $HOME
    char *home = getenv("HOME");
    if(home == NULL) // no $HOME found - will this ever happen???
    {
        homeFolder = new File(*currentFolder);
        File execFolder = executableFile->getParentFile();
        settingsFolder = new File(execFolder);
    }
    else
    {
        String myHome(home);
        myHome += separatorChar;
        homeFolder = new File(myHome);
        settingsFolder = new File(myHome + ".lrt/");
    }

}

bool File::createSingleFolder(const File& folder)
{
	String fname = folder.getName();
	if(!mkdir(fname.cStr(), 0755)) // return value 0 = ok
		return true;
	else return false;
}

// Unix does not need resolveExtra()
bool File::resolveExtra(File *file, const String &fname) const
{
	return false;
}

// if I knew how...
void File::fetchAttributes()
{
	String name = getName();
	const char *filename = name.cStr();
	bool noWrite = access(filename, W_OK);
	bool noRead  = access(filename, R_OK);
	attribs->canRead  = noRead  ? ATTRIB_FALSE : ATTRIB_TRUE;
	attribs->canWrite = noWrite ? ATTRIB_FALSE : ATTRIB_TRUE;
	struct stat status;
	if( !stat( filename, &status ) )
	{
		if( S_ISDIR( status.st_mode ) )
			attribs->isFolder = ATTRIB_TRUE;
		else
			attribs->isFolder = ATTRIB_FALSE;
		attribs->exists = ATTRIB_TRUE;
        attribs->size = (int) status.st_size;
        attribs->lastModified.sec = (int)status.st_ctime;
	}
	else
	{
		attribs->exists = ATTRIB_FALSE;
	}
}

bool File::isAbsolute(const String &fname)
{
	if(fname.startsWith("/"))
		return true;
	else return false;
}

String File::getAbsolutePrefix()
{
  return separator;
}


static int one (const struct dirent *unused)
{
  return 1;
}

Array<File> File::listImpl(const IFilenameFilter *filter)
{
	bool acceptAll = false;
	if(filter == 0) 
		acceptAll = true;
	
	File parentFile = (isFolder() ? (*this) : getParentFile());

	Vector<File> ret(0);

	struct dirent **entries;
	int n;
	n = scandir(parentFile.getName().cStr(), &entries, one, alphasort);
	if( n >= 0 )
	{
		for(int i = 0; i < n; ++i)
		{
			String name = entries[i]->d_name;
			
			if((name != ".") && (name != ".."))	// don't include them
			{
				if( entries[i]->d_type == DT_DIR ) 
					name += separatorChar;
				if( acceptAll || filter->accept(parentFile, name) )
					ret += File(parentFile, name);
			}
		}
	}
	return ret;
}

} // namespace


#endif
