/*
 * rtdelphi.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// robdelphi.cpp: Implementierung der Klasse DelphiDataInputStream.
//
//////////////////////////////////////////////////////////////////////

#include "rtdelphi.h"

#include "rtstreams.h"
#include "rtstring.h"

namespace lrt {

///////////////////// Variant /////////////////////////////////////

Variant::Variant() : curType(typeInt), i(0)
{}

Variant::Variant(int i) : curType(typeInt), i(i)
{}

Variant::Variant(double d) : curType(typeDouble), d(d)
{}

Variant::Variant(const String& str) : curType(typeString), str(new String(str))
{}


Variant::~Variant()
{
	if(curType == typeString)
		delete str;
}

Variant::Variant(const Variant& var) : curType(var.curType)
{
	switch(curType) {
	  case typeInt:
		i = var.i;
		break;
	  case typeDouble:
		d = var.d;
		break;
	  case typeString:
		str = new String(*var.str);
		break;
	}
}

Variant& Variant::operator=(const Variant& var)
{
	if(curType == typeString)
		delete str;

	curType = var.curType;
	switch(curType) {
	  case typeInt:
		i = var.i;
		break;
	  case typeDouble:
		d = var.d;
		break;
	  case typeString:
		str = new String(*var.str);
		break;
	}

	return *this;
}

void Variant::set(int i)
{
	if(curType == typeString)
		delete str;

	curType = typeInt;
	this->i = i;
}

void Variant::set(double d)
{
	if(curType == typeString)
		delete str;

	curType = typeDouble;
	this->d = d;
}

void Variant::set(const String& str)
{
	if(curType == typeString)
		delete this->str;

	curType = typeDouble;
	this->str = new String(str);
}

int Variant::getInt(int def) const
{
	switch(curType) {
	  case typeInt:
		return i;
	  case typeDouble:
		return (int)d;
	  case typeString:
		return str->intValue(def);
	}
	return 0; // unreachable
}

double Variant::getDouble(double def) const
{
	switch(curType) {
	  case typeInt:
		return (double)i;
	  case typeDouble:
		return d;
	  case typeString:
		return str->doubleValue(def);
	}
	return 0; // unreachable
}

String Variant::getString() const
{
	switch(curType) {
	  case typeInt:
		return String(i);
	  case typeDouble:
		return String(d);
	  case typeString:
		return String(*str);
	}
	return String(); // unreachable
}

Variant::VariantType Variant::getCurrentType() const
{
	return curType;
}


/////////////////// DelphiDataInputStream ////////////////////////

DelphiDataInputStream::DelphiDataInputStream(InputStream* in) : FilterInputStream(in)
{
}

DelphiDataInputStream::DelphiDataInputStream(const String& filename) :
	FilterInputStream(new FileInputStream(filename, false))
{
}


DelphiDataInputStream::~DelphiDataInputStream() 
{
}


Variant DelphiDataInputStream::readAny()
{
	int byte = read(); // prefix

	if(byte < 0) return Variant();

	else if((byte == 0x06) || (byte == 0x0C))
		return Variant(doReadString(byte));
	else if((byte >= 0x02) && (byte <= 0x04))
		return Variant(doReadInt(byte));
	else if(byte == 0x05)
		return Variant(doReadDouble(byte));

	else
		System::exit(75, "Unsupported Delphi format 0x" + String(byte, 16));

	return Variant(); // unreachable
}

String DelphiDataInputStream::readString()
{
	int byte = read(); // prefix

	if(byte < 0) return "";

	return doReadString(byte);
}

String DelphiDataInputStream::doReadString(int type)
{
	int byte, length;

	if(type == 0x06) // short string: length as byte
		length = read();
	else if(type == 0x0C) // long string: length as 32Bit
	{
		length = read();
		length += read() << 8;
		length += read() << 16;
		length += read() << 24;
	}
	else
		System::exit(76, "Unsupported Delphi String format 0x" + String(type, 16));

	String ret;
	
	for(int i = 0; i < length; i++)
	{
		byte = read();
		if(byte < 0) return ret; // file end
		ret += ((char)byte);
	}

	return ret;
}

int DelphiDataInputStream::readInt()
{
	int byte = read(); // prefix

	if(byte < 0) return 0;

	return doReadInt(byte);
}

int DelphiDataInputStream::doReadInt(int type)
{
	int ret;

	// 8-bit integer
	if(type == 0x02)
	{
		ret = read();
		if(ret & 0x80) // negative value
			ret |= 0xFFFFFF00;
		return ret;
	}
	// 16-bit integer
	else if(type == 0x03)
	{
		ret = read();
		ret += read() << 8;
		if(ret & 0x8000) // negative value
			ret |= 0xFFFF0000;
		return ret;
	}
	// 32-bit integer
	else if(type == 0x04)
	{
		ret = read();
		ret += read() << 8;
		ret += read() << 16;
		ret += read() << 24;
		return ret;
	}
	else
		System::exit(77, "Unsupported Delphi number format 0x" + String(type, 16));
	return 0; // unreachable
}

double DelphiDataInputStream::readDouble()
{
	int byte = read(); // prefix

	if(byte < 0) return 0;

	return doReadDouble(byte);
}

double DelphiDataInputStream::doReadDouble(int type)
{
	if(type == 0x05)
	{
		Array<unsigned char> ld(10);
		for(int b = 0; b < 10; b++)
			ld[b] = read();
		return ld2d(ld);
	}
	else
		System::exit(74, "Unsupported Delphi floating-point format 0x" + String(type, 16));

	return 0; // unreachable
}

double DelphiDataInputStream::ld2d(Array<unsigned char>& ld)
{
	if(sizeof(double) != 8) 
		System::exit(70, "IEEE754-compliant 8-byte double type required.");

	double ret;
	unsigned char* dd = ((unsigned char*)(&ret));

	ld.reverse(); // work on big endian

	// copy over
		 // sign & bias    // 2 exponent bits      // 4 exponent bits
	dd[0] = (ld[0] & 0xC0) | ((ld[0] & 0x03) << 4) | ((ld[1] & 0xF0) >> 4);
		 // 4 exponent bits     // 4 mantissa bits
	dd[1] = ((ld[1] & 0x0F) << 4) | ((ld[2] & 0x7F) >> 3);
	dd[2] = (ld[2] << 5) | (ld[3] >> 3);
	dd[3] = (ld[3] << 5) | (ld[4] >> 3);
	dd[4] = (ld[4] << 5) | (ld[5] >> 3);
	dd[5] = (ld[5] << 5) | (ld[6] >> 3);
	dd[6] = (ld[6] << 5) | (ld[7] >> 3);
	dd[7] = (ld[7] << 5) | (ld[8] >> 3);

	ld.reverse(); // return unchanged ld

	if(System::isLittleEndian())
		inverseByteOrder(dd, 8); // inverse double to match system b.o.

	return ret;
}

void DelphiDataInputStream::inverseByteOrder(unsigned char* data, int len)
{
  unsigned char t;
  int len_ = len - 1;
  for(int i = (len >> 1) - 1; i >= 0; i--)
  {
	t = data[i];
	data[i] = data[len_-i];
	data[len_-i] = t;
  }
}

///////////////// DelphiDataOutputStream //////////////////////////

DelphiDataOutputStream::DelphiDataOutputStream(OutputStream* out) : FilterOutputStream(out)
{
}

bool DelphiDataOutputStream::writeAny(const Variant& var)
{
	Variant::VariantType type = var.getCurrentType();
	
	switch(type) {
	  case Variant::typeInt:
		return writeInt(var.getInt());
	  case Variant::typeDouble:
		return writeDouble(var.getDouble());
	  case Variant::typeString:
		return writeString(var.getString());
	  default:
		System::message("Warning: Unknown Variant type " + String(type));
		return writeInt(var.getInt());
	}
}

bool DelphiDataOutputStream::writeString(const String& str)
{
	bool ret = true;
	int len = str.length();

	// write prefix
	if(len <= 255)
	{ // small string
		ret &= write(0x06);
		ret &= write(len);
	}
	else
	{ // large string
		ret &= write(0x0C);
		ret &= write4(len);
	}

	ret &= write((const Array<char>&)str);

	return ret;
}

bool DelphiDataOutputStream::writeInt(int num)
{
	bool ret = true;
	int abs = Math::abs(num);

	if(abs <= 0x7F)
	{ // 8bit int
		ret &= write(0x02);
		ret &= write(num);
	}
	else if(abs <= 0x7FFF)
	{ // 16bit int
		ret &= write(0x03);
		ret &= write(num);
		ret &= write(num >> 8);
	}
	else
	{ // 32bit int
		ret &= write(0x04);
		ret &= write4(num);
	}

	return ret;
}

bool DelphiDataOutputStream::writeDouble(double num)
{
	bool ret = true;

	Array<unsigned char> ld(10);
	d2ld(num, ld);

	ret &= write(0x05); // prefix
	for(int b = 0; b < 10; b++)
		ret &= write(ld[b]);

	return ret;
}


void DelphiDataOutputStream::d2ld(double d, Array<unsigned char>& ld)
{
	if(sizeof(double) != 8) 
		System::exit(70, "IEEE754-compliant 8-byte double type required.");

	unsigned char* dd = (unsigned char*)(&d);

	if(System::isLittleEndian())
		inverseByteOrder(dd, 8);

	// is exponent zero? (exp = 7 bits from byte0 & 3 from byte1)
	bool zero = ((dd[0] & 0x7F) == 0);
	zero &= ((dd[1] & 0xE0) == 0);

	// bias exponent sign extend: only if bias = 0 & top exponent bit = 1
					// bias = 0           // next bit = 1
	bool biasextend = (((dd[0] & 0x40) == 0) && ((dd[0] & 0x20) != 0));
	ld[0] = (biasextend ? 0x3C : 0x00);

	// copy over
		  // sign & bias    2 exponent bits
	ld[0] |= (dd[0] & 0xC0) | ((dd[0] & 0x30) >> 4);
		  // 8 exponent bits
	ld[1] = ((dd[0] & 0x0F) << 4) | (dd[1] >> 4);
		  // middle place
	ld[2] = (zero ? 0x00 : 0x80);
		   // 7 mantissa bits
	ld[2] |= ((dd[1] & 0x0F) << 3) | (dd[2] >> 5);
	ld[3] = ((dd[2] & 0x1F) << 3) | (dd[3] >> 5);
	ld[4] = ((dd[3] & 0x1F) << 3) | (dd[4] >> 5);
	ld[5] = ((dd[4] & 0x1F) << 3) | (dd[5] >> 5);
	ld[6] = ((dd[5] & 0x1F) << 3) | (dd[6] >> 5);
	ld[7] = ((dd[6] & 0x1F) << 3) | (dd[7] >> 5);
	ld[8] = ((dd[7] & 0x1F) << 3);
	ld[9] = 0;

	if(System::isLittleEndian())
		inverseByteOrder(dd, 8);  // return unchanged double

	ld.reverse(); // inverse long double to return little-endian.
}

void DelphiDataOutputStream::inverseByteOrder(unsigned char* data, int len)
{
  unsigned char t;
  int len_ = len - 1;
  for(int i = (len >> 1) - 1; i >= 0; i--)
  {
	t = data[i];
	data[i] = data[len_-i];
	data[len_-i] = t;
  }
}

bool DelphiDataOutputStream::write4(int num)
{
	bool ret = true;

	ret &= write(num);
	ret &= write(num >> 8);
	ret &= write(num >> 16);
	ret &= write(num >> 24);

	return ret;
}


} // namespace
