/*
 * rtresource.h
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** \file 
  * Manages included resources. There is the class Resource, which knows about all
  * available resources and provides access to them, and the class ResourceInputStream,
  * which can be used to handle a Resource just like a file on a hard disk. 
  * @see Resource
  * @see ResourceInputStream
  */

#ifndef __LRT_RESOURCE__
#define __LRT_RESOURCE__

struct lrt_resource_type {
	unsigned long length;
	char* name;
	unsigned char* data;
};

#include "rtstreams.h"

namespace lrt {

// fwd declare
template<class T> class StringMap;

/** The resource manager for libRT.
  * In LibRT, resources are files which have been translated to resource header
  * (<tt>*.rh</tt>) files and which have been compiled into the application executable.
  * The tool <tt>resmake</tt> is included with libRT to automate the file translation.
  * Resources can be accessed much like files, using input streams. 
  * @see getResource()
  * @see ResourceInputStream
  */
class Resource  
{
public:
	/** Finds a resource, by a given name. 
	  * This method first tries to interprete <tt>name</tt> as a file name,
	  * resolving it from the current folder. If that file exists, a <tt>FileInputStream</tt>
	  * to this file is returned. If the file doesn't exist, the internal resource table
	  * is searched for the resource. If it is found, a <tt>ResourceInputStream</tt> to 
	  * this resource is returned. If no such resource exists, NULL is returned. 
	  * <p>
	  * <b>Note</b> that resource names are case sensitive and should always be specified
	  * using forward slashes. (e.g. "<tt>images/mybutton.gif</tt>") */
	static InputStream* getResource(const String& name);
	static void addResource(const lrt_resource_type& res);

private:
	static StringMap<lrt_resource_type> resources;
	Resource() {}
	virtual ~Resource() {}
};

/** An input stream which receives data from a resource (a file compiled into the
  * application executable). <tt>ResourceInputStream</tt>s should be obtained using  
  * <tt>Resource::getResource()</tt>. */
class ResourceInputStream : public InputStream {
public:
	ResourceInputStream(const lrt_resource_type& res);
	virtual ~ResourceInputStream() {}
	/** Reads the next byte from the resource. Returns -1 if the resource's end is reached. */
	virtual int read();
	/** Checks if this resource's end has been reached.
	  * @return <tt>true</tt> if the resource has ended, <tt>false</tt> if there is still some data to read.
	  */
	virtual bool eos();
	/** Returns <tt>true</tt>, as ResourceInputStreams support marking. */
	virtual bool markSupported();
	/** Marks the current position in the resource stream. */
	virtual void mark();
	/** Resets the current position to the previously marked one, or to the beginning of 
	  * the resource if none was marked. */
	virtual void reset();

private:
	lrt_resource_type resource;
	unsigned long pos;
	unsigned long markPos;
};

} // namespace

#endif
