#include "robwxstruct.h"

#include <wx/thread.h>

#include <rtlist.h>

#include "robbase.h"
#include "robstrings.h"

using namespace rt; 

///////////////////////////// RequestQueue
RequestQueue::RequestQueue() : critSect(new wxCriticalSection)
{
}

RequestQueue::~RequestQueue()
{
	lrt::deleteAll(begin());
	delete critSect;
}

void RequestQueue::append(const Request& r)
{
	wxCriticalSectionLocker csl(*critSect);

	((lrt::List<Request*>*)this)->append(r.clone()); // calls the List variant because r.clone() is a pointer
}

Request* RequestQueue::removeFirst() // returned Request must be deleted
{
	wxCriticalSectionLocker csl(*critSect);

	if(length() == 0)
		return 0;
	else {
		Request* ret = first()->accessElement();
		remove(first());
		return ret;
	}	
}

void RequestQueue::clear()
{
	wxCriticalSectionLocker csl(*critSect);

	lrt::deleteAll(begin());
}


///////////////////////////////// BotInfo
BotInfo::BotInfo(Bot* bot)
{
	program = -2;
	type = 0;
	for(int i = 0; i < 4; i++)
		head[i] = -1;
	active = 1;

	if(bot)
	{
		program = bot->owner->programNum;
		type = bot->vars[botInstrSet];
		for(int t = 0; t < bot->tasks.length(); t++) {
			Task* ta = bot->tasks[t]; 
			rint curBank = ta->vars[taskCurBank];
			if(curBank >= bot->banks.length())
				continue;  // it might be invalid!
			head[ ta->vars[taskDir] ] = bot->banks[ curBank ]->owner->programNum;
		}
		active = bot->vars[botActive];
		banks = bot->vars[botNumBanks];
	}
}

bool BotInfo::operator==(const BotInfo& b) const
{
	if(program != b.program) return false;
	if(type != b.type) return false;
	for(int i = 0; i < 4; i++)
		if(head[i] != b.head[i])
			return false;
	if(active != b.active) return false;
	return true;
}

/*
bool BotInfo::operator=(const BotInfo& b)
{
	program = b.program;
	type = b.type;
	for(int i = 0; i < 4; i++)
		head[i] = b.head[i];
}
*/

/////////////////////// DebugInfo

bool DebugInfo::BankInfo::updateFrom(Bank* bank)
{
	if(id == (int)bank) {
		hasChanged = false;
		return hasChanged; // already up-to-date
	}
	
	hasChanged = true;
	id = (int)bank;
	owner = bank->owner->programNum;

	if(bank->isSecret)
	{
		name = "[Secret Bank]";
		instructions.clear();
	}
	else 
	{
		name = bank->name;
		instructions.clear();
		for(int i = 0; i < bank->instr.length(); i++)
			instructions[i] = bank->instr[i]->print(i);
	}
	return hasChanged;
}

// set trg = src, and set hasChanged = true, iff trg != src
#define ASSIGN_CHECK_CHANGE(trg, src) \
	if((trg) != (src)) {			  \
		hasChanged = true;			  \
		(trg) = (src);				  \
	}

bool DebugInfo::TaskInfo::updateFrom(Task* task)
{
	hasChanged = false;

	ASSIGN_CHECK_CHANGE(dir, task->vars[taskDir]);
	ASSIGN_CHECK_CHANGE(runBank, task->vars[taskCurBank]);
	ASSIGN_CHECK_CHANGE(instrPos, task->vars[taskCurInstr]);
	
	bool curInstrIsSecret = task->bot->banks[task->vars[taskCurBank]]->isSecret;
	lrt::String newInstrText;
	if(curInstrIsSecret) newInstrText = "<secret instruction>";
	else if(!task->curInstr) newInstrText = "<no instruction>";
	else newInstrText = task->curInstr->print(task->vars[taskCurInstr]);
	ASSIGN_CHECK_CHANGE(instrText, newInstrText);

	return hasChanged;
}

// set trg = src, and set flag = true, iff trg != src
#define ASSIGN_CHECK_CHANGE_EX(flag, trg, src) \
	if((trg) != (src)) {					   \
		flag = true;						   \
		(trg) = (src);						   \
	}

bool DebugInfo::updateFrom(Bot* bot, bool alive)
{
	bool hasChanged = false; // the return value
	
	ASSIGN_CHECK_CHANGE(this->alive, alive);
	ASSIGN_CHECK_CHANGE(botID, (int)bot);
	botInfo = BotInfo(bot);
	programName = bot->owner->headers["name"].value;

	// update tasks
	if(tasks.length() > bot->tasks.length())
		tasks.remove(bot->tasks.length(), tasks.length() - bot->tasks.length());
	for(int t = 0; t < bot->tasks.length(); t++)
		hasChanged |= tasks[t].updateFrom(bot->tasks[t]);

	// update banks
	if(banks.length() > bot->banks.length())
		banks.remove(bot->banks.length(), banks.length() - bot->banks.length());
	for(int b = 0; b < bot->banks.length(); b++)
		hasChanged |= banks[b].updateFrom(bot->banks[b]);
	
	// update [bot] variables
	namedVarsHasChanged = false;
	for(int v = 0; v < rt::botVarLength; v++) {
		ASSIGN_CHECK_CHANGE_EX(namedVarsHasChanged, namedVars[v], bot->vars[v]);
	}
	hasChanged |= namedVarsHasChanged;

	unnamedVarsHasChanged = false;
	for(int uv = 0; uv < bot->owner->sim->glob.maxVars; uv++) {
		ASSIGN_CHECK_CHANGE_EX(unnamedVarsHasChanged, unnamedVars[uv], bot->stdVars[uv+1]);
	}
	hasChanged |= unnamedVarsHasChanged;

	return hasChanged;
}
