#include <rtstring.h>
#include <rtcollect.h>
#include <rtsystem.h>
#include <rtfile.h>

#include "robvars.h"
#include "robmanage.h"
#include "robhtml.h"

// Have Resources => Have Fun!
#include "resources.rh"

// always include this file as the last one
#include <rtmain.h>

namespace rt {

class MkHtmlArgs{
public:
	MkHtmlArgs() : links(true), downloadLinks(true),
		compFolder(lrt::File::getCurrentFolder()), outFolder(lrt::File::getCurrentFolder()),
		files(0), compName("New Competition"), recalc(false), printHelp(false) {}

	bool links, downloadLinks;
	lrt::File compFolder, outFolder;
	lrt::Vector<lrt::String> files;
	lrt::String compName;
	bool recalc;
	bool printHelp;
};

MkHtmlArgs parseArgs(const lrt::Array<lrt::String>& args)
{
	MkHtmlArgs margs;

	for( int i = 0; i < args.length(); i++ )
	{
		if( args[i][0] == '-' )
		{
			lrt::String arg = args[i].substring(1);

			// competition "name"
			if(!arg.compareIgnoreCase("name"))
			{
				i++;
				if(i >= args.length())
					System::exit(1, "Parameter missing for argument -" + arg);
				margs.compName = args[i];
			}
			// "c"ompetition "f"older
			else if(!arg.compareIgnoreCase("cf"))
			{
				i++;
				if(i >= args.length())
					System::exit(1, "Parameter missing for argument -" + arg);
				// ensure that a path was given
				String temp = args[i];
				if(!temp.endsWith("/") && !temp.endsWith(File::separator))
					temp += File::separatorChar;

				margs.compFolder = temp;
			}
			// "o"utput "f"older
			else if(!arg.compareIgnoreCase("of"))
			{
				i++;
				if(i >= args.length())
					System::exit(1, "Parameter missing for argument -" + arg);
				// ensure that a path was given
				String temp = args[i];
				if(!temp.endsWith("/") && !temp.endsWith(File::separator))
					temp += File::separatorChar;

				margs.outFolder = temp;
			}
			// "batch" mode (non-interactive)
			else if(!arg.compareIgnoreCase("batch"))
			{
				System::setInteractive(false);
			}
			// create "no links"
			else if(!arg.compareIgnoreCase("nolinks"))
			{
				margs.links = false;
			}
			// create "no download" links
			else if(!arg.compareIgnoreCase("nodownload"))
			{
				margs.downloadLinks = false;
			}
			// recalculate the cahts
			else if(!arg.compareIgnoreCase("recalc"))
			{
				margs.recalc = true;
			}
			// "h"elp or "help"
			else if((arg == "h") || (arg == "H") || !arg.compareIgnoreCase("help"))
			{
				margs.printHelp = true;
			}

#ifdef __SYMBIAN32__
			// current folder
			else if(!arg.compareIgnoreCase("cd"))
			{
				i++; // look at next arg
				if(i >= args.length())
					System::exit(1, "Parameter missing for argument -" + arg);
				String newCurrentFolder = args[i];
				// check that a path was given
				char c = newCurrentFolder[newCurrentFolder.length() - 1];
				// no path => append slash
				if(!(c == '/') && !(c == File::separatorChar))
					newCurrentFolder += File::separatorChar;

				// get competition folders
				String compFolder = margs.compFolder.getLocalName();
				String outFolder = margs.outFolder.getLocalName();

				// set current folder
				File::setCurrentFolder(File(newCurrentFolder));

				// set competition folders
				margs.compFolder = File(compFolder);
				margs.outFolder = File(outFolder);
			}
#endif
			else
			{
				System::println("Warning: unknown parameter " + args[i]);
			}
		}
		else // file or file pattern
		{
			int starPos = args[i].indexOf('*'); // has got a star?
			if(starPos < 0) // no, so just add it
				margs.files += lrt::File(margs.compFolder, args[i]).getLocalName(margs.compFolder);
			else { // yes, so add all files matching the pattern
				lrt::Array<lrt::File> f(lrt::StarFilenameFilter::getFiles(
					lrt::File(margs.compFolder, args[i]).getName()));
				for(int fi = 0; fi < f.length(); fi++)
					margs.files += f[fi].getLocalName(margs.compFolder);
			}
		}
	}

	return margs;
}

void printHelp()
{
	lrt::System::println("Creates charts and detailed results for competitions");
	lrt::System::println("that were run using the top mode (-t) of RoboTour. ");
	lrt::System::println();

	lrt::System::println("Usage: makehtml [options] [prog1 [prog2 [...]]]");
	lrt::System::println(String("Version ") + version);
	lrt::System::println();
	lrt::System::println("Options:");
#ifdef __SYMBIAN32__
	lrt::System::println("  -cd F      Set current folder to F before running robotour.");
#endif
	lrt::System::println("  -name N     Sets the competition name to N (default: \"New Competition\")");
	lrt::System::println("  -cf   F     Sets the competition folder to F (default: current)");
	lrt::System::println("  -of   F     Sets the output folder to F (default: current)");
	lrt::System::println("  -nolinks    Does not create links between the pages. ");
	lrt::System::println("  -nodownload Does not create download links for the rob files.");
	lrt::System::println("  -recalc     Recalculates the charts (useful after kickbot). ");
 	lrt::System::println("  -h, -help   Print this help text."); 
	lrt::System::println("  progn       Any robot file name, or file pattern.");
	lrt::System::println("              These file names should be entered relative to the");
	lrt::System::println("              competition folder (see -cf). If no files are given,");
	lrt::System::println("              only the charts page, but no detailed pages is created.");
}

} // namespace

/** Since rtmain.h includes system includes, we may not import any namespace in
  * this file. Only in this file, really. */

int rtMain(const lrt::Array<lrt::String>& args)
{
	lrt::System::setAppName("MakeHTML");
	initAllResources();

	rt::MkHtmlArgs margs = rt::parseArgs(args);

	if(margs.printHelp)
	{
		rt::printHelp();
		return 0;
	}

	rt::HtmlCreator html(margs.compName, margs.compFolder, margs.outFolder, 
		margs.links, margs.downloadLinks);
	if(!html.createChartsHtml(margs.recalc)) 
	{
		lrt::System::println("warning: couldn't create charts html.");
		if(!margs.files.length()) // probably, no args at all
			rt::printHelp();
	}

	for(int f = 0; f < margs.files.length(); f++)
	{
		if(!html.createDetailedHtml(margs.files[f]))
			lrt::System::println("warning: couldn't create detailed html for " + 
				margs.files[f] + ".");
	}

	return 0;
}

