//// collect.templ.cpp : Template-Implementierungen

#ifndef __LRT_COLLECT_TEMPL__
#define __LRT_COLLECT_TEMPL__

#include "rtsystem.h"
#include "rtstring.h"

namespace lrt {


///////////// Array<T>

template<class T> Array<T>::Array(int length) : len(length), data(0) 
{ 
	if(len != 0)
		data = new T[length];
}

template<class T> Array<T>::Array(const Array<T>& arr) : len(arr.len), data(0)
{
    if(len != 0) {
		data = new T[arr.len];
		copy(&arr, 0, this, 0, arr.len);
	}
}

template<class T> Array<T>::Array(const T* ptr, int off, int len) : len(len), data(0)
{
	if(len != 0) {
		data = new T[len];
		for(int i = 0; i < len; i++)
			data[i] = ptr[off + i];
	}
}

template<class T> Array<T>::~Array()
{ 
	delete [] data; 
}

template<class T> int Array<T>::length() const 
{ 
	return len;
}

template<class T> const T& Array<T>::operator[](int i) const 
{ 
   return data[checkConst(i)]; 
}

template<class T> T& Array<T>::operator[](int i)
{ 
   return data[check(i)];
}

template<class T> Array<T>& Array<T>::operator= (const Array<T>& arr)
{
   copy(&arr, 0, this, 0, arr.len);
   return *this;
}

template<class T> int Array<T>::indexOf(const T& elem, int pos) const
{
	for(int i = Math::max(0, pos); i < len; i++)
		if(data[i] == elem)
			return i;
	return -1; 
}

template<class T> const T* Array<T>::getData() const
{
	return data;
}



template<class T> void Array<T>::exchange(int p1, int p2)
{
	checkConst(p1);
	checkConst(p2);
	T temp(data[p1]);
	data[p1] = data[p2];
	data[p2] = temp;
}

template<class T> void Array<T>::reverse()
{
	T t;
	int len_ = len - 1;
	for(int i = (len >> 1) - 1; i >= 0; i--)
	{
		t = data[i];
		data[i] = data[len_-i];
		data[len_-i] = t;
	}
}

template<class T> void Array<T>::sort()
{
	sort(stdCompare<T>);
}

template<class T> void Array<T>::sort(int (*compareFun)(const T&, const T&), int l, int r)
{
	if(r >= len) // for the default parameter
		r = len - 1;

	if(r <= l) return; // stop recursion, all done
	
	checkConst(l);
	checkConst(r);

	if(r - l > 5) // for large parts, use the middle-element as pivot to avoid deep recursion for nearly sorted arrays
		exchange((l + r) / 2, r);

	T& pivot = data[r]; // use rightest element as pivot

	int i = l-1;
	int j = r;

	while(true)
	{
		while(compareFun(data[++i], pivot) < 0);
		while((j > l) && (compareFun(data[--j], pivot) > 0));
		if(i >= j)
			break;
		else
			exchange(i, j);
	}

	exchange(i, r);
	sort(compareFun, l, i-1);
	sort(compareFun, i+1, r);
}


template<class T> int Array<T>::check(int i) 
{ 
#ifndef __NOCHECK__
   if (i < 0 || i >= len){
	   String err("Array Index out of bounds: ");
	   err += i;
	   err += String(" >= ");
	   err += len;
	   System::exit(-1, err);
   }
#endif
   return i; 
}

template<class T> int Array<T>::checkConst(int i) const
{ 
#ifndef __NOCHECK__
   if (i < 0 || i >= len){
	   String err("Array Index out of bounds: ");
	   err += i;
	   err += String(" >= ");
	   err += len;
	   System::exit(-1, err);
   }
#endif
   return i; 
}

template<class T> void Array<T>::copy(const Array<T> *source, int srcPos, Array<T> *dest, int destPos, int length)
{
   if(length == 0) return; // there's nothing to do
   
   source->checkConst(srcPos);
   source->checkConst(srcPos + length - 1);
   dest->check(destPos);
   dest->check(destPos + length - 1);

   if(destPos < srcPos)
	   for(int i = 0; i < length; i++)
		   dest->data[destPos + i] = source->data[srcPos + i];
   else
	   for(int i = length - 1; i >= 0; i--)
		   dest->data[destPos + i] = source->data[srcPos + i];
}

/////////// Vector<T>

template<class T> Vector<T>::Vector(int length) : Array<T>(up2pow(length))
{
	capacity = len;
	len = length;
}
template<class T> Vector<T>::Vector(const Vector<T>& vect) : Array<T>(vect.capacity)
{
	capacity = len;
	len = vect.len;
	copy(&vect, 0, this, 0, vect.len);
}

template<class T> const T& Vector<T>::operator[](int i) const 
{ 
   return data[checkConst(i)]; 
}

template<class T> T& Vector<T>::operator[](int i)
{ 
	T& ret = data[check(i)];
	if(len <= i)
	  len = i+1;
	return ret;
}

template<class T> Vector<T>& Vector<T>::operator= (const Vector<T>& arr)
{
	copy(&arr, 0, this, 0, arr.len);
	len = arr.len;
	return *this;
}


template<class T> Vector<T>& Vector<T>::operator +=(const T &elem)
{
	(*this)[len] = elem;
	return *this;
}

template<class T> Vector<T>& Vector<T>::operator += (const Array<T> &arr)
{
	copy(&arr, 0, this, len, arr.length());
	return *this;
}

template<class T> void Vector<T>::insert(T elem, int before)
{
	copy(this, before, this, before + 1, len - before);
	(*this)[before] = elem;
}

template<class T> void Vector<T>::remove(int pos)
{
	copy(this, pos + 1, this, pos, len - (pos + 1));
	len--;
}

template<class T> void Vector<T>::remove(int start, int length)
{
	copy(this, start + length, this, start, len - (start + length));
	len -= length;
}

template<class T> void Vector<T>::clear()
{
	len = 0;
}

template<class T> inline int Vector<T>::up2pow(int val)
{
	int ret = 16;
	while(ret < val)
		ret <<= 1;
	return ret;
}

template<class T> int Vector<T>::check(int i)
{
#ifndef __NOCHECK__
	if(i < 0){
	   String err = "Vector Index out of bounds: ";
	   err += i;
	   err += " > ";
	   err += len;
	   System::exit(-1, err);
	}
#endif
	if(i < capacity) 
		return i;
	// need growing
	int newcap = up2pow(i+1);
	T* newdata = new T[newcap];
	for(int j = 0; j < len; j++)
		newdata[j] = data[j];
	delete [] data;
	data = newdata;
	capacity = newcap;
	return i;
}

template<class T> void Vector<T>::copy(const Array<T> *source, int srcPos, Vector<T> *dest, int destPos, int length)
{
  if(length <= 0) return;    
	if((destPos + length) > dest->len)
	{
		dest->check(destPos + length - 1); // enlarge the dest vector if needed
		dest->len = destPos + length;
	}
	Array<T>::copy(source, srcPos, dest, destPos, length);
}

template<class LRT_NPTR(T)> void deleteAt(Vector<LRT_PTR(T)>& vect, int index)
{
	const Vector<LRT_PTR(T)>& constVect = vect;
	delete constVect[index];
	vect.remove(index);
}

template<class LRT_NPTR(T)> void deleteAll(Vector<LRT_PTR(T)>& vect)
{
	for(int i = 0; i < vect.length(); i++)
		delete vect[i];
	vect.clear();
}

} // namespace

#endif
