# This file is part of qVamps.
#
# qVamps is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# qVamps is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qVamps; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


use strict;
use warnings;


package TitleTableItem;
use Qt;
use Qt::isa qw (Qt::ComboTableItem);
use Qt::attributes qw (ct);


sub NEW
{
  my $this  = shift;
  my $table = shift;
  my $list  = shift;
  my $ttn   = shift;

  $this -> SUPER::NEW ($table, $list);

  set_current_title ($ttn);
}


sub DESTROY
{
  ct = 0;

  SUPER -> DESTROY ();
}


sub current_title
{
  return ct;
}


sub set_current_title
{
  my $ttn = shift;

  setCurrentItem ("$ttn");
  ct = currentText ();
}


sub update_item_list
{
  my $item_list = shift;

  setStringList ($item_list);
  set_current_title (ct);
}


sub title_changed
{
  my $ttn = currentText ();

  return 0 if ($ttn == ct);

  ct = $ttn;

  return $ttn;
}


1;


package StreamTableItem;
use Qt;
use Qt::isa qw (Qt::ComboTableItem);
use Qt::attributes qw (streams);
use QVamps qw (search);


sub NEW
{
  my $this      = shift;
  my $table     = shift;
  my $item_hash = shift;
  my $current   = shift;

  my @streams   = sort { $a <=> $b } keys %{$item_hash};
  my @item_list = map $item_hash -> {$_}, @streams;

  $this -> SUPER::NEW ($table, \@item_list);

  streams = \@streams;

  set_current_stream ($current);
}


sub DESTROY
{
  streams = undef;

  SUPER -> DESTROY ();
}


sub current_stream
{
  return streams -> [currentItem ()];
}


sub set_current_stream
{
  my $stream = shift;

  # search stream in stream list
  my $i = search (streams, $stream);

  # make stream selected item if found
  setCurrentItem ($i) unless ($i < 0);
}


sub update_item_list
{
  my $item_hash = shift;

  # save stream# of currently selected item
  my $stream = current_stream ();

  # update combo box item list
  my @streams   = sort { $a <=> $b } keys %{$item_hash};
  my @item_list = map $item_hash -> {$_}, @streams;
  streams       = \@streams;

  setStringList (\@item_list);

  # restore user's stream selection
  set_current_stream ($stream);
}


1;


package LeftAlignedTableItem;
use Qt;
use Qt::isa qw (Qt::TableItem);


sub NEW
{
  my $this = shift;

  $this -> SUPER::NEW (@_);
}


sub DESTROY
{
  SUPER -> DESTROY ();
}


sub alignment
{
  return &Qt::AlignLeft|&Qt::AlignVCenter;
}


1;


package MenuTableItems;
use Qt;
use Qt::isa qw (Qt::Object);
use Qt::attributes qw (menu_table title_list_items
		       dflt_audio_stream dvd selected_titles item_refs);
use Qt::slots valueChanged  => [ "int", "int" ];
use Iso639;
use QVamps qw (read_setting tr);
use TitleTableItem;
use StreamTableItem;
use LeftAlignedTableItem;
use MenuPreview;
use constant
{
  TITLE_COL    => 0,
  AUDIO_COL    => 1,
  SUBTITLE_COL => 2,
  LABEL_COL    => 3,
  MENU_COL     => 4
};


sub NEW
{
  my $this             = shift;
  my $menu_table       = shift;
  my $name             = shift;
  my $title_list_items = shift;
  my $dvd              = shift;

  $this -> SUPER::NEW ($menu_table, $name);

  menu_table        = $menu_table;
  title_list_items  = $title_list_items;
  dflt_audio_stream = 0;
  dvd               = $dvd;
  item_refs         = [];

  this -> connect ($menu_table,
		   SIGNAL "valueChanged(int, int)",
		   SLOT   "valueChanged(int, int)");
}


sub DESTROY
{
#  print STDERR "MenuTableItems::DESTROY\n";

  item_refs         = undef;
  selected_titles   = undef;
  dvd               = undef;
  dflt_audio_stream = 0;
  title_list_items  = undef;
  menu_table        = undef;

  SUPER -> DESTROY ();
}


sub remove_all
{
#  print STDERR "MenuTableItems::remove_all\n";

  my $table    = menu_table;
  my $num_rows = $table -> numRows ();

  $table -> setUpdatesEnabled (0);

  while ($num_rows > 0)
  {
    remove_row (--$num_rows);
  }

  $table -> setUpdatesEnabled (1);

  parent () -> removeChild (this);

  item_refs         = undef;
  selected_titles   = undef;
  dvd               = undef;
  dflt_audio_stream = 0;
  title_list_items  = undef;
  menu_table        = undef;
}


sub update
{
  my $add_unseen = shift;

  my $table           = menu_table;
  my $selected_titles = title_list_items -> selected_titles ();
  selected_titles     = $selected_titles;
  my $next_label      = 1;
  my @titles;

  for (my $row = 0; $row < $table -> numRows (); $row++)
  {
    my $ttn = $table -> item ($row, &TITLE_COL) -> current_title ();

    if (grep ($_ == $ttn, @{$selected_titles}))
    {
      # title still selected --> update row's
      # combo box item lists and mark title as seen
      update_row ($row, $ttn);
      $titles [$ttn] = 1;

      my $label   = $table -> text ($row, &LABEL_COL);
      $next_label = $label + 1 if ($label =~ /^\d+$/ &&
				   $label >= $next_label);
    }
    else
    {
      # title de-selected --> remove menu entry
      remove_row ($row--);
    }
  }

  my $num_rows = $table -> numRows ();

  if ($add_unseen)
  {
    # add rows for titles not seen yet
    foreach my $ttn (@{$selected_titles})
    {
      insert_row ($num_rows++, $ttn,
		  $next_label++) unless ($titles [$ttn]);
    }
  }

  for (my $col = 0; $col < $table -> numCols (); $col++)
  {
    $table -> adjustColumn ($col);
  }
}


sub remove_row
{
  my $row = shift;

  my $refs = item_refs;

  foreach my $item (@{$refs -> [$row]})
  {
    menu_table -> takeItem ($item) if ($item);
  }

  menu_table -> removeRow ($row);
  splice @{$refs}, $row, 1;
}


sub insert_row
{
  my $row   = shift;
  my $ttn   = shift;
  my $label = shift;

  my $table = menu_table;

  $table -> insertRows ($row);

  my $audio_items    = make_audio_item_hash ($ttn);
  my $subtitle_items = make_subtitle_item_hash ($ttn);

  my $tcell = TitleTableItem ($table, selected_titles, $ttn);
  my $acell = StreamTableItem ($table, $audio_items, dflt_audio_stream);
  my $scell = StreamTableItem ($table, $subtitle_items, 0);
  my $lcell = LeftAlignedTableItem ($table, &Qt::TableItem::WhenCurrent,
				    $label);
  my $mcell = LeftAlignedTableItem ($table, &Qt::TableItem::WhenCurrent,
				    (sprintf "%s %d", &tr ("Title"), $ttn));

  $table -> setItem ($row, &TITLE_COL,    $tcell);
  $table -> setItem ($row, &AUDIO_COL,    $acell);
  $table -> setItem ($row, &SUBTITLE_COL, $scell);
  $table -> setItem ($row, &LABEL_COL,    $lcell);
  $table -> setItem ($row, &MENU_COL,     $mcell);

  $table -> adjustRow ($row);

  my $refs = item_refs;

  splice @{$refs}, $row, 0, [];

  $refs -> [$row] [&TITLE_COL]    = $tcell;
  $refs -> [$row] [&AUDIO_COL]    = $acell;
  $refs -> [$row] [&SUBTITLE_COL] = $scell;
  $refs -> [$row] [&LABEL_COL]    = $lcell;
  $refs -> [$row] [&MENU_COL]     = $mcell;
}


sub update_row
{
  my $row = shift;
  my $ttn = shift;

  my $table = menu_table;

  my $tcell = $table -> item ($row, &TITLE_COL);
  my $acell = $table -> item ($row, &AUDIO_COL);
  my $scell = $table -> item ($row, &SUBTITLE_COL);

  $tcell -> update_item_list (selected_titles);
  $acell -> update_item_list (make_audio_item_hash ($ttn));
  $scell -> update_item_list (make_subtitle_item_hash ($ttn));
}


sub make_audio_item_hash
{
  my $ttn = shift;

  my $streams = title_list_items -> selected_audio_streams ($ttn);
  my $dvd     = dvd;
  my $tsn     = $dvd -> title_set_nr ($ttn);
  my %rv;

  my $high          = 0;
  my $best          = 0;
  my $pref_code     = read_setting ("/Menu/Preferences/audio_language");
  my $pref_format   = read_setting ("/Menu/Preferences/audio_format");
  my $pref_channels = read_setting ("/Menu/Preferences/audio_channels");

  foreach my $stream (@{$streams})
  {
    my $score     = 0;
    my $code      = $dvd -> audio_lang_code ($tsn, $stream);
    my $lang      = Iso639::code2lang ($code);
    $lang         = sprintf "%s \"%s\"",
                    &tr ("Language code"), $code unless ($lang);
    my $format    = $dvd -> audio_format ($tsn, $stream);
    my $channels  = $dvd -> audio_channels ($tsn, $stream);
    $rv {$stream} = sprintf "%s, %s %s", $lang, &tr ($format), &tr ($channels);

    if ($code eq $pref_code)
    {
      $score++;

      if ($format eq $pref_format)
      {
	$score++;
	$score++ if ($channels eq $pref_channels)
      }
    }

    if ($score > $high)
    {
      $high = $score;
      $best = $stream;
    }
  }

  dflt_audio_stream = $best;

  $rv {0} = &tr ("None", "Audio") unless (%rv);

  return \%rv;
}


sub make_subtitle_item_hash
{
  my $ttn = shift;

  my $streams = title_list_items -> selected_subtitle_streams ($ttn);
  my $dvd     = dvd;
  my $tsn     = $dvd -> title_set_nr ($ttn);
  my %rv;

  foreach my $stream (@{$streams})
  {
    my $code      = $dvd -> subtitle_lang_code ($tsn, $stream);
    my $lang      = Iso639::code2lang ($code);
    $lang         = sprintf "%s \"%s\"",
                    &tr ("Language code"), $code unless ($lang);
    $rv {$stream} = $lang;
  }

  $rv {0} = &tr ("None", "Subtitle");

  return \%rv;
}


sub default_audio_stream
{
  return dflt_audio_stream;
}


sub valueChanged : SLOT(int, int)
{
  my $row = shift;
  my $col = shift;

  if ($col == &TITLE_COL)
  {
    my $table = menu_table;
    my $item  = $table -> item ($row, $col);
    my $ttn   = $item -> title_changed ();

    if ($ttn)
    {
      my $acell = $table -> item ($row, &AUDIO_COL);
      my $scell = $table -> item ($row, &SUBTITLE_COL);

      $acell -> update_item_list (make_audio_item_hash ($ttn));
      $acell -> set_current_stream (dflt_audio_stream);
      $scell -> update_item_list (make_subtitle_item_hash ($ttn));
      $scell -> set_current_stream (0);
    }
  }
}


sub up_item
{
  my $table = menu_table;
  my $row   = $table -> currentRow ();

  return unless ($row > 0);

  $table -> swapRows ($row - 1, $row);

  my $refs            = item_refs;
  my $tmp             = $refs -> [$row - 1];
  $refs -> [$row - 1] = $refs -> [$row];
  $refs -> [$row]     = $tmp;

  $table -> clearSelection ();
  $row--;

  # selectRow does not work for some reason
  $table -> selectCells ($row, 0, $row, 4);
  $table -> updateContents ();

  my $rect = $table -> cellGeometry ($row, $table -> currentColumn ());
  $table -> ensureVisible ($rect -> left (), $rect -> top ());
}


sub down_item
{
  my $table = menu_table;
  my $row   = $table -> currentRow ();

  return if ($row > $table -> numRows () - 2);

  $table -> swapRows ($row, $row + 1);

  my $refs            = item_refs;
  my $tmp             = $refs -> [$row];
  $refs -> [$row]     = $refs -> [$row + 1];
  $refs -> [$row + 1] = $tmp;

  $table -> clearSelection ();
  $row++;

  # selectRow does not work for some reason
  $table -> selectCells ($row, 0, $row, 4);
  $table -> updateContents ();

  my $rect = $table -> cellGeometry ($row, $table -> currentColumn ());
  $table -> ensureVisible ($rect -> left (), $rect -> bottom ());
}


sub add_item
{
  my $table    = menu_table;
  my $num_rows = $table -> numRows ();
  my $row      = $table -> currentRow ();
  my $col      = $table -> currentColumn ();
  my $ttn;

  if ($num_rows)
  {
    # table not empty
    $ttn = $table -> item ($row, &TITLE_COL) -> current_title ();
    $row++;
  }
  else
  {
    # table empty - check for any selected title
    return unless (@{&selected_titles});

    $ttn = ${&selected_titles} [0];
    $row = 0;
  }

  # this forces an update of the currently edited cell
  $table -> setCurrentCell (0, 0);

  my $next_label = 1;

  for (my $i = 0; $i < $num_rows; $i++)
  {
    my $label   = $table -> text ($i, &LABEL_COL);
    $next_label = $label + 1 if ($label =~ /^\d+$/ &&
				 $label >= $next_label);
  }

  insert_row ($row, $ttn, $next_label);

  # set new current row
  $table -> setCurrentCell ($row, $col);
}


sub remove_item
{
  my $table    = menu_table;
  my $num_rows = $table -> numRows ();

  # check for table empty
  return unless ($num_rows);

  my $row = $table -> currentRow ();

  # remove current row
  remove_row ($row);

  # check for table empty again now
  return unless (--$num_rows);

  # if current row was last row, adjust current row
  $row-- if ($row >= $num_rows);

  # set new current row
  $table -> setCurrentCell ($row, $table -> currentColumn ());
}


sub renumber_labels
{
  my $table    = menu_table;
  my $num_rows = $table -> numRows ();
  my $row      = $table -> currentRow ();
  my $col      = $table -> currentColumn ();

  $table -> setCurrentCell (0, 0);

  for (my $row = 0; $row < $num_rows; $row++)
  {
    $table -> setText ($row, &LABEL_COL, $row + 1);
  }

  $table -> setCurrentCell ($row, $col);
}


sub end_edit_mode
{
  my $table = menu_table;

  return unless ($table);

  my $row = $table -> currentRow ();
  my $col = $table -> currentColumn ();

  $table -> setCurrentCell (0, 0);
  $table -> setCurrentCell ($row, $col);
}


sub nr_of_items
{
  return menu_table -> numRows ();
}


sub title_nr
{
  my $row = shift;

  my $table = menu_table;

  return 0 unless ($row < $table -> numRows ());

  return $table -> item ($row, &TITLE_COL) -> currentItem () + 1;
}


sub audio_stream
{
  my $row = shift;

  my $table = menu_table;

  return 0 unless ($row < $table -> numRows ());

  return $table -> item ($row, &AUDIO_COL) -> currentItem ();
}


sub subtitle_stream
{
  my $row = shift;

  my $table = menu_table;

  return -1 unless ($row < $table -> numRows ());

  return $table -> item ($row, &SUBTITLE_COL) -> currentItem () - 1;
}


sub label
{
  my $row = shift;

  my $table = menu_table;

  return undef unless ($row < $table -> numRows ());

  return $table -> text ($row, &LABEL_COL);
}


sub menu_text
{
  my $row = shift;

  my $table = menu_table;

  return undef unless ($row < $table -> numRows ());

  return $table -> text ($row, &MENU_COL);
}


1;
