/*****************************************************************************
 ** Class Cache::Thumbs
 **
 ** This class handles the caching of thumbnails created by DialogFiles
 **
 **   Created : Tue Dec 20 07:53:05 2007
 **        by : Varol Okan using the XEmacs
 ** Copyright : (c) Varol Okan
 **   License : GPL v 2.0
 **
 ** The cache is stored under :
 ** $HOME/.qdvdauthor/cache
 ** The following files should/must exist for the cache to work.
 ** o  thumbs_db.xml This file is the database for the cache. 
 **                  The xml structure of this flie is described below
 ** o  virtual_folder.xml This file holds the user created 
 **                  VirtualFolder information
 ** o  /<first Letter Of Video File>/<Hash Name Of File>
 **
 *****************************************************************************/

/* XML File structure :
 <CacheThumbs>
   <Source FileSize="" Rating="" VirtualFolders="" DateCreated="" 
           VideoLength="" NumberOfThumbs="" FileName="" 
	   DateLastUsed="">ThumbFileName</Source> Note: ThumbFileName is generated through a hashing function
   <Source ... >
 </CacheThumbs>

 <VirtualFolder>
   <Folder name="" created="" >
     // Note: ThumbFileName is generated through a hashing function
     //       the hashed name has fileSize/date encoded in it
     <FolderEntry FileName="" >ThumbFileName</FolderEntry>
   </Folder>
 </VirtualFolder>
*/

#ifndef __CACHE_THUMBS_H__
#define __CACHE_THUMBS_H__

#include <qdatetime.h>

#define CACHE_DB_FILE      "cach_db.xml"
#define IMAGE_DB_FILE      "image_db.xml"
#define VIRTUAL_DB_FILE    "virtual_db.xml"
#define CACHE_DOCTYPE      ""

// Deines for the cache xml file
#define CACHE              "CacheThumbs"
#define CACHE_TAG_SOURCE   "Source"
#define CACHE_NR_OF_THUMBS "NrOfThumbs"
#define CACHE_FILE_SIZE    "FileSize"
#define CACHE_RATING       "Rating"
#define CACHE_VIRTUAL      "VirtualFolders"
#define CACHE_CREATED      "DateCreated"
#define CACHE_LENGTH       "VideoLength"
#define CACHE_FILE_NAME    "FileName"
#define CACHE_LAST_USED    "DateLastUsed"

// Defines for Virtual Folder
#define VIRTUAL            "VirtualFolder"
#define VIRTUAL_TAG_FOLDER "Folder"
#define VIRTUAL_TAG_ENTRY  "FolderEntry"
#define VIRTUAL_NAME       "Name"
#define VIRTUAL_HASH       "Hash"
#define VIRTUAL_CREATED    "Created"
#define VIRTUAL_FILE_NAME  "FileName"

class QDomElement;
#include "qplayer/mediascanner.h"


namespace Cache
{

class Thumbs
{
public:
  class Entry {
  public:
    Entry ( Thumbs * );
    bool readXml  ( QDomElement * );
    bool writeXml ( QDomElement * );

    void saveImages ( ); // store images to cache
    void loadImages ( ); // load images from cache
    void freeImages ( ); // frees the memory 
    void scanImages ( QObject *p=NULL ); // scans the video file

    unsigned long long iHashValue;
    bool     bSourceExists; // If the file is not reachable ( removable storage or deleted )
    uint     iScanStatus;   // 0=undef, 1=scanning, 2=scanning done.
    QString  qsFileName;
    QString  qsLength;  // "00:00:00"
    uint     iFileSize; // 1234 in bytes
    QImage **arrayOfThumbs;
    uint     iNumberOfThumbs;
    uint     iStarRating;
    QStringList listVirtualFolders;

    QDateTime dateCreated;  // obtained from meta data or file creation date
    QDateTime dateLastUsed; // to eventually purge from the cache
    Thumbs   *pCache;
  };
  class VirtualFolder {
  public:
    class VFile {
    public:
      VFile ( ) { iFileHashValue = 0LL; };
      QString qsFileName;
      unsigned long long iFileHashValue;
    };

     VirtualFolder ( Thumbs * );
    ~VirtualFolder ( );
    bool readXml   ( QDomElement * );
    bool writeXml  ( QDomElement * );
    void append    ( Entry * );
    void clear     ( );
    uint count     ( );
    VirtualFolder *clone ( );

    unsigned int iHashValue;
    QDateTime    dateCreated;
    QString      qsFolderName;
    Thumbs      *pCache;
    QValueList<VFile *>listOfFiles;
  };
  class MultiThumbRequest : public ExecuteJob  {
  public:
     MultiThumbRequest ( QObject *, Entry * );
    ~MultiThumbRequest ( );
    virtual bool response ( );

    long    iDelta;
    long    iMovieLength;
    int     iCurrentThumb;
    Entry  *pEntry;
  };

public:
   Thumbs ( );
  ~Thumbs ( );

  void clearCache         ( );
  void clearVirtualFolder ( );
  bool saveCacheDB        ( );
  bool saveVirtualFolder  ( );

  Entry *find     ( QString & );
  Entry *find     ( unsigned long long );
  Entry *append   ( QString &, QString q=QString::null, uint i=10,  QImage **pp=NULL );
  Entry *append   ( QString &, QObject *, uint i=10 );

  unsigned int   addVirtualFolder    (  QString &   );
  void           deleteVirtualFolder (  QString &   );
  void           deleteVirtualFolder ( unsigned int );
  VirtualFolder *findVirtualFolder   ( unsigned int );
  VirtualFolder *findVirtualFolder   (  QString &   );
  QValueList<VirtualFolder *> &getVirtualFolders  ( );

  unsigned long long hashFromFile ( QString & );
  
  QValueList<Entry *> getThumbsByStar  ( uint );
  QValueList<Entry *> getScannedThumbs ( );

private:
  bool loadCacheDB       ( ); // Load cache_db.xml file
  void cleanCacheDB      ( ); // After a set amount of time the cache is freed of images based on dateLastUsed
  bool loadVirtualFolder ( );
  void freeImages        ( ); // to free some space after closing dialog
  QString getCachePath   ( QChar   );

  unsigned long long SDBMHash ( QString & );

  QValueList<Entry *>         m_listOfThumbs;
  QValueList<VirtualFolder *> m_listOfVirtualFolders;
};

}

#endif // __CACHE_THUMBS_H__
