/*
 *  OpenDuke
 *  Copyright (C) 1999  Rusty Wagner
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */


#include "poly.h"
#ifndef PLATFORM_UNIX
#include <windows.h>
#endif
#include <math.h>

#define SQ(x) ((x)*(x))

Plane PlaneFromPoly(const Poly& poly)
{
    Vector normal=VectorCrossProduct(Vector(poly.v[1].x-poly.v[0].x,
        poly.v[1].y-poly.v[0].y,poly.v[1].z-poly.v[0].z),Vector(
        poly.v[2].x-poly.v[0].x,poly.v[2].y-poly.v[0].y,poly.v[2].z-
        poly.v[0].z));
    return Plane(normal.x,normal.y,normal.z,-(normal.x*
        poly.v[0].x+normal.y*poly.v[0].y+normal.z*poly.v[0].z));
}

Vector PolyNormal(const Poly& poly)
{
    Vector normal=VectorCrossProduct(Vector(poly.v[1].x-poly.v[0].x,
        poly.v[1].y-poly.v[0].y,poly.v[1].z-poly.v[0].z),Vector(
        poly.v[2].x-poly.v[0].x,poly.v[2].y-poly.v[0].y,poly.v[2].z-
        poly.v[0].z));
    if (poly.flags&PF_REVERSE)
        normal=Vector(-normal.x,-normal.y,-normal.z);
    return normal;
}

int IsBackFace(const Poly& poly,const Vector& pos)
{
    if (poly.flags&PF_TWOSIDED) return 0;
    Vector normal=VectorCrossProduct(Vector(poly.v[1].x-poly.v[0].x,
        poly.v[1].y-poly.v[0].y,poly.v[1].z-poly.v[0].z),Vector(
        poly.v[2].x-poly.v[0].x,poly.v[2].y-poly.v[0].y,poly.v[2].z-
        poly.v[0].z));
    int result=normal.x*pos.x+normal.y*pos.y+normal.z*pos.z-(normal.x*
        poly.v[0].x+normal.y*poly.v[0].y+normal.z*poly.v[0].z);
    return (poly.flags&PF_REVERSE)?(result>0):(result<0);
}

PolyListElem* AddPolyToList(PolyList &list,Poly &poly)
{
    poly.CalcMinRange();
    if (list.first==NULL)
    {
        list.first=list.last=new PolyListElem;
        list.first->poly=*(new Poly(poly));
        return list.first;
    }
    list.last->next=new PolyListElem;
    list.last->next->poly=*(new Poly(poly));
    list.last=list.last->next;
    return list.last;
}

void FreeList(PolyList &list)
{
    PolyListElem *ptr=list.first;
    while (ptr!=NULL)
    {
        PolyListElem *tmp=ptr->next;
        delete ptr;
        ptr=tmp;
    }
    list.first=list.last=NULL;
}

Vector VectorCrossProduct(const Vector& u,const Vector& v)
{
    return Vector(u.y*v.z-u.z*v.y,-(u.x*v.z-u.z*v.x),
        u.x*v.y-u.y*v.x);
}

int IsPointInPolyXY(Vector& pt,Poly &poly,float minDist)
{
    int i,j,c=0;
    for (i=0,j=poly.nVert-1;i<poly.nVert;j=i++)
    {
        if ((((poly.v[i].y<=pt.y)&&(pt.y<poly.v[j].y))||
            ((poly.v[j].y<=pt.y)&&(pt.y<poly.v[i].y)))&&
            (pt.x<(poly.v[j].x-poly.v[i].x)*(pt.y-
            poly.v[i].y)/(poly.v[j].y-poly.v[i].y)+
            poly.v[i].x))
            c=!c;
    }
    if (c) return 1;
    if (minDist==0) return 0;
    for (i=0,j=poly.nVert-1;i<poly.nVert;j=i++)
    {
        Vector edge=Vector(poly.v[j].x-poly.v[i].x,poly.v[j].y-
            poly.v[i].y,poly.v[j].z-poly.v[i].z);
        Vector ptv=Vector(pt.x-poly.v[i].x,pt.y-poly.v[i].y,
            pt.z-poly.v[i].z);
        float scale=(edge.x*ptv.x+edge.y*ptv.y+edge.z*ptv.z)/
            (edge.x*edge.x+edge.y*edge.y+edge.z*edge.z);
        Vector proj=Vector(edge.x*scale,edge.y*scale,
            edge.z*scale);
        float distsq=(ptv.x-proj.x)*(ptv.x-proj.x)+
            (ptv.y-proj.y)*(ptv.y-proj.y)+(ptv.z-proj.z)*
            (ptv.z-proj.z);
        if (distsq<(minDist-0.00001))
            return 1;
    }
    return 0;
}

int IsPointInPolyXZ(Vector& pt,Poly &poly,float minDist)
{
    int i,j,c=0;
    for (i=0,j=poly.nVert-1;i<poly.nVert;j=i++)
    {
        if ((((poly.v[i].z<=pt.z)&&(pt.z<poly.v[j].z))||
            ((poly.v[j].z<=pt.z)&&(pt.z<poly.v[i].z)))&&
            (pt.x<(poly.v[j].x-poly.v[i].x)*(pt.z-
            poly.v[i].z)/(poly.v[j].z-poly.v[i].z)+
            poly.v[i].x))
            c=!c;
    }
    if (c) return 1;
    if (minDist==0) return 0;
    for (i=0,j=poly.nVert-1;i<poly.nVert;j=i++)
    {
        Vector edge=Vector(poly.v[j].x-poly.v[i].x,poly.v[j].y-
            poly.v[i].y,poly.v[j].z-poly.v[i].z);
        Vector ptv=Vector(pt.x-poly.v[i].x,pt.y-poly.v[i].y,
            pt.z-poly.v[i].z);
        float scale=(edge.x*ptv.x+edge.y*ptv.y+edge.z*ptv.z)/
            (edge.x*edge.x+edge.y*edge.y+edge.z*edge.z);
        Vector proj=Vector(edge.x*scale,edge.y*scale,
            edge.z*scale);
        float distsq=(ptv.x-proj.x)*(ptv.x-proj.x)+
            (ptv.y-proj.y)*(ptv.y-proj.y)+(ptv.z-proj.z)*
            (ptv.z-proj.z);
        if (distsq<(minDist-0.00001))
            return 1;
    }
    return 0;
}

int IsPointInPolyYZ(Vector& pt,Poly &poly,float minDist)
{
    int i,j,c=0;
    for (i=0,j=poly.nVert-1;i<poly.nVert;j=i++)
    {
        if ((((poly.v[i].z<=pt.z)&&(pt.z<poly.v[j].z))||
            ((poly.v[j].z<=pt.z)&&(pt.z<poly.v[i].z)))&&
            (pt.y<(poly.v[j].y-poly.v[i].y)*(pt.z-
            poly.v[i].z)/(poly.v[j].z-poly.v[i].z)+
            poly.v[i].y))
            c=!c;
    }
    if (c) return 1;
    if (minDist==0) return 0;
    for (i=0,j=poly.nVert-1;i<poly.nVert;j=i++)
    {
        Vector edge=Vector(poly.v[j].x-poly.v[i].x,poly.v[j].y-
            poly.v[i].y,poly.v[j].z-poly.v[i].z);
        Vector ptv=Vector(pt.x-poly.v[i].x,pt.y-poly.v[i].y,
            pt.z-poly.v[i].z);
        float scale=(edge.x*ptv.x+edge.y*ptv.y+edge.z*ptv.z)/
            (edge.x*edge.x+edge.y*edge.y+edge.z*edge.z);
        Vector proj=Vector(edge.x*scale,edge.y*scale,
            edge.z*scale);
        float distsq=(ptv.x-proj.x)*(ptv.x-proj.x)+
            (ptv.y-proj.y)*(ptv.y-proj.y)+(ptv.z-proj.z)*
            (ptv.z-proj.z);
        if (distsq<(minDist-0.00001))
            return 1;
    }
    return 0;
}

int IsPointInPoly(Vector& pt,Poly &poly,float minDist)
{
    switch(poly.minRange)
    {
        case 0: return IsPointInPolyYZ(pt,poly,minDist);
        case 1: return IsPointInPolyXZ(pt,poly,minDist);
        case 2: return IsPointInPolyXY(pt,poly,minDist);
    }
    return 0;
}
