/*
 *  OpenDuke
 *  Copyright (C) 1999  Rusty Wagner
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */


#include <stdio.h>
#include "Texture.h"

extern int software;

CTexture::CTexture(CDirect3D *d3d)
{
	D3D=d3d;
	lpTexture=NULL;
}

CTexture::~CTexture()
{
	if (lpTexture) { lpTexture->Release(); lpTexture=NULL; }
}

static int wantAlphaTex;

HRESULT CALLBACK FindTextureCallback(DDPIXELFORMAT *ddpf,LPVOID _pf)
{
	DDPIXELFORMAT *pf=(DDPIXELFORMAT*)_pf;
	int alphaBitDepth;

	if (ddpf->dwRGBAlphaBitMask==0)
		alphaBitDepth=0;
	else
	{
		unsigned long v=ddpf->dwRGBAlphaBitMask;
		while (!(v&1)) v>>=1;
		alphaBitDepth=0;
		while (v&1)
		{
			alphaBitDepth++;
			v>>=1;
		}
	}
	if ((wantAlphaTex==ALPHA_ONEBIT)&&(!alphaBitDepth))
		return DDENUMRET_OK;
	if ((wantAlphaTex==ALPHA_FULL)&&(alphaBitDepth<4))
		return DDENUMRET_OK;
	if (ddpf->dwRGBBitCount<=8)
		return DDENUMRET_OK;
	if (!(ddpf->dwFlags&DDPF_RGB))
		return DDENUMRET_OK;
    if ((ddpf->dwGBitMask!=0x3e0)&&(ddpf->dwRBitMask!=0x3e0)&&
        (wantAlphaTex<=ALPHA_ONEBIT))
    	return DDENUMRET_OK;
	if ((pf->dwRGBBitCount==0)||(ddpf->dwRGBBitCount>
		pf->dwRGBBitCount))
		*pf=*ddpf;
	else if ((wantAlphaTex==ALPHA_ONEBIT)&&(alphaBitDepth==1))
		*pf=*ddpf;
	return DDENUMRET_OK;
}

void ChooseTextureFormat(IDirect3DDevice7 *device,int alpha,
	DDPIXELFORMAT *pf)
{
	wantAlphaTex=alpha;
	pf->dwAlphaBitDepth=0;
	pf->dwRGBBitCount=0;
	device->EnumTextureFormats(FindTextureCallback,pf);
}

int CTexture::Create(int width,int height,int alpha,int hint,int mipMap)
{
	DDSURFACEDESC2 ddsd;

	// Fill in DDSURFACEDESC structure with correct information
    memset(&ddsd,0,sizeof(ddsd));
	ddsd.dwSize=sizeof(ddsd);
	ChooseTextureFormat(D3D->GetD3DDevice(),alpha,
		&ddsd.ddpfPixelFormat);
	ddsd.dwFlags=DDSD_CAPS|DDSD_HEIGHT|DDSD_WIDTH|DDSD_PIXELFORMAT|DDSD_MIPMAPCOUNT;
    ddsd.ddpfPixelFormat.dwSize=sizeof(DDPIXELFORMAT);
	ddsd.dwWidth=width;
	ddsd.dwHeight=height;
    ddsd.dwMipMapCount=(width<=16)?1:(width<=32)?2:3;
	ddsd.ddsCaps.dwCaps=DDSCAPS_TEXTURE|(mipMap?DDSCAPS_MIPMAP|
        DDSCAPS_COMPLEX:0);
    ddsd.ddsCaps.dwCaps2=DDSCAPS2_TEXTUREMANAGE|hint;

	// Create the texture surface
	if (D3D->GetDDraw()->GetDD()->CreateSurface(&ddsd,&lpTexture,
		NULL)!=DD_OK)
		return 0;

	return 1;
}

void CTexture::LoadFromFile(char *filename)
{
    fn=new char[strlen(filename)+1]; strcpy(fn,filename);
    afn=NULL;
    FILE *fp=fopen(filename,"rb");
    fseek(fp,sizeof(BITMAPFILEHEADER),SEEK_SET);
    BITMAPINFOHEADER info;
    fread(&info,sizeof(BITMAPINFOHEADER),1,fp);
    Create(info.biWidth,info.biHeight,ALPHA_FULL,0);
    unsigned char *texBits=(unsigned char*)Lock();
    unsigned char *fileBits=new unsigned char[info.biHeight*
        info.biWidth*3];
    fread(fileBits,info.biHeight*info.biWidth*3,1,fp);
    int i=0;
    for (int y=0;y<info.biHeight;y++)
    {
        for (int x=0;x<info.biWidth;x++)
        {
            *(texBits++)=fileBits[i++];
            *(texBits++)=fileBits[i++];
            *(texBits++)=fileBits[i++];
            *(texBits++)=255;
        }
    }
    delete[] fileBits;
    Unlock();
    fclose(fp);
    GenerateMipMap();
}

void CTexture::LoadFromFileWithAlpha(char *filename,char *alphaname)
{
    fn=new char[strlen(filename)+1]; strcpy(fn,filename);
    afn=new char[strlen(alphaname)+1]; strcpy(afn,alphaname);
    FILE *fp=fopen(filename,"rb");
    FILE *alpha=fopen(alphaname,"rb");
    fseek(fp,sizeof(BITMAPFILEHEADER),SEEK_SET);
    BITMAPINFOHEADER info;
    fread(&info,sizeof(BITMAPINFOHEADER),1,fp);
    fseek(alpha,10,SEEK_SET);
    DWORD ofs;
    fread(&ofs,4,1,alpha);
    fseek(alpha,ofs,SEEK_SET);
    Create(info.biWidth,info.biHeight,ALPHA_FULL,0);
    unsigned char *texBits=(unsigned char*)Lock();
    unsigned char *fileBits=new unsigned char[info.biHeight*
        info.biWidth*3];
    unsigned char *alphaBits=new unsigned char[info.biHeight*info.biWidth];
    fread(fileBits,info.biHeight*info.biWidth*3,1,fp);
    fread(alphaBits,info.biHeight*info.biWidth,1,alpha);
    int i=0,j=0;
    for (int y=0;y<info.biHeight;y++)
    {
        for (int x=0;x<info.biWidth;x++)
        {
            *(texBits++)=fileBits[i++];
            *(texBits++)=fileBits[i++];
            *(texBits++)=fileBits[i++];
            *(texBits++)=alphaBits[j++];
        }
    }
    delete[] fileBits;
    delete[] alphaBits;
    Unlock();
    fclose(fp);
    fclose(alpha);
    GenerateMipMap();
}

void CTexture::LoadFromMemoryTexture(Texture *tex)
{
    Create(tex->memTex.width,tex->memTex.height,ALPHA_ONEBIT,0);
    unsigned short *texBits=(unsigned short*)Lock();
    memcpy(texBits,tex->memTex.data,tex->memTex.width*tex->memTex.height*2);
    Unlock();
    GenerateMipMap();
}

void CTexture::UpdateMemoryTexture(Texture *tex)
{
    unsigned short *texBits=(unsigned short*)Lock();
    memcpy(texBits,tex->memTex.data,tex->memTex.width*tex->memTex.height*2);
    Unlock();
    GenerateMipMap();
}

void CTexture::Reload(CDirect3D *d3d)
{
    D3D=d3d;
    if (!afn) LoadFromFile(fn);
    else LoadFromFileWithAlpha(fn,afn);
}

void *CTexture::Lock()
{
	DDSURFACEDESC2 ddsd;

	ddsd.dwSize=sizeof(ddsd);

	int err=lpTexture->Lock(NULL,&ddsd,DDLOCK_WAIT|
		DDLOCK_SURFACEMEMORYPTR|DDLOCK_NOSYSLOCK|
        DDLOCK_WRITEONLY,NULL);
	if (err==DDERR_SURFACELOST)
	{
		lpTexture->Restore();
		if (lpTexture->Lock(NULL,&ddsd,DDLOCK_WAIT|
			DDLOCK_SURFACEMEMORYPTR|DDLOCK_NOSYSLOCK,NULL)!=DD_OK)
			return NULL;
	}
	else if (err!=DD_OK)
		return NULL;
	return (void *)ddsd.lpSurface;
}

void CTexture::Unlock()
{
	lpTexture->Unlock(NULL);
}

void CTexture::GenerateMipMap()
{
    IDirectDrawSurface7 *lpLevel,*lpNextLevel;
    DDSCAPS2 ddsCaps;
    HRESULT result;
    unsigned short *cur,*buf;

    lpLevel=lpTexture;
    lpLevel->AddRef();
    memset(&ddsCaps,0,sizeof(DDSCAPS2));
    ddsCaps.dwCaps=DDSCAPS_TEXTURE|DDSCAPS_MIPMAP;
    result=lpLevel->GetAttachedSurface(&ddsCaps,&lpNextLevel);
    while (result==DD_OK)
    {
    	DDSURFACEDESC2 ddsd;
    	ddsd.dwSize=sizeof(ddsd);
        int err=lpLevel->Lock(NULL,&ddsd,DDLOCK_WAIT|
    		DDLOCK_SURFACEMEMORYPTR|DDLOCK_NOSYSLOCK|
            DDLOCK_WRITEONLY,NULL);
    	if (err==DDERR_SURFACELOST)
	    {
    		lpLevel->Restore();
	    	if (lpLevel->Lock(NULL,&ddsd,DDLOCK_WAIT|
		    	DDLOCK_SURFACEMEMORYPTR|DDLOCK_NOSYSLOCK,NULL)!=DD_OK)
    			break;
	    }
    	else if (err!=DD_OK)
            break;
        cur=(unsigned short*)ddsd.lpSurface;
        int lw=ddsd.lPitch/2;
        err=lpNextLevel->Lock(NULL,&ddsd,DDLOCK_WAIT|
    		DDLOCK_SURFACEMEMORYPTR|DDLOCK_NOSYSLOCK|
            DDLOCK_WRITEONLY,NULL);
    	if (err==DDERR_SURFACELOST)
	    {
    		lpNextLevel->Restore();
	    	if (lpNextLevel->Lock(NULL,&ddsd,DDLOCK_WAIT|
		    	DDLOCK_SURFACEMEMORYPTR|DDLOCK_NOSYSLOCK,NULL)!=DD_OK)
    			break;
	    }
    	else if (err!=DD_OK)
            break;
        buf=(unsigned short*)ddsd.lpSurface;
        for (int y=0;y<ddsd.dwHeight;y++)
        {
            for (int x=0;x<ddsd.dwWidth;x++,cur+=2)
            {
                int r1,r2,r3,r4,g1,g2,g3,g4,b1,b2,b3,b4,a1,a2,a3,a4;
                r1=(cur[0]>>10)&0x1f; r2=(cur[1]>>10)&0x1f; r3=(cur[0+lw]>>10)&0x1f; r4=(cur[1+lw]>>10)&0x1f;
                g1=(cur[0]>>5)&0x1f; g2=(cur[1]>>5)&0x1f; g3=(cur[0+lw]>>5)&0x1f; g4=(cur[1+lw]>>5)&0x1f;
                b1=cur[0]&0x1f; b2=cur[1]&0x1f; b3=cur[0+lw]&0x1f; b4=cur[1+lw]&0x1f;
                a1=cur[0]>>15; a2=cur[1]>>15; a3=cur[0+lw]>>15; a4=cur[1+lw]>>15;
                r1=(r1+r2+r3+r4)>>2;
                g1=(g1+g2+g3+g4)>>2;
                b1=(b1+b2+b3+b4)>>2;
                a1=(a1+a2+a3+a4+2)>>2;
                *(buf++)=(a1<<15)|(r1<<10)|(g1<<5)|b1;
            }
            cur+=lw;
        }
        lpLevel->Unlock(NULL);
        lpNextLevel->Unlock(NULL);
        lpLevel->Release();
        lpLevel=lpNextLevel;
        lpLevel->AddRef();
        ddsCaps.dwCaps=DDSCAPS_TEXTURE|DDSCAPS_MIPMAP;
        result=lpLevel->GetAttachedSurface(&ddsCaps,&lpNextLevel);
    }
    lpLevel->Release();
}

