/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2009 Olivier Rolland <billl@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include "ogmrip-gconf.h"
#include "ogmrip-helper.h"
#include "ogmrip-plugin.h"
#include "ogmrip-options-plugin.h"
#include "ogmrip-preferences.h"
#include "ogmrip-profile-editor.h"

#include <glib/gi18n.h>
#include <glade/glade.h>

#define OGMRIP_GLADE_EXPERT_FILE "ogmrip/ogmrip-profile-editor.glade"
#define OGMRIP_GLADE_SIMPLE_FILE "ogmrip/ogmrip-simple-editor.glade"
#define OGMRIP_GLADE_ROOT "root"

#define OGMRIP_PROFILE_EDITOR_DIALOG_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_PROFILE_EDITOR_DIALOG, OGMRipProfileEditorDialogPriv))

struct _OGMRipProfileEditorDialogPriv
{
  GtkWidget *root;

  GtkWidget *video_codec_combo;
  GtkWidget *audio_codec_combo;
  GtkWidget *subp_codec_combo;

  GtkWidget *container_options_button;
  GtkWidget *video_options_button;
  GtkWidget *audio_options_button;
  GtkWidget *subp_options_button;

  GtkWidget *encoding_combo;

  GtkWidget *bitrate_label;
  GtkWidget *bitrate_hbox;

  GtkWidget *quantizer_spin;
  GtkWidget *quantizer_label;

  GtkWidget *target_label;
  GtkWidget *target_hbox;

  GtkWidget *video_preset_combo;

  GtkWidget *passes_spin;
  GtkWidget *passes_hbox;

  GtkWidget *threads_spin;

  GtkWidget *text_table;

  gchar *profile;
};

enum
{
  SIZE,
  BITRATE,
  QUANTIZER
};

static void ogmrip_profile_editor_dialog_finalize (GObject *gobject);

static const gchar *gconf_dir;

static GType
ogmrip_profile_editor_dialog_get_container_type (OGMRipProfileEditorDialog *dialog, const gchar *name)
{
  GType container;

  if (name)
    container = ogmrip_plugin_get_container_by_name (name);
  else
  {
    gchar *key, *str;

    key = gconf_concat_dir_and_key (dialog->priv->profile, OGMRIP_GCONF_CONTAINER_FORMAT);
    str = ogmrip_preferences_get_string (key, OGMRIP_DEFAULT_CONTAINER_FORMAT);
    g_free (key);

    container = ogmrip_plugin_get_container_by_name (str);
    g_free (str);
  }

  return container;
}

static GType
ogmrip_profile_editor_dialog_get_video_codec_type (OGMRipProfileEditorDialog *dialog, const gchar *name)
{
  GType container;

  if (name)
    container = ogmrip_plugin_get_video_codec_by_name (name);
  else
  {
    gchar *key, *str;

    key = gconf_concat_dir_and_key (dialog->priv->profile, OGMRIP_GCONF_VIDEO_CODEC);
    str = ogmrip_preferences_get_string (key, OGMRIP_DEFAULT_VIDEO_CODEC);
    g_free (key);

    container = ogmrip_plugin_get_video_codec_by_name (str);
    g_free (str);
  }

  return container;
}

static GType
ogmrip_profile_editor_dialog_get_audio_codec_type (OGMRipProfileEditorDialog *dialog, const gchar *name)
{
  GType container;

  if (name)
    container = ogmrip_plugin_get_audio_codec_by_name (name);
  else
  {
    gchar *key, *str;

    key = gconf_concat_dir_and_key (dialog->priv->profile, OGMRIP_GCONF_AUDIO_CODEC);
    str = ogmrip_preferences_get_string (key, OGMRIP_DEFAULT_AUDIO_CODEC);
    g_free (key);

    container = ogmrip_plugin_get_audio_codec_by_name (str);
    g_free (str);
  }

  return container;
}

static GType
ogmrip_profile_editor_dialog_get_subp_codec_type (OGMRipProfileEditorDialog *dialog, const gchar *name)
{
  GType container;

  if (name)
    container = ogmrip_plugin_get_subp_codec_by_name (name);
  else
  {
    gchar *key, *str;

    key = gconf_concat_dir_and_key (dialog->priv->profile, OGMRIP_GCONF_SUBP_CODEC);
    str = ogmrip_preferences_get_string (key, OGMRIP_DEFAULT_SUBP_CODEC);
    g_free (key);

    container = ogmrip_plugin_get_subp_codec_by_name (str);
    g_free (str);
  }

  return container;
}

/*
 * Toggle
 */

static void
ogmrip_profile_editor_dialog_connect_toggle (OGMRipProfileEditorDialog *dialog, GtkWidget *toggle, const gchar *key, gboolean def)
{
  GConfValue *value;
  gchar *real_key;

  if (dialog->priv->profile && key[0] != '/')
    real_key = gconf_concat_dir_and_key (dialog->priv->profile, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_bool (real_key, def);

  ogmrip_preferences_connect_toggle (toggle, real_key);

  g_free (real_key);
}

/*
 * Spin
 */

static void
ogmrip_profile_editor_dialog_connect_spin (OGMRipProfileEditorDialog *dialog, GtkWidget *spin, const gchar *key, gint def)
{
  GConfValue *value;
  gchar *real_key;

  if (dialog->priv->profile)
    real_key = gconf_concat_dir_and_key (dialog->priv->profile, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_int (real_key, def);

  ogmrip_preferences_connect_spin (spin, real_key);

  g_free (real_key);
}

/*
 * Combo
 */

static void
ogmrip_profile_editor_dialog_connect_combo (OGMRipProfileEditorDialog *dialog, GtkWidget *combo, const gchar *key, gint def)
{
  GConfValue *value;
  gchar *real_key;

  if (dialog->priv->profile)
    real_key = gconf_concat_dir_and_key (dialog->priv->profile, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_int (real_key, def);

  ogmrip_preferences_connect_combo (combo, real_key);

  g_free (real_key);
}

/*
 * Common
 */

static GConfValue *
ogmrip_profile_editor_dialog_combo_get_value (GtkWidget *combo)
{
  GConfValue *value;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *name;

  model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo));
  if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX (combo), &iter))
    return NULL;

  gtk_tree_model_get (model, &iter, 1, &name, -1);

  value = gconf_value_new (GCONF_VALUE_STRING);
  gconf_value_set_string (value, name);
  g_free (name);

  return value;
}

static void
ogmrip_profile_editor_dialog_combo_set_active (GtkWidget *combo, const gchar *name)
{
  GtkTreeModel *model;
  GtkTreeIter iter;

  model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo));
  if (gtk_tree_model_iter_children (model, &iter, NULL))
  {
    gchar *str;
    do
    {
      gtk_tree_model_get (model, &iter, 1, &str, -1);
      if (g_str_equal (str, name))
      {
        gtk_combo_box_set_active_iter (GTK_COMBO_BOX (combo), &iter);
        g_free (str);
        break;
      }
      g_free (str);
    }
    while (gtk_tree_model_iter_next (model, &iter));

    if (gtk_combo_box_get_active (GTK_COMBO_BOX (combo)) < 0)
      gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);
  }
}

/*
 * Container
 */

static gboolean
ogmrip_profile_editor_dialog_container_set_value (GtkWidget *combo, GConfValue *value)
{
  GType container;
  const gchar *name;

  if (!value)
    return FALSE;

  name = gconf_value_get_string (value);
  container = ogmrip_gconf_get_container_type (name);

  if (container == G_TYPE_NONE)
  {
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);
    return FALSE;
  }

  ogmrip_profile_editor_dialog_combo_set_active (combo, name);

  return TRUE;
}

static void
ogmrip_profile_editor_dialog_connect_container (OGMRipProfileEditorDialog *dialog, GtkWidget *widget, const gchar *key, const gchar *def)
{
  GConfValue *value;
  gchar *real_key;

  if (dialog->priv->profile)
    real_key = gconf_concat_dir_and_key (dialog->priv->profile, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_string (real_key, def);

  if (widget)
    ogmrip_preferences_connect_combo_custom (widget, real_key,
        ogmrip_profile_editor_dialog_combo_get_value,
        ogmrip_profile_editor_dialog_container_set_value);

  g_free (real_key);
}

typedef struct
{
  GtkListStore *store;
  gboolean (* can_contain) (GType, GType);
  GType container;
} OGMRipComboData;

static void
ogmrip_profile_editor_dialog_add_codecs (GType type, const gchar *name, const gchar *description, OGMRipComboData *data)
{
  GtkTreeIter iter;

  if (!data->can_contain || (* data->can_contain) (data->container, type))
  {
    gtk_list_store_append (data->store, &iter);
    gtk_list_store_set (data->store, &iter, 0, description, 1, name, -1);
  }
}

static void
ogmrip_profile_editor_dialog_add_containers (GtkListStore *store)
{
  OGMRipComboData data =
  {
    store,
    NULL,
    0
  };

  ogmrip_plugin_foreach_container ((OGMRipPluginFunc) ogmrip_profile_editor_dialog_add_codecs, &data);
}

static void
ogmrip_profile_editor_dialog_add_video_codecs (GtkListStore *store, GType container)
{
  OGMRipComboData data =
  {
    store,
    ogmrip_plugin_can_contain_video,
    container
  };

  ogmrip_plugin_foreach_video_codec ((OGMRipPluginFunc) ogmrip_profile_editor_dialog_add_codecs, &data);
}

static void
ogmrip_profile_editor_dialog_add_audio_codecs (GtkListStore *store, GType container)
{
  OGMRipComboData data =
  {
    store,
    ogmrip_plugin_can_contain_audio,
    container
  };

  ogmrip_plugin_foreach_audio_codec ((OGMRipPluginFunc) ogmrip_profile_editor_dialog_add_codecs, &data);
}

static void
ogmrip_profile_editor_dialog_add_subp_codecs (GtkListStore *store, GType container)
{
  OGMRipComboData data =
  {
    store,
    ogmrip_plugin_can_contain_subp,
    container
  };

  ogmrip_plugin_foreach_subp_codec ((OGMRipPluginFunc) ogmrip_profile_editor_dialog_add_codecs, &data);
}

static void
ogmrip_profile_editor_dialog_check_container (GConfValue *value, OGMRipProfileEditorDialog *dialog)
{
  GType container;
  const gchar *name = NULL;

  if (value)
    name = gconf_value_get_string (value);
  container = ogmrip_profile_editor_dialog_get_container_type (dialog, name);

  if (container != G_TYPE_NONE)
  {
    GtkTreeModel *model;
    gchar *codec, *key;

    model = gtk_combo_box_get_model (GTK_COMBO_BOX (dialog->priv->video_codec_combo));
    gtk_list_store_clear (GTK_LIST_STORE (model));
    ogmrip_profile_editor_dialog_add_video_codecs (GTK_LIST_STORE (model), container);
    gtk_widget_set_sensitive (dialog->priv->video_codec_combo, gtk_tree_model_iter_n_children (model, NULL) > 0);

    key = gconf_concat_dir_and_key (dialog->priv->profile, OGMRIP_GCONF_VIDEO_CODEC);
    codec = ogmrip_preferences_get_string (key, OGMRIP_DEFAULT_VIDEO_CODEC);
    g_free (key);

    ogmrip_profile_editor_dialog_combo_set_active (dialog->priv->video_codec_combo, codec);
    g_free (codec);

    model = gtk_combo_box_get_model (GTK_COMBO_BOX (dialog->priv->audio_codec_combo));
    gtk_list_store_clear (GTK_LIST_STORE (model));
    ogmrip_profile_editor_dialog_add_audio_codecs (GTK_LIST_STORE (model), container);
    gtk_widget_set_sensitive (dialog->priv->audio_codec_combo, gtk_tree_model_iter_n_children (model, NULL) > 0);

    key = gconf_concat_dir_and_key (dialog->priv->profile, OGMRIP_GCONF_AUDIO_CODEC);
    codec = ogmrip_preferences_get_string (key, OGMRIP_DEFAULT_AUDIO_CODEC);
    g_free (key);

    ogmrip_profile_editor_dialog_combo_set_active (dialog->priv->audio_codec_combo, codec);
    g_free (codec);

    model = gtk_combo_box_get_model (GTK_COMBO_BOX (dialog->priv->subp_codec_combo));
    gtk_list_store_clear (GTK_LIST_STORE (model));
    ogmrip_profile_editor_dialog_add_subp_codecs (GTK_LIST_STORE (model), container);
    gtk_widget_set_sensitive (dialog->priv->subp_codec_combo, gtk_tree_model_iter_n_children (model, NULL) > 0);

    key = gconf_concat_dir_and_key (dialog->priv->profile, OGMRIP_GCONF_SUBP_CODEC);
    codec = ogmrip_preferences_get_string (key, OGMRIP_DEFAULT_SUBP_CODEC);
    g_free (key);

    ogmrip_profile_editor_dialog_combo_set_active (dialog->priv->subp_codec_combo, codec);
    g_free (codec);
  }
}

/*
 * Video Codec
 */

static gboolean
ogmrip_profile_editor_dialog_video_codec_set_value (GtkWidget *combo, GConfValue *value)
{
  GType codec;
  const gchar *name;

  if (!value)
    return FALSE;

  name = gconf_value_get_string (value);
  codec = ogmrip_gconf_get_video_codec_type (name);

  if (codec == G_TYPE_NONE)
  {
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);
    return FALSE;
  }

  ogmrip_profile_editor_dialog_combo_set_active (combo, name);

  return TRUE;
}

static void
ogmrip_profile_editor_dialog_connect_video_codec (OGMRipProfileEditorDialog *dialog, GtkWidget *widget, const gchar *key, const gchar *def)
{
  GConfValue *value;
  gchar *real_key;

  if (dialog->priv->profile)
    real_key = gconf_concat_dir_and_key (dialog->priv->profile, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_string (real_key, def);

  if (widget)
    ogmrip_preferences_connect_combo_custom (widget, real_key,
        ogmrip_profile_editor_dialog_combo_get_value,
        ogmrip_profile_editor_dialog_video_codec_set_value);

  g_free (real_key);
}

static void
ogmrip_profile_editor_dialog_check_video_codec (GConfValue *value, OGMRipProfileEditorDialog *dialog)
{
  GType codec;
  const gchar *name = NULL;

  if (value)
    name = gconf_value_get_string (value);
  codec = ogmrip_profile_editor_dialog_get_video_codec_type (dialog, name);

  if (codec != G_TYPE_NONE)
  {
    gint value;

    value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->priv->passes_spin));
    gtk_spin_button_set_range (GTK_SPIN_BUTTON (dialog->priv->passes_spin), 1.0,
        MIN (100, ogmrip_plugin_get_video_codec_passes (codec)));
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->passes_spin), value);

    if (dialog->priv->threads_spin)
    {
      value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->priv->threads_spin));
      gtk_spin_button_set_range (GTK_SPIN_BUTTON (dialog->priv->threads_spin), 1.0,
          MIN (100, ogmrip_plugin_get_video_codec_threads (codec)));
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->threads_spin), value);
    }
  }
}


/*
 * Audio Codec
 */

static gboolean
ogmrip_profile_editor_dialog_audio_codec_set_value (GtkWidget *combo, GConfValue *value)
{
  GType codec;
  const gchar *name;

  if (!value)
    return FALSE;

  name = gconf_value_get_string (value);
  codec = ogmrip_gconf_get_audio_codec_type (name);

  if (codec == G_TYPE_NONE)
  {
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);
    return FALSE;
  }

  ogmrip_profile_editor_dialog_combo_set_active (combo, name);

  return TRUE;
}

static void
ogmrip_profile_editor_dialog_connect_audio_codec (OGMRipProfileEditorDialog *dialog, GtkWidget *widget, const gchar *key, const gchar *def)
{
  GConfValue *value;
  gchar *real_key;

  if (dialog->priv->profile)
    real_key = gconf_concat_dir_and_key (dialog->priv->profile, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_string (real_key, def);

  if (widget)
    ogmrip_preferences_connect_combo_custom (widget, real_key,
        ogmrip_profile_editor_dialog_combo_get_value,
        ogmrip_profile_editor_dialog_audio_codec_set_value);

  g_free (real_key);
}

static void
ogmrip_profile_editor_dialog_check_audio_codec (GConfValue *value, OGMRipProfileEditorDialog *dialog)
{
}

/*
 * Subp Codec
 */

static gboolean
ogmrip_profile_editor_dialog_subp_codec_set_value (GtkWidget *combo, GConfValue *value)
{
  GType codec;
  const gchar *name;

  if (!value)
    return FALSE;

  name = gconf_value_get_string (value);
  codec = ogmrip_gconf_get_subp_codec_type (name);

  if (codec == G_TYPE_NONE)
  {
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);
    return FALSE;
  }

  ogmrip_profile_editor_dialog_combo_set_active (combo, name);

  return TRUE;
}

static void
ogmrip_profile_editor_dialog_connect_subp_codec (OGMRipProfileEditorDialog *dialog, GtkWidget *widget, const gchar *key, const gchar *def)
{
  GConfValue *value;
  gchar *real_key;

  if (dialog->priv->profile)
    real_key = gconf_concat_dir_and_key (dialog->priv->profile, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_string (real_key, def);

  if (widget)
    ogmrip_preferences_connect_combo_custom (widget, real_key,
        ogmrip_profile_editor_dialog_combo_get_value,
        ogmrip_profile_editor_dialog_subp_codec_set_value);

  g_free (real_key);
}

static void
ogmrip_profile_editor_dialog_check_subp_codec (GConfValue *value, OGMRipProfileEditorDialog *dialog)
{
  GType codec;
  const gchar *name = NULL;

  if (value)
    name = gconf_value_get_string (value);
  codec = ogmrip_profile_editor_dialog_get_subp_codec_type (dialog, name);

  if (codec != G_TYPE_NONE)
    gtk_widget_set_sensitive (dialog->priv->text_table, ogmrip_plugin_get_subp_codec_text (codec));
}

/*
 * Dialog
 */

G_DEFINE_TYPE (OGMRipProfileEditorDialog, ogmrip_profile_editor_dialog, GTK_TYPE_DIALOG);

static void
ogmrip_profile_editor_dialog_construct_container_combo (OGMRipProfileEditorDialog *dialog, GtkWidget *combo)
{
  GtkCellRenderer *cell;
  GtkListStore *store;
  gint n_containers;

  store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
  g_object_unref (store);

  cell = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), cell, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), cell, "text", 0, NULL);

  n_containers = ogmrip_plugin_get_n_containers ();
  gtk_widget_set_sensitive (combo, n_containers > 0);

  ogmrip_profile_editor_dialog_add_containers (store);
}

static void
ogmrip_profile_editor_dialog_container_options_button_clicked (OGMRipProfileEditorDialog *dialog)
{
  GtkWidget *plugin_dialog;
  GType container;

  container = ogmrip_profile_editor_dialog_get_container_type (dialog, NULL);
  plugin_dialog = ogmrip_container_options_plugin_dialog_new (container, dialog->priv->profile);
  if (plugin_dialog)
  {
    gtk_window_set_parent (GTK_WINDOW (plugin_dialog), GTK_WINDOW (dialog));
    g_signal_connect (plugin_dialog, "delete-event", G_CALLBACK (gtk_true), NULL);
    g_signal_connect (plugin_dialog, "response", G_CALLBACK (gtk_widget_destroy), NULL);
    gtk_widget_show_all (plugin_dialog);
  }
}

static void
ogmrip_profile_editor_dialog_construct_video_codec_combo (OGMRipProfileEditorDialog *dialog, GtkWidget *combo)
{
  GtkListStore *store;
  GtkCellRenderer *cell;
  gint n_codecs;

  store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
  g_object_unref (store);

  cell = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), cell, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), cell, "text", 0, NULL);

  n_codecs = ogmrip_plugin_get_n_video_codecs ();
  gtk_widget_set_sensitive (combo, n_codecs > 0);
}

static void
ogmrip_profile_editor_dialog_video_options_button_clicked (OGMRipProfileEditorDialog *dialog)
{
  GtkWidget *plugin_dialog;
  GType codec;

  codec = ogmrip_profile_editor_dialog_get_video_codec_type  (dialog, NULL);
  plugin_dialog = ogmrip_video_options_plugin_dialog_new (codec, dialog->priv->profile);
  if (plugin_dialog)
  {
    gtk_window_set_parent (GTK_WINDOW (plugin_dialog), GTK_WINDOW (dialog));
    g_signal_connect (plugin_dialog, "delete-event", G_CALLBACK (gtk_true), NULL);
    g_signal_connect (plugin_dialog, "response", G_CALLBACK (gtk_widget_destroy), NULL);
    gtk_widget_show_all (plugin_dialog);
  }
}

static void
ogmrip_profile_editor_dialog_construct_audio_codec_combo (OGMRipProfileEditorDialog *dialog, GtkWidget *combo)
{
  GtkListStore *store;
  GtkCellRenderer *cell;
  gint n_codecs;

  store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
  g_object_unref (store);

  cell = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), cell, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), cell, "text", 0, NULL);

  n_codecs = ogmrip_plugin_get_n_audio_codecs ();
  gtk_widget_set_sensitive (combo, n_codecs > 0);
}

static void
ogmrip_profile_editor_dialog_video_encoding_combo_changed (OGMRipProfileEditorDialog *dialog, GtkWidget *combo)
{
  gint active;
  gboolean expert = TRUE;

  active = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
  // expert = ogmrip_preferences_get_bool (OGMRIP_GCONF_EXPERT_MODE, OGMRIP_DEFAULT_EXPERT_MODE);

  gtk_widget_set_sensitive (dialog->priv->target_label, active == SIZE);
  gtk_widget_set_sensitive (dialog->priv->target_hbox, active == SIZE);

  gtk_widget_set_sensitive (dialog->priv->bitrate_label, active == BITRATE);
  gtk_widget_set_sensitive (dialog->priv->bitrate_hbox, active == BITRATE);

  gtk_widget_set_sensitive (dialog->priv->quantizer_label, active == QUANTIZER);
  gtk_widget_set_sensitive (dialog->priv->quantizer_spin, active == QUANTIZER);

  gtk_widget_set_sensitive (dialog->priv->passes_hbox, active != QUANTIZER);
  if (active == QUANTIZER)
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->passes_spin), 1);

  g_object_set (dialog->priv->target_label, "visible", expert || active == SIZE, NULL);
  g_object_set (dialog->priv->target_hbox,  "visible", expert || active == SIZE, NULL);

  g_object_set (dialog->priv->bitrate_label, "visible", expert || active == BITRATE, NULL);
  g_object_set (dialog->priv->bitrate_hbox,  "visible", expert || active == BITRATE, NULL);

  g_object_set (dialog->priv->quantizer_label, "visible", expert || active == QUANTIZER, NULL);
  g_object_set (dialog->priv->quantizer_spin,  "visible", expert || active == QUANTIZER, NULL);
}

static void
ogmrip_profile_editor_dialog_video_preset_combo_changed (OGMRipProfileEditorDialog *dialog, GtkWidget *combo)
{
  if (dialog->priv->video_options_button)
    gtk_widget_set_sensitive (dialog->priv->video_options_button,
        gtk_combo_box_get_active (GTK_COMBO_BOX (combo)) == OGMRIP_VIDEO_PRESET_USER);
}

static void
ogmrip_profile_editor_dialog_audio_options_button_clicked (OGMRipProfileEditorDialog *dialog)
{
  GtkWidget *plugin_dialog;
  GType codec;

  codec = ogmrip_profile_editor_dialog_get_audio_codec_type  (dialog, NULL);
  plugin_dialog = ogmrip_audio_options_plugin_dialog_new (codec, dialog->priv->profile);
  if (plugin_dialog)
  {
    gtk_window_set_parent (GTK_WINDOW (plugin_dialog), GTK_WINDOW (dialog));
    g_signal_connect (plugin_dialog, "delete-event", G_CALLBACK (gtk_true), NULL);
    g_signal_connect (plugin_dialog, "response", G_CALLBACK (gtk_widget_destroy), NULL);
    gtk_widget_show_all (plugin_dialog);
  }
}

static void
ogmrip_profile_editor_dialog_construct_subp_codec_combo (OGMRipProfileEditorDialog *dialog, GtkWidget *combo)
{
  GtkListStore *store;
  GtkCellRenderer *cell;
  gint n_codecs;

  store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
  g_object_unref (store);

  cell = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), cell, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), cell, "text", 0, NULL);

  n_codecs = ogmrip_plugin_get_n_subp_codecs ();
  gtk_widget_set_sensitive (combo, n_codecs > 0);
}

static void
ogmrip_profile_editor_dialog_container_combo_changed (OGMRipProfileEditorDialog *dialog, GtkWidget *combo)
{
  GtkTreeIter iter;

  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (combo), &iter))
  {
    GType codec;
    GtkTreeModel *model;
    gchar *name;

    model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo));
    gtk_tree_model_get (model, &iter, 1, &name, -1);

    codec = ogmrip_profile_editor_dialog_get_container_type (dialog, name);
    g_free (name);

    if (dialog->priv->container_options_button)
      gtk_widget_set_sensitive (dialog->priv->container_options_button,
          ogmrip_options_plugin_exists (codec));
  }
}

static void
ogmrip_profile_editor_dialog_video_codec_combo_changed (OGMRipProfileEditorDialog *dialog, GtkWidget *combo)
{
  GtkTreeIter iter;

  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (combo), &iter))
  {
    GType codec;
    GtkTreeModel *model;
    gboolean exists;
    gint active;
    gchar *name;

    model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo));
    gtk_tree_model_get (model, &iter, 1, &name, -1);

    codec = ogmrip_profile_editor_dialog_get_video_codec_type (dialog, name);
    g_free (name);

    gtk_combo_box_remove_text (GTK_COMBO_BOX (dialog->priv->video_preset_combo), OGMRIP_VIDEO_PRESET_USER);

    exists = ogmrip_options_plugin_exists (codec);
    if (exists)
      gtk_combo_box_append_text (GTK_COMBO_BOX (dialog->priv->video_preset_combo), _("User"));

    active = gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->priv->video_preset_combo));
    
    if (active < 0)
    {
      active = exists ? OGMRIP_VIDEO_PRESET_USER : OGMRIP_VIDEO_PRESET_EXTREME;
      gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->video_preset_combo), active);
    }
  }
}

static void
ogmrip_profile_editor_dialog_audio_codec_combo_changed (OGMRipProfileEditorDialog *dialog, GtkWidget *combo)
{
  GtkTreeIter iter;

  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (combo), &iter))
  {
    GType codec;
    GtkTreeModel *model;
    gchar *name;

    model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo));
    gtk_tree_model_get (model, &iter, 1, &name, -1);

    codec = ogmrip_profile_editor_dialog_get_audio_codec_type (dialog, name);
    g_free (name);

    if (dialog->priv->audio_options_button)
      gtk_widget_set_sensitive (dialog->priv->audio_options_button,
          ogmrip_options_plugin_exists (codec));
  }
}

static void
ogmrip_profile_editor_dialog_subp_codec_combo_changed (OGMRipProfileEditorDialog *dialog, GtkWidget *combo)
{
  GtkTreeIter iter;

  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (combo), &iter))
  {
    GType codec;
    GtkTreeModel *model;
    gchar *name;

    model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo));
    gtk_tree_model_get (model, &iter, 1, &name, -1);

    codec = ogmrip_profile_editor_dialog_get_subp_codec_type (dialog, name);
    g_free (name);

    if (dialog->priv->subp_options_button)
      gtk_widget_set_sensitive (dialog->priv->subp_options_button,
          ogmrip_options_plugin_exists (codec));
  }
}

static void
ogmrip_profile_editor_dialog_subp_options_button_clicked (OGMRipProfileEditorDialog *dialog)
{
  GtkWidget *plugin_dialog;
  GType codec;

  codec = ogmrip_profile_editor_dialog_get_subp_codec_type  (dialog, NULL);
  plugin_dialog = ogmrip_subp_options_plugin_dialog_new (codec, dialog->priv->profile);
  if (plugin_dialog)
  {
    gtk_window_set_parent (GTK_WINDOW (plugin_dialog), GTK_WINDOW (dialog));
    g_signal_connect (plugin_dialog, "delete-event", G_CALLBACK (gtk_true), NULL);
    g_signal_connect (plugin_dialog, "response", G_CALLBACK (gtk_widget_destroy), NULL);
    gtk_widget_show_all (plugin_dialog);
  }
}

static void
ogmrip_profile_editor_dialog_class_init (OGMRipProfileEditorDialogClass *klass)
{
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);
  gobject_class->finalize = ogmrip_profile_editor_dialog_finalize;

  g_type_class_add_private (klass, sizeof (OGMRipProfileEditorDialogPriv));
}

static void
ogmrip_profile_editor_dialog_construct (OGMRipProfileEditorDialog *dialog, GladeXML *xml)
{
  GtkWidget *widget;

  dialog->priv->root = glade_xml_get_widget (xml, OGMRIP_GLADE_ROOT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), dialog->priv->root, TRUE, TRUE, 0);
  gtk_widget_show (dialog->priv->root);

  /*
   * Container
   */

  widget = glade_xml_get_widget (xml, "container-combo");
  g_signal_connect_swapped (widget, "changed",
      G_CALLBACK (ogmrip_profile_editor_dialog_container_combo_changed), dialog);

  ogmrip_profile_editor_dialog_construct_container_combo (dialog, widget);
  ogmrip_profile_editor_dialog_connect_container (dialog, widget,
      OGMRIP_GCONF_CONTAINER_FORMAT, OGMRIP_DEFAULT_CONTAINER_FORMAT);

  dialog->priv->encoding_combo = glade_xml_get_widget (xml, "encoding-combo");
  g_signal_connect_swapped (dialog->priv->encoding_combo, "changed",
      G_CALLBACK (ogmrip_profile_editor_dialog_video_encoding_combo_changed), dialog);

  dialog->priv->target_label = glade_xml_get_widget (xml, "target-label");
  dialog->priv->target_hbox = glade_xml_get_widget (xml, "target-hbox");

  widget = glade_xml_get_widget (xml, "tnumber-spin");
  ogmrip_profile_editor_dialog_connect_spin (dialog, widget,
      OGMRIP_GCONF_CONTAINER_TNUMBER, OGMRIP_DEFAULT_CONTAINER_TNUMBER);

  widget = glade_xml_get_widget (xml, "tsize-spin");
  ogmrip_profile_editor_dialog_connect_spin (dialog, widget,
      OGMRIP_GCONF_CONTAINER_TSIZE, OGMRIP_DEFAULT_CONTAINER_TSIZE);

  dialog->priv->bitrate_label = glade_xml_get_widget (xml, "bitrate-label");
  dialog->priv->bitrate_hbox = glade_xml_get_widget (xml, "bitrate-hbox");

  widget = glade_xml_get_widget (xml, "bitrate-spin");
  ogmrip_profile_editor_dialog_connect_spin (dialog, widget,
      OGMRIP_GCONF_VIDEO_BITRATE, OGMRIP_DEFAULT_VIDEO_BITRATE);

  dialog->priv->quantizer_label = glade_xml_get_widget (xml, "quantizer-label");

  dialog->priv->quantizer_spin = glade_xml_get_widget (xml, "quantizer-spin");
  ogmrip_profile_editor_dialog_connect_spin (dialog, dialog->priv->quantizer_spin,
      OGMRIP_GCONF_VIDEO_QUANTIZER, OGMRIP_DEFAULT_VIDEO_QUANTIZER);

  /*
   * Video
   */

  dialog->priv->video_codec_combo = glade_xml_get_widget (xml, "video-codec-combo");
  g_signal_connect_swapped (dialog->priv->video_codec_combo, "changed",
      G_CALLBACK (ogmrip_profile_editor_dialog_video_codec_combo_changed), dialog);

  ogmrip_profile_editor_dialog_construct_video_codec_combo (dialog, dialog->priv->video_codec_combo);
  ogmrip_profile_editor_dialog_connect_video_codec (dialog, dialog->priv->video_codec_combo,
      OGMRIP_GCONF_VIDEO_CODEC, OGMRIP_DEFAULT_VIDEO_CODEC);

  dialog->priv->video_preset_combo = glade_xml_get_widget (xml, "preset-combo");
  g_signal_connect_swapped (dialog->priv->video_preset_combo, "changed",
      G_CALLBACK (ogmrip_profile_editor_dialog_video_preset_combo_changed), dialog);

  ogmrip_profile_editor_dialog_connect_combo (dialog, dialog->priv->video_preset_combo,
      OGMRIP_GCONF_VIDEO_PRESET, OGMRIP_DEFAULT_VIDEO_PRESET);

  dialog->priv->passes_spin = glade_xml_get_widget (xml, "passes-spin");
  ogmrip_profile_editor_dialog_connect_spin (dialog, dialog->priv->passes_spin,
      OGMRIP_GCONF_VIDEO_PASSES, OGMRIP_DEFAULT_VIDEO_PASSES);

  dialog->priv->passes_hbox = glade_xml_get_widget (xml, "passes-hbox");

  /*
   * Audio
   */

  dialog->priv->audio_codec_combo = glade_xml_get_widget (xml, "audio-codec-combo");
  g_signal_connect_swapped (dialog->priv->audio_codec_combo, "changed",
      G_CALLBACK (ogmrip_profile_editor_dialog_audio_codec_combo_changed), dialog);

  ogmrip_profile_editor_dialog_construct_audio_codec_combo (dialog, dialog->priv->audio_codec_combo);
  ogmrip_profile_editor_dialog_connect_audio_codec (dialog, dialog->priv->audio_codec_combo,
      OGMRIP_GCONF_AUDIO_CODEC, OGMRIP_DEFAULT_AUDIO_CODEC);

  widget = glade_xml_get_widget (xml, "quality-spin");
  ogmrip_profile_editor_dialog_connect_spin (dialog, widget,
      OGMRIP_GCONF_AUDIO_QUALITY, OGMRIP_DEFAULT_AUDIO_QUALITY);

  /*
   * Subp
   */

  dialog->priv->subp_codec_combo = glade_xml_get_widget (xml, "subp-codec-combo");
  g_signal_connect_swapped (dialog->priv->subp_codec_combo, "changed",
      G_CALLBACK (ogmrip_profile_editor_dialog_subp_codec_combo_changed), dialog);

  ogmrip_profile_editor_dialog_construct_subp_codec_combo (dialog, dialog->priv->subp_codec_combo);
  ogmrip_profile_editor_dialog_connect_subp_codec (dialog, dialog->priv->subp_codec_combo,
      OGMRIP_GCONF_SUBP_CODEC, OGMRIP_DEFAULT_SUBP_CODEC);

  widget = glade_xml_get_widget (xml, "spell-check");
  ogmrip_profile_editor_dialog_connect_toggle (dialog, widget,
      OGMRIP_GCONF_SPELL_CHECK, OGMRIP_DEFAULT_SPELL_CHECK);

  widget = glade_xml_get_widget (xml, "forced-subs-check");
  ogmrip_profile_editor_dialog_connect_toggle (dialog, widget,
      OGMRIP_GCONF_FORCED_SUBS, OGMRIP_DEFAULT_FORCED_SUBS);

  /*
   * Others
   */

  dialog->priv->text_table = glade_xml_get_widget (xml, "text-table");

  widget = glade_xml_get_widget (xml, "spell-check");
#if HAVE_ENCHANT_SUPPORT
  gtk_widget_set_sensitive (widget, TRUE);
#else
  gtk_widget_set_sensitive (widget, FALSE);
#endif

  ogmrip_profile_editor_dialog_connect_combo (dialog, dialog->priv->encoding_combo,
      OGMRIP_GCONF_VIDEO_ENCODING, OGMRIP_DEFAULT_VIDEO_ENCODING);
}

static void
ogmrip_profile_editor_dialog_construct_simple (OGMRipProfileEditorDialog *dialog)
{
  GladeXML *xml;
  GtkWidget *widget;

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_SIMPLE_FILE, OGMRIP_GLADE_ROOT, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_SIMPLE_FILE);
    return;
  }

  if (dialog->priv->root)
    gtk_container_remove (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), dialog->priv->root);

  dialog->priv->container_options_button = NULL;
  dialog->priv->video_options_button = NULL;
  dialog->priv->audio_options_button = NULL;
  dialog->priv->subp_options_button = NULL;
  dialog->priv->threads_spin = NULL;

  widget = glade_xml_get_widget (xml, "encoding-table");
  gtk_table_set_row_spacing (GTK_TABLE (widget), 0, 6);
  gtk_table_set_row_spacing (GTK_TABLE (widget), 1, 6);

  ogmrip_profile_editor_dialog_construct (dialog, xml);

  g_object_unref (xml);
}

static void
ogmrip_profile_editor_dialog_construct_expert (OGMRipProfileEditorDialog *dialog)
{
  GtkWidget *widget;
  GladeXML *xml;

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_EXPERT_FILE, OGMRIP_GLADE_ROOT, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_EXPERT_FILE);
    return;
  }

  dialog->priv->container_options_button = glade_xml_get_widget (xml, "container-options-button");
  g_signal_connect_swapped (dialog->priv->container_options_button, "clicked",
      G_CALLBACK (ogmrip_profile_editor_dialog_container_options_button_clicked), dialog);

  dialog->priv->video_options_button = glade_xml_get_widget (xml, "video-options-button");
  g_signal_connect_swapped (dialog->priv->video_options_button, "clicked",
      G_CALLBACK (ogmrip_profile_editor_dialog_video_options_button_clicked), dialog);

  dialog->priv->audio_options_button = glade_xml_get_widget (xml, "audio-options-button");
  g_signal_connect_swapped (dialog->priv->audio_options_button, "clicked",
      G_CALLBACK (ogmrip_profile_editor_dialog_audio_options_button_clicked), dialog);

  dialog->priv->subp_options_button = glade_xml_get_widget (xml, "subp-options-button");
  g_signal_connect_swapped (dialog->priv->subp_options_button, "clicked",
      G_CALLBACK (ogmrip_profile_editor_dialog_subp_options_button_clicked), dialog);

  if (dialog->priv->root)
    gtk_container_remove (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), dialog->priv->root);

  ogmrip_profile_editor_dialog_construct (dialog, xml);

  /*
   * Container
   */

  widget = glade_xml_get_widget (xml, "fourcc-combo");
  ogmrip_profile_editor_dialog_connect_combo (dialog, widget,
      OGMRIP_GCONF_CONTAINER_FOURCC, OGMRIP_DEFAULT_CONTAINER_FOURCC);

  widget = glade_xml_get_widget (xml, "ensure-sync-check");
  ogmrip_profile_editor_dialog_connect_toggle (dialog, widget,
      OGMRIP_GCONF_CONTAINER_ENSURE_SYNC, OGMRIP_DEFAULT_CONTAINER_ENSURE_SYNC);

  /*
   * Video
   */

  widget = glade_xml_get_widget (xml, "threads-spin");
  ogmrip_profile_editor_dialog_connect_spin (dialog, widget,
      OGMRIP_GCONF_VIDEO_THREADS, OGMRIP_DEFAULT_VIDEO_THREADS);

  widget = glade_xml_get_widget (xml, "trellis-check");
  ogmrip_profile_editor_dialog_connect_toggle (dialog, widget,
      OGMRIP_GCONF_VIDEO_TRELLIS, OGMRIP_DEFAULT_VIDEO_TRELLIS);

  widget = glade_xml_get_widget (xml, "qpel-check");
  ogmrip_profile_editor_dialog_connect_toggle (dialog, widget,
      OGMRIP_GCONF_VIDEO_QPEL, OGMRIP_DEFAULT_VIDEO_QPEL);

  widget = glade_xml_get_widget (xml, "turbo-check");
  ogmrip_profile_editor_dialog_connect_toggle (dialog, widget,
      OGMRIP_GCONF_VIDEO_TURBO, OGMRIP_DEFAULT_VIDEO_TURBO);

  widget = glade_xml_get_widget (xml, "denoise-check");
  ogmrip_profile_editor_dialog_connect_toggle (dialog, widget,
      OGMRIP_GCONF_VIDEO_DENOISE, OGMRIP_DEFAULT_VIDEO_DENOISE);

  widget = glade_xml_get_widget (xml, "deblock-check");
  ogmrip_profile_editor_dialog_connect_toggle (dialog, widget,
      OGMRIP_GCONF_VIDEO_DEBLOCK, OGMRIP_DEFAULT_VIDEO_DEBLOCK);

  widget = glade_xml_get_widget (xml, "dering-check");
  ogmrip_profile_editor_dialog_connect_toggle (dialog, widget,
      OGMRIP_GCONF_VIDEO_DERING, OGMRIP_DEFAULT_VIDEO_DERING);

  widget = glade_xml_get_widget (xml, "scaler-combo");
  ogmrip_profile_editor_dialog_connect_combo (dialog, widget,
      OGMRIP_GCONF_VIDEO_SCALER, OGMRIP_DEFAULT_VIDEO_SCALER);

  /*
   * Audio
   */

  widget = glade_xml_get_widget (xml, "normalize-check");
  ogmrip_profile_editor_dialog_connect_toggle (dialog, widget,
      OGMRIP_GCONF_AUDIO_NORMALIZE, OGMRIP_DEFAULT_AUDIO_NORMALIZE);

  widget = glade_xml_get_widget (xml, "srate-combo");
  ogmrip_profile_editor_dialog_connect_combo (dialog, widget,
      OGMRIP_GCONF_AUDIO_SRATE, OGMRIP_DEFAULT_AUDIO_SRATE);

  widget = glade_xml_get_widget (xml, "channels-combo");
  ogmrip_profile_editor_dialog_connect_combo (dialog, widget,
      OGMRIP_GCONF_AUDIO_CHANNELS, OGMRIP_DEFAULT_AUDIO_CHANNELS);

  /*
   * Subp
   */

  widget = glade_xml_get_widget (xml, "charset-combo");
  ogmrip_profile_editor_dialog_connect_combo (dialog, widget,
      OGMRIP_GCONF_SUBP_CHARSET, OGMRIP_DEFAULT_SUBP_CHARSET);

  widget = glade_xml_get_widget (xml, "eol-combo");
  ogmrip_profile_editor_dialog_connect_combo (dialog, widget,
      OGMRIP_GCONF_SUBP_EOL, OGMRIP_DEFAULT_SUBP_EOL);

  g_object_unref (xml);
}
/*
static void
ogmrip_profile_editor_dialog_expert_notify (GConfValue *value, OGMRipProfileEditorDialog *dialog)
{
  gboolean expert = FALSE;

  if (value)
    expert = gconf_value_get_bool (value);

  if (expert)
    ogmrip_profile_editor_dialog_construct_expert (dialog);
  else
    ogmrip_profile_editor_dialog_construct_simple (dialog);

  ogmrip_profile_editor_dialog_check_container (NULL, dialog);
  ogmrip_profile_editor_dialog_check_video_codec (NULL, dialog);
  ogmrip_profile_editor_dialog_check_audio_codec (NULL, dialog);
  ogmrip_profile_editor_dialog_check_subp_codec (NULL, dialog);

  ogmrip_profile_editor_dialog_video_encoding_combo_changed (dialog, dialog->priv->encoding_combo);
}
*/
static void
ogmrip_profile_editor_dialog_init (OGMRipProfileEditorDialog *dialog)
{
/*
  GtkWidget *widget;
*/
  gchar *key;
  gboolean expert = TRUE;

  dialog->priv = OGMRIP_PROFILE_EDITOR_DIALOG_GET_PRIVATE (dialog);

  dialog->priv->profile = g_strdup (gconf_dir);

  gtk_dialog_add_buttons (GTK_DIALOG (dialog),
      GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
      NULL);
  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_CLOSE);
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_PREFERENCES);
/*
  expert = ogmrip_preferences_get_bool (OGMRIP_GCONF_EXPERT_MODE, OGMRIP_DEFAULT_EXPERT_MODE);
*/
  if (expert)
    ogmrip_profile_editor_dialog_construct_expert (dialog);
  else
    ogmrip_profile_editor_dialog_construct_simple (dialog);
/*
  widget = gtk_check_button_new_with_mnemonic (_("_Expert mode"));
  gtk_box_pack_end (GTK_BOX (GTK_DIALOG (dialog)->vbox), widget, TRUE, TRUE, 0);
  gtk_widget_show (widget);

  ogmrip_profile_editor_dialog_connect_toggle (dialog, widget,
      OGMRIP_GCONF_EXPERT_MODE, OGMRIP_DEFAULT_EXPERT_MODE);
  ogmrip_preferences_add_notify_while_alive (OGMRIP_GCONF_EXPERT_MODE,
      (GFunc) ogmrip_profile_editor_dialog_expert_notify, dialog, widget);
*/
  key = gconf_concat_dir_and_key (gconf_dir, OGMRIP_GCONF_CONTAINER_FORMAT);
  ogmrip_preferences_add_notify_while_alive (key,
      (GFunc) ogmrip_profile_editor_dialog_check_container, dialog, dialog);
  ogmrip_profile_editor_dialog_check_container (NULL, dialog);
  g_free (key);

  key = gconf_concat_dir_and_key (gconf_dir, OGMRIP_GCONF_VIDEO_CODEC);
  ogmrip_preferences_add_notify_while_alive (key,
      (GFunc) ogmrip_profile_editor_dialog_check_video_codec, dialog, dialog);
  ogmrip_profile_editor_dialog_check_video_codec (NULL, dialog);
  g_free (key);

  key = gconf_concat_dir_and_key (gconf_dir, OGMRIP_GCONF_AUDIO_CODEC);
  ogmrip_preferences_add_notify_while_alive (key,
      (GFunc) ogmrip_profile_editor_dialog_check_audio_codec, dialog, dialog);
  ogmrip_profile_editor_dialog_check_audio_codec (NULL, dialog);
  g_free (key);

  key = gconf_concat_dir_and_key (gconf_dir, OGMRIP_GCONF_SUBP_CODEC);
  ogmrip_preferences_add_notify_while_alive (key,
      (GFunc) ogmrip_profile_editor_dialog_check_subp_codec, dialog, dialog);
  ogmrip_profile_editor_dialog_check_subp_codec (NULL, dialog);
  g_free (key);
}

static void
ogmrip_profile_editor_dialog_finalize (GObject *gobject)
{
  OGMRipProfileEditorDialog *dialog;

  dialog = OGMRIP_PROFILE_EDITOR_DIALOG (gobject);

  g_free (dialog->priv->profile);
  dialog->priv->profile = NULL;

  (*G_OBJECT_CLASS (ogmrip_profile_editor_dialog_parent_class)->finalize) (gobject);
}

GtkWidget *
ogmrip_profile_editor_dialog_new (const gchar *key)
{
  gconf_dir = key;

  return g_object_new (OGMRIP_TYPE_PROFILE_EDITOR_DIALOG, NULL);
}

