/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2009 Olivier Rolland <billl@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ogmrip-fs.h"
#include "ogmrip-mplayer.h"
#include "ogmrip-plugin.h"
#include "ogmrip-version.h"
#include "ogmrip-x264.h"

#include "ogmjob-exec.h"
#include "ogmjob-queue.h"

#include <math.h>
#include <stdio.h>
#include <unistd.h>
#include <glib/gstdio.h>
#include <glib/gi18n-lib.h>

#define OGMRIP_X264_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_X264, OGMRipX264Priv))

enum
{
  PROP_0,
  PROP_B_PYRAMID,
  PROP_BRDO,
  PROP_PSY_RD,
  PROP_PSY_TRELLIS,
  PROP_CABAC,
  PROP_FRAMEREF,
  PROP_LEVEL_IDC,
  PROP_ME,
  PROP_MIXED_REFS,
  PROP_SUBQ,
  PROP_VBV_BUFSIZE,
  PROP_VBV_MAXRATE,
  PROP_WEIGHT_B,
  PROP_8X8DCT,
  PROP_GLOBAL_HEADER
};

static void ogmrip_x264_get_property    (GObject           *gobject,
                                         guint             property_id,
                                         GValue            *value,
                                         GParamSpec        *pspec);
static void ogmrip_x264_set_property    (GObject           *gobject,
                                         guint             property_id,
                                         const GValue      *value,
                                         GParamSpec        *pspec);
static gint ogmrip_x264_run             (OGMJobSpawn       *spawn);
static gint ogmrip_x264_get_start_delay (OGMRipVideoCodec  *video);
static void ogmrip_x264_set_quality     (OGMRipVideoCodec  *video,
                                         OGMRipQualityType quality);

struct _OGMRipX264Priv
{
  guint frameref;
  guint level_idc;
  guint me;
  guint subq;
  guint vbv_bufsize;
  guint vbv_maxrate;
  gboolean b_pyramid;
  gboolean brdo;
  gboolean cabac;
  gboolean mixed_refs;
  gboolean weight_b;
  gboolean x88dct;
  gboolean global_header;
  gdouble psy_rd;
  gdouble psy_trellis;
};

static const gchar *me_name[] =
{
  NULL, "dia", "hex", "umh", "esa"
};

gboolean x264_have_brdo = FALSE;
gboolean x264_have_bime = FALSE;
gboolean x264_have_psy  = FALSE;

static gint
ogmrip_x264_get_crf (OGMRipVideoCodec *video)
{
  gdouble quantizer;
  gint crf;

  quantizer = ogmrip_video_codec_get_quantizer (video);
  if (quantizer < 0)
    quantizer = 2.3;

  crf = 12 + (unsigned int) (6.0 * log (quantizer) / log (2.0));

  return CLAMP (crf, 1, 50);
}

static gchar **
ogmrip_x264_command (OGMRipVideoCodec *video, guint pass, guint passes, const gchar *log_file)
{
  OGMRipX264 *x264;
  OGMDvdTitle *title;
  GPtrArray *argv;
  GString *options;

  const gchar *output;
  gint quality, bitrate, vid, threads, bframes;
  gboolean trellis;

  g_return_val_if_fail (OGMRIP_IS_VIDEO_CODEC (video), NULL);

  output = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_val_if_fail (title != NULL, NULL);

  g_return_val_if_fail (pass == 1 || log_file != NULL, NULL);

  x264 = OGMRIP_X264 (video);
  quality = ogmrip_video_codec_get_quality (video);

  argv = ogmrip_mencoder_video_command (video, pass == passes ? output : "/dev/null", pass);

  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("x264"));

  options = g_string_new (NULL);
  g_string_append_printf (options, "subq=%u:frameref=%u", x264->priv->subq, x264->priv->frameref);

  if (x264->priv->b_pyramid)
    g_string_append (options, ":b_pyramid");
  if (x264->priv->weight_b)
    g_string_append (options, ":weight_b");
  if (x264->priv->global_header)
    g_string_append (options, ":global_header");

  if (MPLAYER_CHECK_VERSION (1,0,1,0))
  {
    if (x264_have_brdo && x264->priv->brdo)
      g_string_append (options, ":brdo");
    if (x264->priv->x88dct)
      g_string_append (options, ":8x8dct");
    if (x264->priv->mixed_refs)
      g_string_append (options, ":mixed_refs");
    g_string_append_printf (options, ":me=%s", me_name[CLAMP (x264->priv->me, 1, 4)]);
  }
  else
    g_string_append_printf (options, ":me=%u", x264->priv->me);

  if (x264->priv->vbv_maxrate > 0 && x264->priv->vbv_bufsize > 0)
    g_string_append_printf (options, ":vbv_maxrate=%d:vbv_bufsize=%d",
        x264->priv->vbv_maxrate, x264->priv->vbv_bufsize);

  if (x264->priv->level_idc > 0)
    g_string_append_printf (options, ":level_idc=%d", CLAMP (x264->priv->level_idc, 10, 51));

  g_string_append (options, x264->priv->cabac ? ":cabac" : ":nocabac");

  if (MPLAYER_CHECK_VERSION (1,0,0,8) && pass != passes)
  {
    if (ogmrip_video_codec_get_turbo (video))
      g_string_append (options, ":turbo=2");
    else
      g_string_append (options, ":turbo=1");
  }

  if (ogmrip_video_codec_get_4mv (video))
  {
    if (MPLAYER_CHECK_VERSION (1,0,1,0))
      g_string_append (options, ":partitions=all");
      // g_string_append (options, ":partitions=p8x8,b8x8,i8x8,i4x4");
    else
      g_string_append (options, ":4x4mv");
  }

  trellis = ogmrip_video_codec_get_trellis (video);
  if (trellis)
  {
    if (quality == OGMRIP_QUALITY_EXTREME)
      g_string_append (options, ":trellis=2");
    else
      g_string_append (options, ":trellis=1");
  }
  else
    g_string_append (options, ":trellis=0");

  bframes = ogmrip_video_codec_get_max_b_frames (video);
  g_string_append_printf (options, ":bframes=%d", bframes);

  if (x264_have_bime && bframes > 0)
    g_string_append (options, ":bime");

  if (x264_have_psy)
  {
    gchar psy_rd[G_ASCII_DTOSTR_BUF_SIZE], psy_trellis[G_ASCII_DTOSTR_BUF_SIZE];

    g_ascii_formatd (psy_rd, G_ASCII_DTOSTR_BUF_SIZE, "%.1f", x264->priv->subq >= 6 ? x264->priv->psy_rd : 0.0);
    g_ascii_formatd (psy_trellis, G_ASCII_DTOSTR_BUF_SIZE, "%.1f", trellis ? x264->priv->psy_trellis : 0.0);

    g_string_append_printf (options, ":psy-rd=%s,%s", psy_rd, psy_trellis);
  }

  bitrate = ogmrip_video_codec_get_bitrate (video);
  if (bitrate > 0)
    g_string_append_printf (options, ":bitrate=%u", bitrate / 1000);
  else
    g_string_append_printf (options, ":crf=%u", ogmrip_x264_get_crf (video));

  if (passes > 1 && log_file)
  {
    if (pass == 1)
      g_string_append (options, ":pass=1");
    else
    {
      if (passes == 2)
        g_string_append (options, ":pass=2");
      else
        g_string_append (options, ":pass=3");
    }
    g_string_append (options, ":direct_pred=auto");

    g_ptr_array_add (argv, g_strdup ("-passlogfile"));
    g_ptr_array_add (argv, g_strdup (log_file));
  }
  
  threads = ogmrip_video_codec_get_threads (video);
  if (threads > 0)
    g_string_append_printf (options, ":threads=%u", CLAMP (threads, 1, 16));
  else
    g_string_append (options, ":threads=auto");

  if (!ogmrip_video_codec_get_cartoon (video) && quality == OGMRIP_QUALITY_EXTREME)
    g_string_append (options, ":nodct_decimate");

  g_ptr_array_add (argv, g_strdup ("-x264encopts"));
  g_ptr_array_add (argv, g_string_free (options, FALSE));

  vid = ogmdvd_title_get_nr (title);

  if (MPLAYER_CHECK_VERSION (1,0,0,1))
    g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
  else
  {
    g_ptr_array_add (argv, g_strdup ("-dvd"));
    g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
  }

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

G_DEFINE_TYPE (OGMRipX264, ogmrip_x264, OGMRIP_TYPE_VIDEO_CODEC)

static void
ogmrip_x264_class_init (OGMRipX264Class *klass)
{
  GObjectClass *gobject_class;
  OGMJobSpawnClass *spawn_class;
  OGMRipVideoCodecClass *video_class;

  gobject_class = G_OBJECT_CLASS (klass);
  gobject_class->get_property = ogmrip_x264_get_property;
  gobject_class->set_property = ogmrip_x264_set_property;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);
  spawn_class->run = ogmrip_x264_run;

  video_class = OGMRIP_VIDEO_CODEC_CLASS (klass);
  video_class->get_start_delay = ogmrip_x264_get_start_delay;
  video_class->set_quality = ogmrip_x264_set_quality;

  g_object_class_install_property (gobject_class, PROP_FRAMEREF, 
        g_param_spec_uint ("frameref", "Frameref property", "Set frameref", 
           1, 15, 1, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_LEVEL_IDC, 
        g_param_spec_uint ("level_idc", "Level IDC property", "Set level IDC", 
           0, 51, 51, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_ME, 
        g_param_spec_uint ("me", "Motion estimation property", "Set motion estimation", 
           1, 4, 2, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_SUBQ, 
        g_param_spec_uint ("subq", "Subpel quality property", "Set subpel quality", 
           1, 7, 5, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_VBV_BUFSIZE, 
        g_param_spec_uint ("vbv_bufsize", "Buffer size property", "Set buffer size", 
           0, G_MAXINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_VBV_MAXRATE, 
        g_param_spec_uint ("vbv_maxrate", "Max rate property", "Set max rate", 
           0, G_MAXINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_B_PYRAMID, 
        g_param_spec_boolean ("b_pyramid", "B pyramid property", "Set b pyramid", 
           FALSE, G_PARAM_READWRITE));

  if (x264_have_brdo)
    g_object_class_install_property (gobject_class, PROP_BRDO, 
          g_param_spec_boolean ("brdo", "Brdo property", "Set brdo", 
             FALSE, G_PARAM_READWRITE));

  if (x264_have_psy)
  {
    g_object_class_install_property (gobject_class, PROP_PSY_RD, 
          g_param_spec_double ("psy_rd", "Psy RD property", "Set psy-rd", 
             0.0, G_MAXDOUBLE, 1.0, G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class, PROP_PSY_TRELLIS, 
          g_param_spec_double ("psy_trellis", "Psy trellis property", "Set psy-trellis", 
             0.0, G_MAXDOUBLE, 0.0, G_PARAM_READWRITE));
  }

  g_object_class_install_property (gobject_class, PROP_CABAC, 
        g_param_spec_boolean ("cabac", "Cabac property", "Set cabac", 
           TRUE, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_MIXED_REFS, 
        g_param_spec_boolean ("mixed_refs", "Mixed refs property", "Set mixed refs", 
           FALSE, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_WEIGHT_B, 
        g_param_spec_boolean ("weight_b", "Weight B property", "Set weight B", 
           FALSE, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_8X8DCT, 
        g_param_spec_boolean ("dct8x8", "8x8 dct property", "Set 8x8 dct", 
           FALSE, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_GLOBAL_HEADER, 
        g_param_spec_boolean ("global_header", "global header property", "Set global header", 
           FALSE, G_PARAM_READWRITE));

  g_type_class_add_private (klass, sizeof (OGMRipX264Priv));
}

static void
ogmrip_x264_init (OGMRipX264 *x264)
{
  x264->priv = OGMRIP_X264_GET_PRIVATE (x264);
  x264->priv->cabac = TRUE;
}

static void
ogmrip_x264_get_property (GObject *gobject, guint property_id, GValue *value, GParamSpec *pspec)
{
  OGMRipX264 *x264;

  x264 = OGMRIP_X264 (gobject);

  switch (property_id) 
  {
    case PROP_B_PYRAMID:
      g_value_set_boolean (value, x264->priv->b_pyramid);
      break;
    case PROP_BRDO:
      g_value_set_boolean (value, x264->priv->brdo);
      break;
    case PROP_CABAC:
      g_value_set_boolean (value, x264->priv->cabac);
      break;
    case PROP_FRAMEREF:
      g_value_set_uint (value, x264->priv->frameref);
      break;
    case PROP_LEVEL_IDC:
      g_value_set_uint (value, x264->priv->level_idc);
      break;
    case PROP_ME:
      g_value_set_uint (value, x264->priv->me);
      break;
    case PROP_MIXED_REFS:
      g_value_set_boolean (value, x264->priv->mixed_refs);
      break;
    case PROP_SUBQ:
      g_value_set_uint (value, x264->priv->subq);
      break;
    case PROP_VBV_BUFSIZE:
      g_value_set_uint (value, x264->priv->vbv_bufsize);
      break;
    case PROP_VBV_MAXRATE:
      g_value_set_uint (value, x264->priv->vbv_maxrate);
      break;
    case PROP_WEIGHT_B:
      g_value_set_boolean (value, x264->priv->weight_b);
      break;
    case PROP_8X8DCT:
      g_value_set_boolean (value, x264->priv->x88dct);
      break;
    case PROP_GLOBAL_HEADER:
      g_value_set_boolean (value, x264->priv->global_header);
      break;
    case PROP_PSY_RD:
      g_value_set_double (value, x264->priv->psy_rd);
      break;
    case PROP_PSY_TRELLIS:
      g_value_set_double (value, x264->priv->psy_trellis);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, property_id, pspec);
      break;
  }
}

static void
ogmrip_x264_set_property (GObject *gobject, guint property_id, const GValue *value, GParamSpec *pspec)
{
  OGMRipX264 *x264;

  x264 = OGMRIP_X264 (gobject);

  switch (property_id) 
  {
    case PROP_B_PYRAMID:
      x264->priv->b_pyramid = g_value_get_boolean (value);
      break;
    case PROP_BRDO:
      x264->priv->brdo = g_value_get_boolean (value);
      break;
    case PROP_CABAC:
      x264->priv->cabac = g_value_get_boolean (value);
      break;
    case PROP_FRAMEREF:
      x264->priv->frameref = g_value_get_uint (value);
      break;
    case PROP_LEVEL_IDC:
      x264->priv->level_idc = g_value_get_uint (value);
      break;
    case PROP_ME:
      x264->priv->me = g_value_get_uint (value);
      break;
    case PROP_MIXED_REFS:
      x264->priv->mixed_refs = g_value_get_boolean (value);
      break;
    case PROP_SUBQ:
      x264->priv->subq = g_value_get_uint (value);
      break;
    case PROP_VBV_BUFSIZE:
      x264->priv->vbv_bufsize = g_value_get_uint (value);
      break;
    case PROP_VBV_MAXRATE:
      x264->priv->vbv_maxrate = g_value_get_uint (value);
      break;
    case PROP_WEIGHT_B:
      x264->priv->weight_b = g_value_get_boolean (value);
      break;
    case PROP_8X8DCT:
      x264->priv->x88dct = g_value_get_boolean (value);
      break;
    case PROP_GLOBAL_HEADER:
      x264->priv->global_header = g_value_get_boolean (value);
      break;
    case PROP_PSY_RD:
      x264->priv->psy_rd = g_value_get_double (value);
      break;
    case PROP_PSY_TRELLIS:
      x264->priv->psy_trellis = g_value_get_double (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, property_id, pspec);
      break;
  }
}

static gint
ogmrip_x264_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *queue, *child;
  gchar **argv, *log_file;
  gint pass, passes, result;

  queue = ogmjob_queue_new ();
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), queue);
  g_object_unref (queue);

  passes = ogmrip_video_codec_get_passes (OGMRIP_VIDEO_CODEC (spawn));

  log_file = NULL;
  if (passes > 1)
    log_file = ogmrip_fs_mktemp ("log.XXXXXX", NULL);

  for (pass = 0; pass < passes; pass ++)
  {
    argv = ogmrip_x264_command (OGMRIP_VIDEO_CODEC (spawn), pass + 1, passes, log_file);
    if (!argv)
      return OGMJOB_RESULT_ERROR;

    child = ogmjob_exec_newv (argv);
    ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_mencoder_codec_watch, spawn, TRUE, FALSE, FALSE);
    ogmjob_container_add (OGMJOB_CONTAINER (queue), child);
    g_object_unref (child);
  }

  result = OGMJOB_SPAWN_CLASS (ogmrip_x264_parent_class)->run (spawn);

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), queue);

  g_unlink (log_file);
  g_free (log_file);

  return result;
}

static gint
ogmrip_x264_get_start_delay (OGMRipVideoCodec *video)
{
  if (ogmrip_video_codec_get_max_b_frames (video) > 0)
    return 2;
  
  return 1;
}

static void
ogmrip_x264_set_quality (OGMRipVideoCodec *video, OGMRipQualityType quality)
{
  OGMRipX264 *x264;

  x264 = OGMRIP_X264 (video);

  switch (quality)
  {
    case OGMRIP_QUALITY_EXTREME:
      ogmrip_video_codec_set_max_b_frames (video, 3);
      ogmrip_video_codec_set_4mv (video, TRUE);
      x264->priv->subq = 6;
      x264->priv->b_pyramid = TRUE;
      x264->priv->weight_b = TRUE;
      x264->priv->frameref = 5;
      x264->priv->me = 3;
      break;
    case OGMRIP_QUALITY_HIGH:
      ogmrip_video_codec_set_max_b_frames (video, 3);
      ogmrip_video_codec_set_4mv (video, FALSE);
      x264->priv->subq = 5;
      x264->priv->b_pyramid = TRUE;
      x264->priv->weight_b = TRUE;
      x264->priv->frameref = 2;
      x264->priv->me = 2;
      break;
    case OGMRIP_QUALITY_NORMAL:
      ogmrip_video_codec_set_max_b_frames (video, 2);
      ogmrip_video_codec_set_4mv (video, FALSE);
      x264->priv->subq = 4;
      x264->priv->b_pyramid = TRUE;
      x264->priv->weight_b = TRUE;
      x264->priv->frameref = 1;
      x264->priv->me = 2;
      break;
  }

  if (MPLAYER_CHECK_VERSION (1,0,1,0))
  {
    switch (quality)
    {
      case OGMRIP_QUALITY_EXTREME:
        x264->priv->brdo = TRUE;
        x264->priv->x88dct = TRUE;
        x264->priv->mixed_refs = TRUE;
        x264->priv->psy_rd = 1.0;
        break;
      case OGMRIP_QUALITY_HIGH:
        x264->priv->brdo = FALSE;
        x264->priv->x88dct = TRUE;
        x264->priv->mixed_refs = TRUE;
        x264->priv->psy_rd = 0.0;
        break;
      case OGMRIP_QUALITY_NORMAL:
        x264->priv->brdo = FALSE;
        x264->priv->x88dct = FALSE;
        x264->priv->mixed_refs = FALSE;
        x264->priv->psy_rd = 0.0;
        break;
    }
  }
}

/**
 * ogmrip_x264_new:
 * @title: An #OGMDvdTitle
 * @output: The output file
 *
 * Creates a new #OGMRipX264.
 *
 * Returns: the new #OGMRipX264
 */
OGMJobSpawn *
ogmrip_x264_new (OGMDvdTitle *title, const gchar *output)
{
  g_return_val_if_fail (title != NULL, NULL);
  g_return_val_if_fail (output && *output, NULL);

  return g_object_new (OGMRIP_TYPE_X264, "input", title, "output", output, NULL);
}

static OGMRipVideoPlugin x264_plugin =
{
  NULL,
  G_TYPE_NONE,
  "x264",
  N_("X264"),
  OGMRIP_FORMAT_H264,
  G_MAXINT,
  16
};

static gboolean
ogmrip_x264_check_option (const gchar *option)
{
  GPtrArray *argv;
  gint status = 1;

  argv = g_ptr_array_new ();

  g_ptr_array_add (argv, g_strdup ("mencoder"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));
  g_ptr_array_add (argv, g_strdup ("-nosound"));
  g_ptr_array_add (argv, g_strdup ("-really-quiet"));
  g_ptr_array_add (argv, g_strdup ("-frames"));
  g_ptr_array_add (argv, g_strdup ("0"));
  g_ptr_array_add (argv, g_strdup ("-rawvideo"));
  g_ptr_array_add (argv, g_strdup ("pal:fps=2"));
  g_ptr_array_add (argv, g_strdup ("-demuxer"));
  g_ptr_array_add (argv, g_strdup ("rawvideo"));
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup ("/dev/null"));
  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("x264"));
  g_ptr_array_add (argv, g_strdup ("-x264encopts"));
  g_ptr_array_add (argv, g_strdup_printf ("%s:bitrate=800", option));
  g_ptr_array_add (argv, g_strdup ("/dev/zero"));
  g_ptr_array_add (argv, NULL);

  g_spawn_sync (NULL, (gchar **) argv->pdata, NULL,
      G_SPAWN_SEARCH_PATH | G_SPAWN_STDOUT_TO_DEV_NULL | G_SPAWN_STDERR_TO_DEV_NULL,
      NULL, NULL, NULL, NULL, &status, NULL);
  
  g_ptr_array_free (argv, TRUE);

  return status == 0;

}

OGMRipVideoPlugin *
ogmrip_init_plugin (void)
{
  gchar *output;
  gboolean match;

  if (!ogmrip_check_mencoder ())
    return NULL;

  if (!g_spawn_command_line_sync ("mencoder -ovc help", &output, NULL, NULL, NULL))
    return NULL;

  match = g_regex_match_simple ("^ *x264 *- .*$", output, G_REGEX_MULTILINE, 0);
  g_free (output);

  if (!match)
    return NULL;

  x264_have_brdo = ogmrip_x264_check_option ("brdo");
  x264_have_bime = ogmrip_x264_check_option ("bime");
  x264_have_psy  = ogmrip_x264_check_option ("psy_rd=1,1");

  x264_plugin.type = OGMRIP_TYPE_X264;

  return &x264_plugin;
}

