/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2009 Olivier Rolland <billl@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n.h>
#include <gtk/gtkdialog.h>
#include <glade/glade.h>

#include "ogmrip-plugin.h"
#include "ogmrip-options-plugin.h"
#include "ogmrip-preferences.h"
#include "ogmrip-helper.h"
#include "ogmrip-lavc.h"

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-lavc.glade"
#define OGMRIP_GLADE_ROOT "root"

#define OGMRIP_TYPE_LAVC_DIALOG          (ogmrip_lavc_dialog_get_type ())
#define OGMRIP_LAVC_DIALOG(obj)          (G_TYPE_CHECK_INSTANCE_CAST ((obj), OGMRIP_TYPE_LAVC_DIALOG, OGMRipLavcDialog))
#define OGMRIP_LAVC_DIALOG_CLASS(klass)  (G_TYPE_CHECK_CLASS_CAST ((klass), OGMRIP_TYPE_LAVC_DIALOG, OGMRipLavcDialogClass))
#define OGMRIP_IS_LAVC_DIALOG(obj)       (G_TYPE_CHECK_INSTANCE_TYPE ((obj), OGMRIP_TYPE_LAVC_DIALOG))
#define OGMRIP_IS_LAVC_DIALOG_CLASS(obj) (G_TYPE_CHECK_CLASS_TYPE ((klass), OGMRIP_TYPE_LAVC_DIALOG))

#define OGMRIP_GCONF_LAVC             "lavc"

#define OGMRIP_GCONF_LAVC_CMP         "cmp"
#define OGMRIP_GCONF_LAVC_PRECMP      "precmp"
#define OGMRIP_GCONF_LAVC_SUBCMP      "subcmp"
#define OGMRIP_GCONF_LAVC_DIA         "dia"
#define OGMRIP_GCONF_LAVC_PREDIA      "predia"
#define OGMRIP_GCONF_LAVC_KEYINT      "keyint"
#define OGMRIP_GCONF_LAVC_BUF_SIZE    "buf_size"
#define OGMRIP_GCONF_LAVC_MIN_RATE    "min_rate"
#define OGMRIP_GCONF_LAVC_MAX_RATE    "max_rate"
#define OGMRIP_GCONF_LAVC_STRICT      "strict"
#define OGMRIP_GCONF_LAVC_DC          "dc"
#define OGMRIP_GCONF_LAVC_MBD         "mbd"
#define OGMRIP_GCONF_LAVC_QNS         "qns"
#define OGMRIP_GCONF_LAVC_VB_STRATEGY "vb_strategy"
#define OGMRIP_GCONF_LAVC_LAST_PRED   "last_pred"
#define OGMRIP_GCONF_LAVC_PREME       "preme"
#define OGMRIP_GCONF_LAVC_VQCOMP      "vqcomp"
#define OGMRIP_GCONF_LAVC_MV0         "mv0"
#define OGMRIP_GCONF_LAVC_V4MV        "v4mv"

#define OGMRIP_DEFAULT_LAVC_CMP         2
#define OGMRIP_DEFAULT_LAVC_PRECMP      2
#define OGMRIP_DEFAULT_LAVC_SUBCMP      2
#define OGMRIP_DEFAULT_LAVC_DIA         2
#define OGMRIP_DEFAULT_LAVC_PREDIA      2
#define OGMRIP_DEFAULT_LAVC_KEYINT      250
#define OGMRIP_DEFAULT_LAVC_BUF_SIZE    0
#define OGMRIP_DEFAULT_LAVC_MIN_RATE    0
#define OGMRIP_DEFAULT_LAVC_MAX_RATE    0
#define OGMRIP_DEFAULT_LAVC_STRICT      2
#define OGMRIP_DEFAULT_LAVC_DC          8
#define OGMRIP_DEFAULT_LAVC_MBD         0
#define OGMRIP_DEFAULT_LAVC_QNS         0
#define OGMRIP_DEFAULT_LAVC_VB_STRATEGY 0
#define OGMRIP_DEFAULT_LAVC_LAST_PRED   0
#define OGMRIP_DEFAULT_LAVC_PREME       1
#define OGMRIP_DEFAULT_LAVC_VQCOMP      0.5
#define OGMRIP_DEFAULT_LAVC_MV0         FALSE
#define OGMRIP_DEFAULT_LAVC_V4MV        TRUE

enum
{
  PROP_0,
  PROP_KEY
};

typedef struct _OGMRipLavcDialog      OGMRipLavcDialog;
typedef struct _OGMRipLavcDialogClass OGMRipLavcDialogClass;
typedef struct _OGMRipLavcOptions     OGMRipLavcOptions;

struct _OGMRipLavcDialog
{
  GtkDialog parent_instance;

  GtkWidget *mv0_check;
  GtkWidget *v4mv_check;

  GtkWidget *mbd_combo;
  GtkWidget *strict_combo;
  GtkWidget *vb_strategy_combo;
  GtkWidget *qns_combo;
  GtkWidget *keyint_spin;
  GtkWidget *last_pred_spin;
  GtkWidget *vqcomp_spin;
  GtkWidget *dc_spin;

  GtkWidget *preme_combo;
  GtkWidget *cmp_spin;
  GtkWidget *precmp_spin;
  GtkWidget *subcmp_spin;
  GtkWidget *dia_spin;
  GtkWidget *predia_spin;

  GtkWidget *buf_size_spin;
  GtkWidget *min_rate_spin;
  GtkWidget *max_rate_spin;
};

struct _OGMRipLavcDialogClass
{
  GtkDialogClass parent_class;
};

struct _OGMRipLavcOptions
{
  gchar *name;
  gchar *key;
  gchar *type;
  gint  value[3];
};

static void ogmrip_lavc_dialog_set_property  (GObject          *gobject,
                                              guint            property_id,
                                              const GValue     *value,
                                              GParamSpec       *pspec);
static void ogmrip_lavc_dialog_set_gconf_key (OGMRipLavcDialog *dialog,
                                              const gchar      *key);

static void
ogmrip_lavc_dialog_connect_spin (GtkWidget *spin, const gchar *dir, const gchar *key, gint def)
{
  GConfValue *value;
  gchar *real_key;

  if (dir)
    real_key = gconf_concat_dir_and_key (dir, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_int (real_key, def);

  ogmrip_preferences_connect_spin (spin, real_key);

  g_free (real_key);
}

static void
ogmrip_lavc_dialog_connect_toggle (GtkWidget *toggle, const gchar *dir, const gchar *key, gboolean def)
{
  GConfValue *value;
  gchar *real_key;

  if (dir)
    real_key = gconf_concat_dir_and_key (dir, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_bool (real_key, def);

  ogmrip_preferences_connect_toggle (toggle, real_key);

  g_free (real_key);
}

static void
ogmrip_lavc_dialog_connect_combo (GtkWidget *combo, const gchar *dir, const gchar *key, gint def)
{
  GConfValue *value;
  gchar *real_key;

  if (dir)
    real_key = gconf_concat_dir_and_key (dir, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_int (real_key, def);

  ogmrip_preferences_connect_combo (combo, real_key);

  g_free (real_key);
}

G_DEFINE_TYPE (OGMRipLavcDialog, ogmrip_lavc_dialog, GTK_TYPE_DIALOG)

static void
ogmrip_lavc_dialog_class_init (OGMRipLavcDialogClass *klass)
{
  GObjectClass *gobject_class;

  gobject_class = (GObjectClass *) klass;
  gobject_class->set_property = ogmrip_lavc_dialog_set_property;

  g_object_class_install_property (gobject_class, PROP_KEY, 
        g_param_spec_string ("key", "GConf key property", "Set the gconf key", 
           NULL, G_PARAM_WRITABLE));
}

static void
ogmrip_lavc_dialog_init (OGMRipLavcDialog *dialog)
{
  GtkWidget *widget;
  GladeXML *xml;

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, OGMRIP_GLADE_ROOT, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return;
  }

  gtk_dialog_add_buttons (GTK_DIALOG (dialog),
      GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
      NULL);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Lavc Options"));
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_PREFERENCES);

  widget = glade_xml_get_widget (xml, OGMRIP_GLADE_ROOT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), widget, TRUE, TRUE, 0);
  gtk_widget_show (widget);

  dialog->mv0_check = glade_xml_get_widget (xml, "mv0-check");
  dialog->v4mv_check = glade_xml_get_widget (xml, "v4mv-check");

  dialog->mbd_combo = glade_xml_get_widget (xml, "mbd-combo");
  dialog->strict_combo = glade_xml_get_widget (xml, "strict-combo");
  dialog->vb_strategy_combo = glade_xml_get_widget (xml, "vb_strategy-combo");
  dialog->qns_combo = glade_xml_get_widget (xml, "qns-combo");
  dialog->keyint_spin = glade_xml_get_widget (xml, "keyint-spin");
  dialog->last_pred_spin = glade_xml_get_widget (xml, "last_pred-spin");
  dialog->vqcomp_spin = glade_xml_get_widget (xml, "vqcomp-spin");
  dialog->dc_spin = glade_xml_get_widget (xml, "dc-spin");

  dialog->preme_combo = glade_xml_get_widget (xml, "preme-combo");
  dialog->cmp_spin = glade_xml_get_widget (xml, "cmp-spin");
  dialog->precmp_spin = glade_xml_get_widget (xml, "precmp-spin");
  dialog->subcmp_spin = glade_xml_get_widget (xml, "subcmp-spin");
  dialog->dia_spin = glade_xml_get_widget (xml, "dia-spin");
  dialog->predia_spin = glade_xml_get_widget (xml, "predia-spin");

  dialog->buf_size_spin = glade_xml_get_widget (xml, "buf_size-spin");
  dialog->min_rate_spin = glade_xml_get_widget (xml, "min_rate-spin");
  dialog->max_rate_spin = glade_xml_get_widget (xml, "max_rate-spin");

  g_object_unref (xml);
}

static void
ogmrip_lavc_dialog_set_property (GObject *gobject, guint property_id, const GValue *value, GParamSpec *pspec)
{
  switch (property_id) 
  {
    case PROP_KEY:
      ogmrip_lavc_dialog_set_gconf_key (OGMRIP_LAVC_DIALOG (gobject), g_value_get_string (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, property_id, pspec);
      break;
  }
}

static void
ogmrip_lavc_dialog_set_gconf_key (OGMRipLavcDialog *dialog, const gchar *key)
{
  ogmrip_lavc_dialog_connect_toggle (dialog->mv0_check, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_MV0, OGMRIP_DEFAULT_LAVC_MV0);
  ogmrip_lavc_dialog_connect_toggle (dialog->v4mv_check, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_V4MV, OGMRIP_DEFAULT_LAVC_V4MV);

  ogmrip_lavc_dialog_connect_combo (dialog->mbd_combo, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_MBD, OGMRIP_DEFAULT_LAVC_MBD);
  ogmrip_lavc_dialog_connect_combo (dialog->strict_combo, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_STRICT, OGMRIP_DEFAULT_LAVC_STRICT);
  ogmrip_lavc_dialog_connect_combo (dialog->vb_strategy_combo, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_VB_STRATEGY, OGMRIP_DEFAULT_LAVC_VB_STRATEGY);
  ogmrip_lavc_dialog_connect_combo (dialog->qns_combo, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_QNS, OGMRIP_DEFAULT_LAVC_QNS);
  ogmrip_lavc_dialog_connect_spin (dialog->keyint_spin, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_KEYINT, OGMRIP_DEFAULT_LAVC_KEYINT);
  ogmrip_lavc_dialog_connect_spin (dialog->dc_spin, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_DC, OGMRIP_DEFAULT_LAVC_DC);
  ogmrip_lavc_dialog_connect_spin (dialog->last_pred_spin, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_LAST_PRED, OGMRIP_DEFAULT_LAVC_LAST_PRED);
  ogmrip_lavc_dialog_connect_spin (dialog->vqcomp_spin, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_VQCOMP, OGMRIP_DEFAULT_LAVC_VQCOMP);

  ogmrip_lavc_dialog_connect_combo (dialog->preme_combo, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_PREME, OGMRIP_DEFAULT_LAVC_PREME);
  ogmrip_lavc_dialog_connect_spin (dialog->cmp_spin, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_CMP, OGMRIP_DEFAULT_LAVC_CMP);
  ogmrip_lavc_dialog_connect_spin (dialog->precmp_spin, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_PRECMP, OGMRIP_DEFAULT_LAVC_PRECMP);
  ogmrip_lavc_dialog_connect_spin (dialog->subcmp_spin, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_SUBCMP, OGMRIP_DEFAULT_LAVC_SUBCMP);
  ogmrip_lavc_dialog_connect_spin (dialog->dia_spin, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_DIA, OGMRIP_DEFAULT_LAVC_DIA);
  ogmrip_lavc_dialog_connect_spin (dialog->predia_spin, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_PREDIA, OGMRIP_DEFAULT_LAVC_PREDIA);

  ogmrip_lavc_dialog_connect_spin (dialog->buf_size_spin, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_BUF_SIZE, OGMRIP_DEFAULT_LAVC_BUF_SIZE);
  ogmrip_lavc_dialog_connect_spin (dialog->min_rate_spin, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_MIN_RATE, OGMRIP_DEFAULT_LAVC_MIN_RATE);
  ogmrip_lavc_dialog_connect_spin (dialog->max_rate_spin, key,
      OGMRIP_GCONF_LAVC "/" OGMRIP_GCONF_LAVC_MAX_RATE, OGMRIP_DEFAULT_LAVC_MAX_RATE);
}

static void
ogmrip_lavc_set_options (OGMRipVideoCodec *codec)
{
#define GET_BOOL(key,def)   key, ogmrip_preferences_get_bool   (OGMRIP_GCONF_LAVC "/" key, def)
#define GET_INT(key,def)    key, ogmrip_preferences_get_int    (OGMRIP_GCONF_LAVC "/" key, def)
#define GET_DOUBLE(key,def) key, ogmrip_preferences_get_double (OGMRIP_GCONF_LAVC "/" key, def)

  if (ogmrip_preferences_get_int ("video/preset", 0) == OGMRIP_VIDEO_PRESET_USER)
    g_object_set (codec,
        GET_INT    (OGMRIP_GCONF_LAVC_CMP,         OGMRIP_DEFAULT_LAVC_CMP),
        GET_INT    (OGMRIP_GCONF_LAVC_PRECMP,      OGMRIP_DEFAULT_LAVC_PRECMP),
        GET_INT    (OGMRIP_GCONF_LAVC_SUBCMP,      OGMRIP_DEFAULT_LAVC_SUBCMP),
        GET_INT    (OGMRIP_GCONF_LAVC_DIA,         OGMRIP_DEFAULT_LAVC_DIA),
        GET_INT    (OGMRIP_GCONF_LAVC_PREDIA,      OGMRIP_DEFAULT_LAVC_PREDIA),
        GET_INT    (OGMRIP_GCONF_LAVC_KEYINT,      OGMRIP_DEFAULT_LAVC_KEYINT),
        GET_INT    (OGMRIP_GCONF_LAVC_BUF_SIZE,    OGMRIP_DEFAULT_LAVC_BUF_SIZE),
        GET_INT    (OGMRIP_GCONF_LAVC_MIN_RATE,    OGMRIP_DEFAULT_LAVC_MIN_RATE),
        GET_INT    (OGMRIP_GCONF_LAVC_MAX_RATE,    OGMRIP_DEFAULT_LAVC_MAX_RATE),
        GET_INT    (OGMRIP_GCONF_LAVC_STRICT,      OGMRIP_DEFAULT_LAVC_STRICT),
        GET_INT    (OGMRIP_GCONF_LAVC_DC,          OGMRIP_DEFAULT_LAVC_DC),
        GET_INT    (OGMRIP_GCONF_LAVC_MBD,         OGMRIP_DEFAULT_LAVC_MBD),
        GET_INT    (OGMRIP_GCONF_LAVC_QNS,         OGMRIP_DEFAULT_LAVC_QNS),
        GET_INT    (OGMRIP_GCONF_LAVC_VB_STRATEGY, OGMRIP_DEFAULT_LAVC_VB_STRATEGY),
        GET_INT    (OGMRIP_GCONF_LAVC_LAST_PRED,   OGMRIP_DEFAULT_LAVC_LAST_PRED),
        GET_INT    (OGMRIP_GCONF_LAVC_PREME,       OGMRIP_DEFAULT_LAVC_PREME),
        GET_DOUBLE (OGMRIP_GCONF_LAVC_VQCOMP,      OGMRIP_DEFAULT_LAVC_VQCOMP),
        GET_BOOL   (OGMRIP_GCONF_LAVC_MV0,         OGMRIP_DEFAULT_LAVC_MV0),
        GET_BOOL   (OGMRIP_GCONF_LAVC_V4MV,        OGMRIP_DEFAULT_LAVC_V4MV),
        NULL);

#undef GET_BOOL
#undef GET_INT
#undef GET_DOUBLE
}

static OGMRipVideoOptionsPlugin lavc_options_plugin =
{
  NULL,
  G_TYPE_NONE,
  G_TYPE_NONE,
  NULL
};

OGMRipVideoOptionsPlugin *
ogmrip_init_options_plugin (void)
{
  // lavc_options_plugin.type = ogmrip_plugin_get_video_codec_by_name ("lavc-mpeg4");
  lavc_options_plugin.type = OGMRIP_TYPE_LAVC;
  if (lavc_options_plugin.type == G_TYPE_NONE)
    return NULL;

  lavc_options_plugin.dialog = OGMRIP_TYPE_LAVC_DIALOG;

  lavc_options_plugin.set_options = ogmrip_lavc_set_options;

  return &lavc_options_plugin;
}

