/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*-
 *
 * nautilus-burn-drive.c: easy to use cd burner software
 *
 * Copyright (C) 2002-2004 Red Hat, Inc.
 * Copyright (C) 2005 William Jon McCann <mccann@jhu.edu>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * Authors: Alexander Larsson <alexl@redhat.com>
 *          Bastien Nocera <hadess@hadess.net>
 *          William Jon McCann <mccann@jhu.edu>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <math.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>

#include <sys/ioctl.h>

#include <glib.h>
#include <glib/gstdio.h>

#include "nautilus-burn-drive.h"
#include "nautilus-burn-drive-common.h"

#ifdef HAVE_SYS_CDIO_H
#include <sys/cdio.h>
#endif

#ifdef __linux__
#include "nautilus-burn-drive-linux.h"
#endif

#ifdef __FreeBSD__
#include "nautilus-burn-drive-freebsd.h"
#endif

/* For dvd_plus_rw_utils.cpp */
int get_mmc_profile      (int        fd);
int get_disc_status      (int        fd, 
                          int        *empty, 
                          int        *is_rewritable, 
                          int        *is_blank);
int get_dvd_r_rw_profile (const char *name);
int get_read_write_speed (int        fd, 
                          int        *read_speed, 
                          int        *write_speed);

#ifdef __FreeBSD__

gpointer
open_ioctl_handle (const char *device)
{
  struct cam_device *cam;

  cam = cam_open_device (device, O_RDWR);

  return (cam ? (gpointer) cam : INVALID_HANDLE);
}

void
close_ioctl_handle (gpointer handle)
{
  cam_close_device ((struct cam_device *) handle);
}

#else

gpointer
open_ioctl_handle (const char *device)
{
  int fd;

  if ((fd = g_open (device, O_RDWR | O_EXCL | O_NONBLOCK, 0)) < 0
      && (fd = g_open (device, O_RDONLY | O_EXCL | O_NONBLOCK, 0)) < 0)
    return INVALID_HANDLE;

  return GINT_TO_POINTER (fd);
}

void
close_ioctl_handle (gpointer handle)
{
  close (GPOINTER_TO_INT (handle));
}

#endif

int
get_device_max_write_speed (char *device)
{
  gpointer ioctl_handle;
  int fd, max_speed, read_speed, write_speed;

  max_speed = -1;

  ioctl_handle = open_ioctl_handle (device);
  if (ioctl_handle == INVALID_HANDLE)
    return -1;

  fd = get_ioctl_handle_fd (ioctl_handle);
  get_read_write_speed (fd, &read_speed, &write_speed);
  close_ioctl_handle (ioctl_handle);
  max_speed = (int) floor (write_speed) / CD_ROM_SPEED;

  return max_speed;
}

static int
get_device_max_read_speed (char *device)
{
  gpointer ioctl_handle;
  int fd, max_speed, read_speed, write_speed;

  max_speed = -1;

  ioctl_handle = open_ioctl_handle (device);
  if (ioctl_handle == INVALID_HANDLE)
    return -1;

  fd = get_ioctl_handle_fd (ioctl_handle);
  get_read_write_speed (fd, &read_speed, &write_speed);
  close_ioctl_handle (ioctl_handle);
  max_speed = (int) floor (read_speed) / CD_ROM_SPEED;

  return max_speed;
}

void
nautilus_burn_drive_add_dvd_plus (NautilusBurnDrive * drive)
{
  int caps;

  caps = get_dvd_r_rw_profile (drive->device);

  if (caps == -1)
    return;

  if (caps == 2)
  {
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_PLUS_RW_RECORDER;
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_PLUS_R_RECORDER;
  }
  else if (caps == 0)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_PLUS_R_RECORDER;
  else if (caps == 1)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_PLUS_RW_RECORDER;
}

void
nautilus_burn_drive_add_whitelist (NautilusBurnDrive *drive)
{
  NautilusBurnWhiteDrive recorder_whitelist[] =
  {
    { "IOMEGA - CDRW9602EXT-B", TRUE, TRUE,  FALSE, FALSE },
    { "SONY - CD-R CDU948S",    TRUE, FALSE, FALSE, FALSE },
  };

  guint i;

  for (i = 0; i < G_N_ELEMENTS (recorder_whitelist); i++)
  {
    if (!strcmp (drive->display_name, recorder_whitelist[i].name))
    {
      if (recorder_whitelist[i].can_write_cdr)
        drive->type |= NAUTILUS_BURN_DRIVE_TYPE_CD_RECORDER;
      if (recorder_whitelist[i].can_write_cdrw)
        drive->type |= NAUTILUS_BURN_DRIVE_TYPE_CDRW_RECORDER;
      if (recorder_whitelist[i].can_write_dvdr)
        drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_RW_RECORDER;
      if (recorder_whitelist[i].can_write_dvdram)
        drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_RAM_RECORDER;
    }
  }
}

static char *
cdrecord_get_stdout_for_id (char *id)
{
  int max_speed, i;
  const char *argv[20];         /* Shouldn't need more than 20 arguments */
  char *dev_str, *stdout_data;

  max_speed = -1;

  i = 0;
  argv[i++] = "cdrecord";
  argv[i++] = "-prcap";
  dev_str = g_strdup_printf ("dev=%s", id);
  argv[i++] = dev_str;
  argv[i++] = NULL;

  if (g_spawn_sync (NULL, (char **) argv, NULL, G_SPAWN_SEARCH_PATH | G_SPAWN_STDERR_TO_DEV_NULL, 
        NULL, NULL, &stdout_data, NULL, NULL, NULL))
  {
    g_free (dev_str);
    return stdout_data;
  }

  g_free (dev_str);
  return NULL;
}



void
nautilus_burn_drive_get_cd_properties (NautilusBurnDrive *drive)
{
  char *stdout_data, *drive_cap;

  drive->max_speed_read = -1;
  drive->max_speed_write = -1;
  drive->type = 0;

  drive->max_speed_read = get_device_max_read_speed (drive->device);
  drive->max_speed_write = get_device_max_write_speed (drive->device);

  stdout_data = cdrecord_get_stdout_for_id (drive->cdrecord_id);
  if (stdout_data == NULL)
    return;

  drive_cap = strstr (stdout_data, "Does write DVD-RAM media");
  if (drive_cap != NULL)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_RAM_RECORDER;
  drive_cap = strstr (stdout_data, "Does read DVD-R media");
  if (drive_cap != NULL)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_RW_RECORDER;
  drive_cap = strstr (stdout_data, "Does read DVD-ROM media");
  if (drive_cap != NULL)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_DRIVE;
  drive_cap = strstr (stdout_data, "Does write CD-RW media");
  if (drive_cap != NULL)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_CDRW_RECORDER;
  drive_cap = strstr (stdout_data, "Does write CD-R media");
  if (drive_cap != NULL)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_CD_RECORDER;
  drive_cap = strstr (stdout_data, "Does read CD-R media");
  if (drive_cap != NULL)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_CD_DRIVE;
  g_free (stdout_data);
}

static char *
cdrecord_scan_get_stdout (void)
{
  int max_speed, i;
  const char *argv[20];         /* Shouldn't need more than 20 arguments */
  char *stdout_data;

  max_speed = -1;

  i = 0;
  argv[i++] = "cdrecord";
  argv[i++] = "-scanbus";
  argv[i++] = NULL;

  if (g_spawn_sync (NULL, (char **) argv, NULL, G_SPAWN_SEARCH_PATH | G_SPAWN_STDERR_TO_DEV_NULL, 
        NULL, NULL, &stdout_data, NULL, NULL, NULL))
    return stdout_data;

  return NULL;
}

#define DEFAULT_SPEED 2

GList *
cdrecord_scan (gboolean recorder_only)
{
  GList *drives_list;
  NautilusBurnDrive *drive;
  char *stdout_data, **lines, vendor[9], model[17];
  int i, bus, id, lun, index;

  drives_list = NULL;

  stdout_data = cdrecord_scan_get_stdout ();
  if (stdout_data == NULL)
    return drives_list;

  lines = g_strsplit (stdout_data, "\n", 0);
  g_free (stdout_data);

  for (i = 0; lines[i] != NULL; i++)
  {
    if (sscanf (lines[i], "\t%d,%d,%d\t  %d) '%8c' '%16c'", &bus, &id, &lun, &index, vendor, model) != 6)
      continue;

    vendor[8] = '\0';
    model[16] = '\0';

    drive = nautilus_burn_drive_new ();
    drive->display_name = g_strdup_printf ("%s - %s", g_strstrip (vendor), g_strstrip (model));
    drive->cdrecord_id = g_strdup_printf ("%d,%d,%d", bus, id, lun);
    /*
     * FIXME we don't have any way to guess the real device
     * from the info we get from CDRecord 
     */
    drive->device = g_strdup_printf ("/dev/pg%d", index);
    nautilus_burn_drive_get_cd_properties (drive);
    nautilus_burn_drive_add_whitelist (drive);
    if (drive->type & NAUTILUS_BURN_DRIVE_TYPE_CD_RECORDER      || 
        drive->type & NAUTILUS_BURN_DRIVE_TYPE_CDRW_RECORDER    || 
        drive->type & NAUTILUS_BURN_DRIVE_TYPE_DVD_RAM_RECORDER || 
        drive->type & NAUTILUS_BURN_DRIVE_TYPE_DVD_RW_RECORDER  || !recorder_only)
    {
      if (drive->max_speed_read == -1)
        drive->max_speed_read = DEFAULT_SPEED;
      if (drive->max_speed_write == -1)
        drive->max_speed_write = DEFAULT_SPEED;

      if (drive->type & NAUTILUS_BURN_DRIVE_TYPE_DVD_DRIVE)
        nautilus_burn_drive_add_dvd_plus (drive);

      drives_list = g_list_append (drives_list, drive);
    }
    else
      g_object_unref (drive);
  }

  g_strfreev (lines);

  return drives_list;
}

static gboolean
poll_tray_opened (NautilusBurnDrive *drive)
{
  gboolean is_open;

  is_open = nautilus_burn_drive_door_is_open (drive);

  /* check for no change */
  if (is_open == drive->monitor_tray_open)
    return TRUE;

  if (is_open) 
  {
    /* see if we lost media */
    if (drive->monitor_media_type != NAUTILUS_BURN_MEDIA_TYPE_ERROR) 
    {
      drive->monitor_media_type = NAUTILUS_BURN_MEDIA_TYPE_ERROR;
      g_signal_emit_by_name (drive, "media-removed");
    }
  } 
  else 
  {
    NautilusBurnDriveType type;

    /* see if there is now media in the drive */
    type = nautilus_burn_drive_get_media_type (drive);

    if (type != NAUTILUS_BURN_MEDIA_TYPE_ERROR) 
    {
      drive->monitor_media_type = type;
      g_signal_emit_by_name (drive, "media-added");
    }
  }

  drive->monitor_tray_open = is_open;

  return TRUE;
}

void
nautilus_burn_drive_set_monitor (NautilusBurnDrive *drive, gboolean enabled)
{
  if (drive->poll_id > 0)
    g_source_remove (drive->poll_id);

  if (enabled) 
  {
    drive->monitor_media_type = NAUTILUS_BURN_MEDIA_TYPE_ERROR;
    drive->monitor_tray_open = FALSE;
    drive->poll_id = g_timeout_add (2000, (GSourceFunc)poll_tray_opened, drive);
  } 
  else
  {
    drive->monitor_media_type = NAUTILUS_BURN_MEDIA_TYPE_ERROR;
    drive->monitor_tray_open = FALSE;
    drive->poll_id = 0;
  }
}

gboolean
drive_door_is_open (int fd)
{
  if (fd < 0)
    return FALSE;

#ifdef __linux__
  {
    int status;

    status = ioctl (fd, CDROM_DRIVE_STATUS, CDSL_CURRENT);
    if (status < 0)
      return FALSE;

    return status == CDS_TRAY_OPEN;
  }
#else
  return FALSE;
#endif
}

/* adapted from gnome-vfs */
static int
get_disc_type (const char *dev_path)
{
  int fd = 0;
  int type;
#if defined (HAVE_SYS_MNTTAB_H) /* Solaris */
  GString *new_dev_path;
  struct cdrom_tocentry entry;
  struct cdrom_tochdr header;

  /*
   * For ioctl call to work dev_path has to be /vol/dev/rdsk.
   * There has to be a nicer way to do this.
   */
  new_dev_path = g_string_new (dev_path);
  new_dev_path = g_string_insert_c (new_dev_path, 9, 'r');
  fd = open (new_dev_path->str, O_RDONLY | O_NONBLOCK);
  g_string_free (new_dev_path, TRUE);

  if (fd < 0)
  {
    return -1;
  }

  if (ioctl (fd, CDROMREADTOCHDR, &header) < 0)
  {
    return -1;
  }

  type = CDS_DATA_1;

  for (entry.cdte_track = 1; entry.cdte_track <= header.cdth_trk1; entry.cdte_track++)
  {
    entry.cdte_format = CDROM_LBA;
    if (ioctl (fd, CDROMREADTOCENTRY, &entry) == 0)
    {
      if (!(entry.cdte_ctrl & CDROM_DATA_TRACK))
      {
        type = CDS_AUDIO;
        break;
      }
    }
  }

  return type;
#elif defined(HAVE_SYS_MNTCTL_H)
  return CDS_NO_INFO;
#elif defined(__FreeBSD__) || defined(__FreeBSD_kernel__) || defined(__OpenBSD__) || defined(__NetBSD__) || defined(__DragonFly__)
  struct ioc_toc_header header;
#if defined(__FreeBSD__) || defined(__FreeBSD_kernel__) || defined(__DragonFly__)
  struct ioc_read_toc_single_entry entry;
#else
  struct ioc_read_toc_entry entries;
  struct cd_toc_entry entry;
  int i;
#endif

#ifndef CDROM_DATA_TRACK
#define CDROM_DATA_TRACK 4
#endif

  fd = open (dev_path, O_RDONLY | O_NONBLOCK);
  if (fd < 0)
  {
    return -1;
  }

  if (ioctl (fd, CDIOREADTOCHEADER, &header) < 0)
  {
    close (fd);
    return -1;
  }

  type = CDS_DATA_1;
#if defined(__FreeBSD__) || defined(__FreeBSD_kernel__) || defined(__DragonFly__)
  for (entry.track = header.starting_track; entry.track <= header.ending_track; entry.track++)
  {
    entry.address_format = CD_LBA_FORMAT;
    if (ioctl (fd, CDIOREADTOCENTRY, &entry) == 0)
    {
      if (!(entry.entry.control & CDROM_DATA_TRACK))
      {
        type = CDS_AUDIO;
        break;
      }
    }
  }

#else /* defined(__FreeBSD__) || defined(__FreeBSD_kernel__) || defined(__DragonFly__) */
  entries.data_len = sizeof (entry);
  entries.data = &entry;
  for (i = header.starting_track; i <= header.ending_track; i++)
  {
    entries.starting_track = i;
    entries.address_format = CD_LBA_FORMAT;
    if (ioctl (fd, CDIOREADTOCENTRYS, &entries) == 0)
    {
      if (!(entry.control & CDROM_DATA_TRACK))
      {
        type = CDS_AUDIO;
        break;
      }
    }
  }

#endif /* defined(__FreeBSD__) || defined(__FreeBSD_kernel__) || defined(__DragonFly__) */
  close (fd);
  return type;
#elif defined(__linux__)
  fd = open (dev_path, O_RDONLY | O_NONBLOCK);
  if (fd < 0)
  {
    return -1;
  }
  if (ioctl (fd, CDROM_DRIVE_STATUS, CDSL_CURRENT) != CDS_DISC_OK)
  {
    close (fd);
    fd = -1;
    return -1;
  }

  type = ioctl (fd, CDROM_DISC_STATUS, CDSL_CURRENT);
  close (fd);

  return type;
#else /* defined(__linux__) */
  return CDS_NO_INFO;
#endif /* defined(__linux__) */
}

static void
disc_type_get_info (int type, gboolean *is_blank, gboolean *has_data, gboolean *has_audio)
{
  gboolean _has_audio = FALSE;
  gboolean _has_data = FALSE;
  gboolean _is_blank = FALSE;

  switch (type)
  {
    case CDS_AUDIO:              /* audio CD */
      _has_audio = TRUE;
      break;
    case CDS_MIXED:         /* mixed mode CD */
      _has_audio = TRUE;
      _has_data = TRUE;
      break;
    case CDS_DATA_1:              /* data CD */
    case CDS_DATA_2:
    case CDS_XA_2_1:
    case CDS_XA_2_2:
      _has_data = TRUE;
      break;
    case CDS_NO_INFO: /* blank or invalid CD */
      _is_blank = TRUE;
      break;
    default:        /* should never see this */
      break;
  }

  if (is_blank)
    *is_blank = _is_blank;
  if (has_data)
    *has_data = _has_data;
  if (has_audio)
    *has_audio = _has_audio;
}

NautilusBurnMediaType
nautilus_burn_drive_get_media_type_from_path_full (const char *device)
{
  gpointer ioctl_handle;
  int mmc_profile;
  int disc_type;
  int fd;

  g_return_val_if_fail (device != NULL, NAUTILUS_BURN_MEDIA_TYPE_ERROR);

  ioctl_handle = open_ioctl_handle (device);
  if (ioctl_handle == INVALID_HANDLE)
  {
    if (errno == EBUSY)
      return NAUTILUS_BURN_MEDIA_TYPE_BUSY;

    g_warning ("Could not open %s: %s", device, g_strerror (errno));

    return NAUTILUS_BURN_MEDIA_TYPE_ERROR;
  }

  fd = get_ioctl_handle_fd (ioctl_handle);
  mmc_profile = get_mmc_profile (fd);

  /*
   * Couldn't get the data about the media 
   */
  if (mmc_profile < 0)
  {
    gboolean door_is_open;

    door_is_open = drive_door_is_open (fd);

    if (door_is_open != FALSE)
    {
      close_ioctl_handle (ioctl_handle);
      return NAUTILUS_BURN_MEDIA_TYPE_ERROR;
    }
    else
    {
      int blank, rewrite, empty;

      if (get_disc_status (fd, &empty, &rewrite, &blank) == 0)
      {
        close_ioctl_handle (ioctl_handle);

        disc_type = get_disc_type (device);
        disc_type_get_info (disc_type, NULL, NULL, NULL);

        if (empty)
          return NAUTILUS_BURN_MEDIA_TYPE_ERROR;
        else
          return NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN;
      }

      close_ioctl_handle (ioctl_handle);

      return NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN;
    }
  }

  close_ioctl_handle (ioctl_handle);

  disc_type = get_disc_type (device);
  disc_type_get_info (disc_type, NULL, NULL, NULL);

  switch (mmc_profile & 0xFFFF)
  {
    case -1:
      g_assert_not_reached ();
    case 0:                    /* No Media */
      return NAUTILUS_BURN_MEDIA_TYPE_ERROR;
    case 0x8:                  /* Commercial CDs and Audio CD  */
      return NAUTILUS_BURN_MEDIA_TYPE_CD;
    case 0x9:                  /* CD-R                         */
      return NAUTILUS_BURN_MEDIA_TYPE_CDR;
    case 0xa:                  /* CD-RW                        */
      return NAUTILUS_BURN_MEDIA_TYPE_CDRW;
    case 0x10:                 /* Commercial DVDs              */
      return NAUTILUS_BURN_MEDIA_TYPE_DVD;
    case 0x11:                 /* DVD-R                        */
      return NAUTILUS_BURN_MEDIA_TYPE_DVDR;
    case 0x13:                 /* DVD-RW Restricted Overwrite  */
    case 0x14:                 /* DVD-RW Sequential            */
      return NAUTILUS_BURN_MEDIA_TYPE_DVDRW;
    case 0x1B:                 /* DVD+R                        */
      return NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_R;
    case 0x2B:                 /* DVD+R Double Layer           */
      return NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_R_DL;
    case 0x1A:                 /* DVD+RW                       */
      return NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW;
    case 0x12:                 /* DVD-RAM                      */
      return NAUTILUS_BURN_MEDIA_TYPE_DVD_RAM;
    default:
      return NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN;
  }
}

