/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2008 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n.h>
#include <gtk/gtkdialog.h>
#include <glade/glade.h>

#include "ogmrip-plugin.h"
#include "ogmrip-options-plugin.h"
#include "ogmrip-preferences.h"
#include "ogmrip-helper.h"
#include "ogmrip-x264.h"

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-x264.glade"
#define OGMRIP_GLADE_ROOT "root"

#define OGMRIP_TYPE_X264_DIALOG          (ogmrip_x264_dialog_get_type ())
#define OGMRIP_X264_DIALOG(obj)          (G_TYPE_CHECK_INSTANCE_CAST ((obj), OGMRIP_TYPE_X264_DIALOG, OGMRipX264Dialog))
#define OGMRIP_X264_DIALOG_CLASS(klass)  (G_TYPE_CHECK_CLASS_CAST ((klass), OGMRIP_TYPE_X264_DIALOG, OGMRipX264DialogClass))
#define OGMRIP_IS_X264_DIALOG(obj)       (G_TYPE_CHECK_INSTANCE_TYPE ((obj), OGMRIP_TYPE_X264_DIALOG))
#define OGMRIP_IS_X264_DIALOG_CLASS(obj) (G_TYPE_CHECK_CLASS_TYPE ((klass), OGMRIP_TYPE_X264_DIALOG))

#define OGMRIP_GCONF_X264               "x264"

#define OGMRIP_GCONF_X264_8X8DCT        "dct8x8"
#define OGMRIP_GCONF_X264_BFRAMES       "bframes"
#define OGMRIP_GCONF_X264_B_PYRAMID     "b_pyramid"
#define OGMRIP_GCONF_X264_BRDO          "brdo"
#define OGMRIP_GCONF_X264_CABAC         "cabac"
#define OGMRIP_GCONF_X264_FRAMEREF      "frameref"
#define OGMRIP_GCONF_X264_LEVEL_IDC     "level_idc"
#define OGMRIP_GCONF_X264_ME            "me"
#define OGMRIP_GCONF_X264_MIXED_REFS    "mixed_refs"
#define OGMRIP_GCONF_X264_SUBQ          "subq"
#define OGMRIP_GCONF_X264_VBV_BUFSIZE   "vbv_bufsize"
#define OGMRIP_GCONF_X264_VBV_MAXRATE   "vbv_maxrate"
#define OGMRIP_GCONF_X264_WEIGHT_B      "weight_b"
#define OGMRIP_GCONF_X264_GLOBAL_HEADER "global_header"

#define OGMRIP_DEFAULT_X264_8X8DCT        TRUE
#define OGMRIP_DEFAULT_X264_BFRAMES       3
#define OGMRIP_DEFAULT_X264_B_PYRAMID     TRUE
#define OGMRIP_DEFAULT_X264_BRDO          TRUE
#define OGMRIP_DEFAULT_X264_CABAC         TRUE
#define OGMRIP_DEFAULT_X264_FRAMEREF      5
#define OGMRIP_DEFAULT_X264_LEVEL_IDC     0
#define OGMRIP_DEFAULT_X264_ME            3
#define OGMRIP_DEFAULT_X264_MIXED_REFS    TRUE
#define OGMRIP_DEFAULT_X264_SUBQ          6
#define OGMRIP_DEFAULT_X264_VBV_BUFSIZE   0
#define OGMRIP_DEFAULT_X264_VBV_MAXRATE   0
#define OGMRIP_DEFAULT_X264_WEIGHT_B      TRUE
#define OGMRIP_DEFAULT_X264_GLOBAL_HEADER FALSE

enum
{
  PROP_0,
  PROP_KEY
};

typedef struct _OGMRipX264Dialog      OGMRipX264Dialog;
typedef struct _OGMRipX264DialogClass OGMRipX264DialogClass;
typedef struct _OGMRipX264Options     OGMRipX264Options;

struct _OGMRipX264Dialog
{
  GtkDialog parent_instance;

  GtkWidget *b_pyramid_check;
  GtkWidget *brdo_check;
  GtkWidget *cabac_check;
  GtkWidget *mixed_refs_check;
  GtkWidget *weight_b_check;
  GtkWidget *x88dct_check;
  GtkWidget *global_header_check;

  GtkWidget *bframes_spin;
  GtkWidget *frameref_spin;
  GtkWidget *level_idc_spin;
  GtkWidget *subq_spin;
  GtkWidget *vbv_bufsize_spin;
  GtkWidget *vbv_maxrate_spin;

  GtkWidget *me_combo;
};

struct _OGMRipX264DialogClass
{
  GtkDialogClass parent_class;
};

struct _OGMRipX264Options
{
  gchar *name;
  gchar *key;
  gchar *type;
  gint  value[3];
};

static void ogmrip_x264_dialog_set_property  (GObject          *gobject,
                                              guint            property_id,
                                              const GValue     *value,
                                              GParamSpec       *pspec);
static void ogmrip_x264_dialog_set_gconf_key (OGMRipX264Dialog *dialog,
                                              const gchar      *key);

static void
ogmrip_x264_dialog_connect_spin (GtkWidget *spin, const gchar *dir, const gchar *key, gint def)
{
  GConfValue *value;
  gchar *real_key;

  if (dir)
    real_key = gconf_concat_dir_and_key (dir, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_int (real_key, def);

  ogmrip_preferences_connect_spin (spin, real_key);

  g_free (real_key);
}

static void
ogmrip_x264_dialog_connect_toggle (GtkWidget *toggle, const gchar *dir, const gchar *key, gboolean def)
{
  GConfValue *value;
  gchar *real_key;

  if (dir)
    real_key = gconf_concat_dir_and_key (dir, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_bool (real_key, def);

  ogmrip_preferences_connect_toggle (toggle, real_key);

  g_free (real_key);
}

static void
ogmrip_x264_dialog_connect_combo (GtkWidget *combo, const gchar *dir, const gchar *key, gint def)
{
  GConfValue *value;
  gchar *real_key;

  if (dir)
    real_key = gconf_concat_dir_and_key (dir, key);
  else
    real_key = g_strdup (key);

  value = ogmrip_preferences_get (real_key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_int (real_key, def);

  ogmrip_preferences_connect_combo (combo, real_key);

  g_free (real_key);
}

static void
ogmrip_x264_dialog_bframes_changed (OGMRipX264Dialog *dialog)
{
  gint bframes;

  bframes = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->bframes_spin));

  gtk_widget_set_sensitive (dialog->b_pyramid_check, bframes > 1);
  gtk_widget_set_sensitive (dialog->weight_b_check, bframes > 1);

  if (bframes <= 1)
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->b_pyramid_check), FALSE);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->weight_b_check), FALSE);
  }
}

static void
ogmrip_x264_dialog_subq_changed (OGMRipX264Dialog *dialog)
{
  gint subq;

  subq = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->subq_spin));

  gtk_widget_set_sensitive (dialog->brdo_check, subq > 5);
  if (subq <= 5)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->brdo_check), FALSE);
}

static void
ogmrip_x264_dialog_frameref_changed (OGMRipX264Dialog *dialog)
{
  gint frameref;

  frameref = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->frameref_spin));

  gtk_widget_set_sensitive (dialog->mixed_refs_check, frameref > 1);
  if (frameref <= 1)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->mixed_refs_check), FALSE);
}

static void
ogmrip_x264_dialog_vbv_maxrate_changed (OGMRipX264Dialog *dialog)
{
  gtk_widget_set_sensitive (dialog->vbv_bufsize_spin,
      gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->vbv_maxrate_spin)) > 0);
}

G_DEFINE_TYPE (OGMRipX264Dialog, ogmrip_x264_dialog, GTK_TYPE_DIALOG)

static void
ogmrip_x264_dialog_class_init (OGMRipX264DialogClass *klass)
{
  GObjectClass *gobject_class;

  gobject_class = (GObjectClass *) klass;
  gobject_class->set_property = ogmrip_x264_dialog_set_property;

  g_object_class_install_property (gobject_class, PROP_KEY, 
        g_param_spec_string ("key", "GConf key property", "Set the gconf key", 
           NULL, G_PARAM_WRITABLE));
}

static void
ogmrip_x264_dialog_init (OGMRipX264Dialog *dialog)
{
  GtkWidget *widget;
  GladeXML *xml;

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, OGMRIP_GLADE_ROOT, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return;
  }

  gtk_dialog_add_buttons (GTK_DIALOG (dialog),
      GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
      NULL);
  gtk_window_set_title (GTK_WINDOW (dialog), _("X264 Options"));
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_PREFERENCES);

  widget = glade_xml_get_widget (xml, OGMRIP_GLADE_ROOT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), widget, TRUE, TRUE, 0);
  gtk_widget_show (widget);

  dialog->bframes_spin = glade_xml_get_widget (xml, "bframes-spin");
  g_signal_connect_swapped (dialog->bframes_spin, "value-changed",
      G_CALLBACK (ogmrip_x264_dialog_bframes_changed), dialog);

  dialog->cabac_check = glade_xml_get_widget (xml, "cabac-check");

  dialog->subq_spin = glade_xml_get_widget (xml, "subq-spin");
  g_signal_connect_swapped (dialog->subq_spin, "value-changed",
      G_CALLBACK (ogmrip_x264_dialog_subq_changed), dialog);
  
  dialog->global_header_check = glade_xml_get_widget (xml, "global_header-check");
  dialog->b_pyramid_check = glade_xml_get_widget (xml, "b_pyramid-check");
  dialog->weight_b_check = glade_xml_get_widget (xml, "weight_b-check");
  
  dialog->frameref_spin = glade_xml_get_widget (xml, "frameref-spin");
  g_signal_connect_swapped (dialog->frameref_spin, "value-changed",
      G_CALLBACK (ogmrip_x264_dialog_frameref_changed), dialog);
  
  dialog->me_combo = glade_xml_get_widget (xml, "me-combo");
  dialog->brdo_check = glade_xml_get_widget (xml, "brdo-check");
  dialog->x88dct_check = glade_xml_get_widget (xml, "dct8x8-check");
  dialog->mixed_refs_check = glade_xml_get_widget (xml, "mixed_refs-check");
  
  dialog->vbv_maxrate_spin = glade_xml_get_widget (xml, "vbv_maxrate-spin");
  g_signal_connect_swapped (dialog->vbv_maxrate_spin, "value-changed",
      G_CALLBACK (ogmrip_x264_dialog_vbv_maxrate_changed), dialog);
  
  dialog->vbv_bufsize_spin = glade_xml_get_widget (xml, "vbv_bufsize-spin");
  dialog->level_idc_spin = glade_xml_get_widget (xml, "level_idc-spin");

  g_object_unref (xml);
}

static void
ogmrip_x264_dialog_set_property (GObject *gobject, guint property_id, const GValue *value, GParamSpec *pspec)
{
  switch (property_id) 
  {
    case PROP_KEY:
      ogmrip_x264_dialog_set_gconf_key (OGMRIP_X264_DIALOG (gobject), g_value_get_string (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, property_id, pspec);
      break;
  }
}

static void
ogmrip_x264_dialog_set_gconf_key (OGMRipX264Dialog *dialog, const gchar *key)
{
  ogmrip_x264_dialog_connect_spin (dialog->bframes_spin, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_BFRAMES, OGMRIP_DEFAULT_X264_BFRAMES);

  ogmrip_x264_dialog_connect_toggle (dialog->cabac_check, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_CABAC, OGMRIP_DEFAULT_X264_CABAC);

  ogmrip_x264_dialog_connect_spin (dialog->subq_spin, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_SUBQ, OGMRIP_DEFAULT_X264_SUBQ);

  ogmrip_x264_dialog_connect_toggle (dialog->b_pyramid_check, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_B_PYRAMID, OGMRIP_DEFAULT_X264_B_PYRAMID);

  ogmrip_x264_dialog_connect_toggle (dialog->weight_b_check, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_WEIGHT_B, OGMRIP_DEFAULT_X264_WEIGHT_B);

  ogmrip_x264_dialog_connect_spin (dialog->frameref_spin, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_FRAMEREF, OGMRIP_DEFAULT_X264_FRAMEREF);

  ogmrip_x264_dialog_connect_combo (dialog->me_combo, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_ME, OGMRIP_DEFAULT_X264_ME);

  ogmrip_x264_dialog_connect_toggle (dialog->brdo_check, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_BRDO, OGMRIP_DEFAULT_X264_BRDO);

  ogmrip_x264_dialog_connect_toggle (dialog->x88dct_check, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_8X8DCT, OGMRIP_DEFAULT_X264_8X8DCT);

  ogmrip_x264_dialog_connect_toggle (dialog->mixed_refs_check, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_MIXED_REFS, OGMRIP_DEFAULT_X264_MIXED_REFS);

  ogmrip_x264_dialog_connect_toggle (dialog->global_header_check, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_GLOBAL_HEADER, OGMRIP_DEFAULT_X264_GLOBAL_HEADER);

  ogmrip_x264_dialog_connect_spin (dialog->vbv_maxrate_spin, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_VBV_MAXRATE, OGMRIP_DEFAULT_X264_VBV_MAXRATE);

  ogmrip_x264_dialog_connect_spin (dialog->vbv_bufsize_spin, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_VBV_BUFSIZE, OGMRIP_DEFAULT_X264_VBV_BUFSIZE);

  ogmrip_x264_dialog_connect_spin (dialog->level_idc_spin, key,
      OGMRIP_GCONF_X264 "/" OGMRIP_GCONF_X264_LEVEL_IDC, OGMRIP_DEFAULT_X264_LEVEL_IDC);
}

static void
ogmrip_x264_set_options (OGMRipVideoCodec *codec)
{
#define GET_BOOL(key,def) key, ogmrip_preferences_get_bool (OGMRIP_GCONF_X264 "/" key, def)
#define GET_INT(key,def)  key, ogmrip_preferences_get_int  (OGMRIP_GCONF_X264 "/" key, def)

  if (ogmrip_preferences_get_int ("video/preset", 0) == OGMRIP_VIDEO_PRESET_USER)
    g_object_set (codec,
        GET_BOOL (OGMRIP_GCONF_X264_8X8DCT,        OGMRIP_DEFAULT_X264_8X8DCT),
        GET_INT  (OGMRIP_GCONF_X264_BFRAMES,       OGMRIP_DEFAULT_X264_BFRAMES),
        GET_BOOL (OGMRIP_GCONF_X264_B_PYRAMID,     OGMRIP_DEFAULT_X264_B_PYRAMID),
        GET_BOOL (OGMRIP_GCONF_X264_BRDO,          OGMRIP_DEFAULT_X264_BRDO),
        GET_BOOL (OGMRIP_GCONF_X264_CABAC,         OGMRIP_DEFAULT_X264_CABAC),
        GET_INT  (OGMRIP_GCONF_X264_FRAMEREF,      OGMRIP_DEFAULT_X264_FRAMEREF),
        GET_INT  (OGMRIP_GCONF_X264_LEVEL_IDC,     OGMRIP_DEFAULT_X264_LEVEL_IDC),
        GET_INT  (OGMRIP_GCONF_X264_ME,            OGMRIP_DEFAULT_X264_ME),
        GET_BOOL (OGMRIP_GCONF_X264_MIXED_REFS,    OGMRIP_DEFAULT_X264_MIXED_REFS),
        GET_INT  (OGMRIP_GCONF_X264_SUBQ,          OGMRIP_DEFAULT_X264_SUBQ),
        GET_INT  (OGMRIP_GCONF_X264_VBV_BUFSIZE,   OGMRIP_DEFAULT_X264_VBV_BUFSIZE),
        GET_INT  (OGMRIP_GCONF_X264_VBV_MAXRATE,   OGMRIP_DEFAULT_X264_VBV_MAXRATE),
        GET_BOOL (OGMRIP_GCONF_X264_WEIGHT_B,      OGMRIP_DEFAULT_X264_WEIGHT_B),
        GET_BOOL (OGMRIP_GCONF_X264_GLOBAL_HEADER, OGMRIP_DEFAULT_X264_GLOBAL_HEADER),
        NULL);

#undef GET_BOOL
#undef GET_INT
}

static OGMRipVideoOptionsPlugin x264_options_plugin =
{
  NULL,
  G_TYPE_NONE,
  G_TYPE_NONE,
  NULL
};

OGMRipVideoOptionsPlugin *
ogmrip_init_options_plugin (void)
{
  x264_options_plugin.type = ogmrip_plugin_get_video_codec_by_name ("x264");
  if (x264_options_plugin.type == G_TYPE_NONE)
    return NULL;

  x264_options_plugin.dialog = OGMRIP_TYPE_X264_DIALOG;

  x264_options_plugin.set_options = ogmrip_x264_set_options;

  return &x264_options_plugin;
}

