/* OGMDvd - A wrapper library around libdvdread
 * Copyright (C) 2004-2008 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/**
 * SECTION:ogmdvd-drive-chooser-dialog
 * @title: OGMDvdDriveChooserDialog
 * @include: ogmdvd-drive-chooser-dialog.h
 * @short_description: A DVD drive chooser dialog
 */

#include "ogmdvd-drive-chooser-dialog.h"
#include "ogmdvd-drive-chooser-widget.h"

#include <glib/gi18n.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkalignment.h>

#define OGMDVD_DRIVE_CHOOSER_DIALOG_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMDVD_TYPE_DRIVE_CHOOSER_DIALOG, OGMDvdDriveChooserDialogPriv))

enum
{
  EJECT,
  LAST_SIGNAL
};

struct _OGMDvdDriveChooserDialogPriv
{
  GtkWidget *chooser;

  GtkWidget *eject_button;
  GtkWidget *load_button;
};

/*
 * GObject vfuncs
 */

static void ogmdvd_drive_chooser_init (OGMDvdDriveChooserIface  *iface);

/*
 * OGMDvdDriveChooser vfuncs
 */

static NautilusBurnDrive * ogmdvd_drive_chooser_dialog_get_drive (OGMDvdDriveChooser *chooser);

/*
 * Internal fucntions
 */

static void ogmdvd_drive_chooser_dialog_eject_clicked  (GtkDialog                *dialog);
static void ogmdvd_drive_chooser_dialog_device_changed (OGMDvdDriveChooserDialog *dialog,
                                                        const gchar              *device,
                                                        GtkWidget                *chooser);

static int signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE_WITH_CODE (OGMDvdDriveChooserDialog, ogmdvd_drive_chooser_dialog, GTK_TYPE_DIALOG,
    G_IMPLEMENT_INTERFACE (OGMDVD_TYPE_DRIVE_CHOOSER, ogmdvd_drive_chooser_init))

static void
ogmdvd_drive_chooser_dialog_class_init (OGMDvdDriveChooserDialogClass *klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);

  /**
   * OGMDvdDriveChooserDialog::eject
   * @dialog: the widget that received the signal
   *
   * Emitted each time the eject button is clicked.
   */
  signals[EJECT] = g_signal_new ("eject", G_TYPE_FROM_CLASS (object_class), 
      G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
      G_STRUCT_OFFSET (OGMDvdDriveChooserDialogClass, eject), NULL, NULL,
      g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

  g_type_class_add_private (klass, sizeof (OGMDvdDriveChooserDialogPriv));
}

static void
ogmdvd_drive_chooser_init (OGMDvdDriveChooserIface  *iface)
{
  iface->get_drive = ogmdvd_drive_chooser_dialog_get_drive;
}

static void
ogmdvd_drive_chooser_dialog_init (OGMDvdDriveChooserDialog *dialog)
{
  NautilusBurnDrive *drive;

  GtkWidget *alignment;
  GtkWidget *image;
  GtkWidget *label;
  GtkWidget *hbox;
  GtkWidget *vbox;

  dialog->priv = OGMDVD_DRIVE_CHOOSER_DIALOG_GET_PRIVATE (dialog);

  gtk_window_set_title (GTK_WINDOW (dialog), _("Open DVD Disk"));
  gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
  gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);

  gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);

  dialog->priv->eject_button = gtk_button_new ();
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), dialog->priv->eject_button, FALSE, FALSE, 0);
  gtk_widget_show (dialog->priv->eject_button);

  g_signal_connect_swapped (dialog->priv->eject_button, "clicked", G_CALLBACK (ogmdvd_drive_chooser_dialog_eject_clicked), dialog);

  alignment = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (dialog->priv->eject_button), alignment);
  gtk_widget_show (alignment);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (alignment), hbox);
  gtk_widget_show (hbox);

  image = gtk_image_new_from_stock (GTK_STOCK_REFRESH, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
  gtk_widget_show (image);

  label = gtk_label_new (_("_Eject"));
  gtk_label_set_use_underline (GTK_LABEL (label), TRUE);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  dialog->priv->load_button = gtk_button_new ();
  gtk_dialog_add_action_widget (GTK_DIALOG (dialog), dialog->priv->load_button, GTK_RESPONSE_OK);
  gtk_widget_show (dialog->priv->load_button);

  alignment = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (dialog->priv->load_button), alignment);
  gtk_widget_show (alignment);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (alignment), hbox);
  gtk_widget_show (hbox);

  image = gtk_image_new_from_stock (GTK_STOCK_CDROM, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
  gtk_widget_show (image);

  label = gtk_label_new (_("_Load"));
  gtk_label_set_use_underline (GTK_LABEL (label), TRUE);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  vbox = gtk_vbox_new (FALSE, 6);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, TRUE, TRUE, 0);
  gtk_widget_show (vbox);

  label = gtk_label_new (_("<b>Select _DVD Drive:</b>"));
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
  gtk_label_set_use_underline (GTK_LABEL (label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  dialog->priv->chooser = ogmdvd_drive_chooser_widget_new ();
  gtk_box_pack_start (GTK_BOX (vbox), dialog->priv->chooser, TRUE, TRUE, 0);
  gtk_widget_show (dialog->priv->chooser);

  g_signal_connect_swapped (dialog->priv->chooser, "device-changed", 
      G_CALLBACK (ogmdvd_drive_chooser_dialog_device_changed), dialog);

  drive = ogmdvd_drive_chooser_get_drive (OGMDVD_DRIVE_CHOOSER (dialog->priv->chooser));

  gtk_widget_set_sensitive (dialog->priv->eject_button, drive != NULL);
  gtk_widget_set_sensitive (dialog->priv->load_button, drive != NULL);

  if (drive)
    g_object_unref (drive);
}

/*
 * OGMDvdDriveChooser vfuncs
 */

static NautilusBurnDrive *
ogmdvd_drive_chooser_dialog_get_drive (OGMDvdDriveChooser *chooser)
{
  OGMDvdDriveChooserDialog *dialog;

  g_return_val_if_fail (OGMDVD_IS_DRIVE_CHOOSER_DIALOG (chooser), NULL);

  dialog = OGMDVD_DRIVE_CHOOSER_DIALOG (chooser);

  return ogmdvd_drive_chooser_get_drive (OGMDVD_DRIVE_CHOOSER (dialog->priv->chooser));
}

/*
 * Internal functions
 */

static void
ogmdvd_drive_chooser_dialog_eject_clicked (GtkDialog *dialog)
{
  NautilusBurnDrive *drive;
    
  drive = ogmdvd_drive_chooser_get_drive (OGMDVD_DRIVE_CHOOSER (OGMDVD_DRIVE_CHOOSER_DIALOG (dialog)->priv->chooser));
  if (drive)
  {
    g_signal_emit (dialog, signals[EJECT], 0, NULL);
    nautilus_burn_drive_eject ((NautilusBurnDrive *) drive);
    g_object_unref (drive);
  }
}

static void
ogmdvd_drive_chooser_dialog_device_changed (OGMDvdDriveChooserDialog *dialog, const gchar *device, GtkWidget *chooser)
{
  gtk_widget_set_sensitive (dialog->priv->load_button, device != NULL);
  gtk_widget_set_sensitive (dialog->priv->eject_button, device != NULL);
}

/**
 * ogmdvd_drive_chooser_dialog_new:
 *
 * Creates a new #OGMDvdDriveChooserDialog.
 *
 * Returns: The new #OGMDvdDriveChooserDialog
 */
GtkWidget *
ogmdvd_drive_chooser_dialog_new (void)
{
  GtkWidget *widget;

  widget = g_object_new (OGMDVD_TYPE_DRIVE_CHOOSER_DIALOG, NULL);

  return widget;
}

