/* SPUdec.c
   Skeleton of function spudec_process_controll() is from xine sources.
   Further works:
   LGB,... (yeah, try to improve it and insert your name here! ;-)

   Kim Minh Kaplan
   implement fragments reassembly, RLE decoding.
   read brightness from the IFO.

   For information on SPU format see <URL:http://sam.zoy.org/doc/dvd/subtitles/>
   and <URL:http://members.aol.com/mpucoder/DVD/spu.html>

 */
// #include "config.h"
#include "mp_msg.h"

#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
// #include "libvo/video_out.h"
#include "spudec.h"
// #include "postproc/swscale.h"

#define MIN(a, b)	((a)<(b)?(a):(b))

/* Valid values for spu_aamode:
   0: none (fastest, most ugly)
   1: approximate
   2: full (slowest)
   3: bilinear (similiar to vobsub, fast and not too bad)
   4: uses swscaler gaussian (this is the only one that looks good)
 */

extern int sub_pos;

static void
spudec_queue_packet (spudec_handle_t * this, packet_t * packet)
{
  if (this->queue_head == NULL)
    this->queue_head = packet;
  else
    this->queue_tail->next = packet;
  this->queue_tail = packet;
}

static packet_t *
spudec_dequeue_packet (spudec_handle_t * this)
{
  packet_t *retval = this->queue_head;

  this->queue_head = retval->next;
  if (this->queue_head == NULL)
    this->queue_tail = NULL;

  return retval;
}

static void
spudec_free_packet (packet_t * packet)
{
  if (packet->packet != NULL)
    free (packet->packet);
  free (packet);
}

static inline unsigned int
get_be16 (const unsigned char *p)
{
  return (p[0] << 8) + p[1];
}

static inline unsigned int
get_be24 (const unsigned char *p)
{
  return (get_be16 (p) << 8) + p[2];
}

static void
next_line (packet_t * packet)
{
  if (packet->current_nibble[packet->deinterlace_oddness] % 2)
    packet->current_nibble[packet->deinterlace_oddness]++;
  packet->deinterlace_oddness = (packet->deinterlace_oddness + 1) % 2;
}

static inline unsigned char
get_nibble (packet_t * packet)
{
  unsigned char nib;
  unsigned int *nibblep =
    packet->current_nibble + packet->deinterlace_oddness;
  if (*nibblep / 2 >= packet->control_start)
  {
    mp_msg (MSGT_SPUDEC, MSGL_WARN,
            "SPUdec: ERROR: get_nibble past end of packet\n");
    return 0;
  }
  nib = packet->packet[*nibblep / 2];
  if (*nibblep % 2)
    nib &= 0xf;
  else
    nib >>= 4;
  ++*nibblep;
  return nib;
}

static inline int
mkalpha (int i)
{
  /*
   * In mplayer's alpha planes, 0 is transparent, then 1 is nearly
   * opaque upto 255 which is transparent 
   */
  switch (i)
  {
    case 0xf:
      return 1;
    case 0:
      return 0;
    default:
      return (0xf - i) << 4;
  }
}

/* Cut the sub to visible part */
static inline void
spudec_cut_image (spudec_handle_t * this)
{
  unsigned int fy, ly;
  unsigned int first_y, last_y;
  unsigned char *image;
  unsigned char *aimage;

  if (this->stride == 0 || this->height == 0)
  {
    return;
  }

  for (fy = 0; fy < this->image_size && !this->aimage[fy]; fy++);
  for (ly = this->stride * this->height - 1; ly && !this->aimage[ly]; ly--);
  first_y = fy / this->stride;
  last_y = ly / this->stride;
  this->start_row += first_y;

  // Some subtitles trigger this condition
  if (last_y + 1 > first_y)
  {
    this->height = last_y - first_y + 1;
  }
  else
  {
    this->height = 0;
    this->image_size = 0;
    return;
  }

  image = malloc (2 * this->stride * this->height);
  if (image)
  {
    this->image_size = this->stride * this->height;
    aimage = image + this->image_size;
    memcpy (image, this->image + this->stride * first_y, this->image_size);
    memcpy (aimage, this->aimage + this->stride * first_y, this->image_size);
    free (this->image);
    this->image = image;
    this->aimage = aimage;
  }
  else
  {
    mp_msg (MSGT_SPUDEC, MSGL_FATAL,
            "Fatal: update_spu: malloc requested %d bytes\n",
            2 * this->stride * this->height);
  }
}

static void
spudec_process_data (spudec_handle_t * this, packet_t * packet)
{
  unsigned int cmap[4], alpha[4];
  unsigned int i, x, y;

  this->scaled_frame_width = 0;
  this->scaled_frame_height = 0;
  this->start_col = packet->start_col;
  this->end_col = packet->end_col;
  this->start_row = packet->start_row;
  this->end_row = packet->end_row;
  this->height = packet->height;
  this->width = packet->width;
  this->stride = packet->stride;
  for (i = 0; i < 4; ++i)
  {
    alpha[i] = mkalpha (packet->alpha[i]);
    if (alpha[i] == 0)
      cmap[i] = 0;
    else if (this->custom)
    {
      cmap[i] = ((this->cuspal[i] >> 16) & 0xff);
      if (cmap[i] + alpha[i] > 255)
        cmap[i] = 256 - alpha[i];
    }
    else
    {
      cmap[i] = ((this->global_palette[packet->palette[i]] >> 16) & 0xff);
      if (cmap[i] + alpha[i] > 255)
        cmap[i] = 256 - alpha[i];
    }
  }

  if (this->image_size < this->stride * this->height)
  {
    if (this->image != NULL)
    {
      free (this->image);
      this->image_size = 0;
    }
    this->image = malloc (2 * this->stride * this->height);
    if (this->image)
    {
      this->image_size = this->stride * this->height;
      this->aimage = this->image + this->image_size;
    }
  }
  if (this->image == NULL)
    return;

  /*
   * Kludge: draw_alpha needs width multiple of 8. 
   */
  if (this->width < this->stride)
    for (y = 0; y < this->height; ++y)
    {
      memset (this->aimage + y * this->stride + this->width, 0,
              this->stride - this->width);
      /*
       * FIXME: Why is this one needed? 
       */
      memset (this->image + y * this->stride + this->width, 0,
              this->stride - this->width);
    }

  i = packet->current_nibble[1];
  x = 0;
  y = 0;
  while (packet->current_nibble[0] < i
         && packet->current_nibble[1] / 2 < packet->control_start
         && y < this->height)
  {
    unsigned int len, color;
    unsigned int rle = 0;
    rle = get_nibble (packet);
    if (rle < 0x04)
    {
      rle = (rle << 4) | get_nibble (packet);
      if (rle < 0x10)
      {
        rle = (rle << 4) | get_nibble (packet);
        if (rle < 0x040)
        {
          rle = (rle << 4) | get_nibble (packet);
          if (rle < 0x0004)
            rle |= ((this->width - x) << 2);
        }
      }
    }
    color = 3 - (rle & 0x3);
    len = rle >> 2;
    if (len > this->width - x || len == 0)
      len = this->width - x;
    /*
     * FIXME have to use palette and alpha map
     */
    memset (this->image + y * this->stride + x, cmap[color], len);
    memset (this->aimage + y * this->stride + x, alpha[color], len);
    x += len;
    if (x >= this->width)
    {
      next_line (packet);
      x = 0;
      ++y;
    }
  }
  spudec_cut_image (this);
}


/*
  This function tries to create a usable palette.
  It determines how many non-transparent colors are used, and assigns different
gray scale values to each color.
  I tested it with four streams and even got something readable. Half of the
times I got black characters with white around and half the reverse.
*/
static void
compute_palette (spudec_handle_t * this, packet_t * packet)
{
  int used[16], i, cused, start, step, color;

  memset (used, 0, sizeof (used));
  for (i = 0; i < 4; i++)
    if (packet->alpha[i])       /* !Transparent? */
      used[packet->palette[i]] = 1;
  for (cused = 0, i = 0; i < 16; i++)
    if (used[i])
      cused++;
  if (!cused)
    return;
  if (cused == 1)
  {
    start = 0x80;
    step = 0;
  }
  else
  {
    start = this->font_start_level;
    step = (0xF0 - this->font_start_level) / (cused - 1);
  }
  memset (used, 0, sizeof (used));
  for (i = 0; i < 4; i++)
  {
    color = packet->palette[i];
    if (packet->alpha[i] && !used[color])
    {                           /* not assigned? */
      used[color] = 1;
      this->global_palette[color] = start << 16;
      start += step;
    }
  }
}

static void
spudec_process_control (spudec_handle_t * this, unsigned int pts100)
{
  int a, b;                     /* Temporary vars */
  unsigned int date, type;
  unsigned int off;
  unsigned int start_off = 0;
  unsigned int next_off;
  unsigned int start_pts = 0;
  unsigned int end_pts = 0;
  unsigned int current_nibble[2] = { 0, 0 };
  unsigned int control_start;
  unsigned int display = 0;
  unsigned int start_col = 0;
  unsigned int end_col = 0;
  unsigned int start_row = 0;
  unsigned int end_row = 0;
  unsigned int width = 0;
  unsigned int height = 0;
  unsigned int stride = 0;

  control_start = get_be16 (this->packet + 2);
  next_off = control_start;
  while (start_off != next_off)
  {
    start_off = next_off;
    date = get_be16 (this->packet + start_off) * 1024;
    next_off = get_be16 (this->packet + start_off + 2);
    mp_msg (MSGT_SPUDEC, MSGL_DBG2, "date=%d\n", date);
    off = start_off + 4;
    for (type = this->packet[off++]; type != 0xff; type = this->packet[off++])
    {
      mp_msg (MSGT_SPUDEC, MSGL_DBG2, "cmd=%d  ", type);
      switch (type)
      {
        case 0x00:
          /*
           * Menu ID, 1 byte 
           */
          mp_msg (MSGT_SPUDEC, MSGL_DBG2, "Menu ID\n");
          /*
           * shouldn't a Menu ID type force display start? 
           */
          start_pts = pts100 + date;
          end_pts = UINT_MAX;
          display = 1;
          this->is_forced_sub = ~0; // current subtitle is forced
          break;
        case 0x01:
          /*
           * Start display 
           */
          mp_msg (MSGT_SPUDEC, MSGL_DBG2, "Start display!\n");
          start_pts = pts100 + date;
          end_pts = UINT_MAX;
          display = 1;
          this->is_forced_sub = 0;
          break;
        case 0x02:
          /*
           * Stop display 
           */
          mp_msg (MSGT_SPUDEC, MSGL_DBG2, "Stop display!\n");
          end_pts = pts100 + date;
          break;
        case 0x03:
          /*
           * Palette 
           */
          this->palette[0] = this->packet[off] >> 4;
          this->palette[1] = this->packet[off] & 0xf;
          this->palette[2] = this->packet[off + 1] >> 4;
          this->palette[3] = this->packet[off + 1] & 0xf;
          mp_msg (MSGT_SPUDEC, MSGL_DBG2, "Palette %d, %d, %d, %d\n",
                  this->palette[0], this->palette[1], this->palette[2],
                  this->palette[3]);
          off += 2;
          break;
        case 0x04:
          /*
           * Alpha 
           */
          this->alpha[0] = this->packet[off] >> 4;
          this->alpha[1] = this->packet[off] & 0xf;
          this->alpha[2] = this->packet[off + 1] >> 4;
          this->alpha[3] = this->packet[off + 1] & 0xf;
          mp_msg (MSGT_SPUDEC, MSGL_DBG2, "Alpha %d, %d, %d, %d\n",
                  this->alpha[0], this->alpha[1], this->alpha[2],
                  this->alpha[3]);
          off += 2;
          break;
        case 0x05:
          /*
           * Co-ords 
           */
          a = get_be24 (this->packet + off);
          b = get_be24 (this->packet + off + 3);
          start_col = a >> 12;
          end_col = a & 0xfff;
          width = (end_col < start_col) ? 0 : end_col - start_col + 1;
          stride = (width + 7) & ~7;  /* Kludge: draw_alpha needs width multiple of 8 */
          start_row = b >> 12;
          end_row = b & 0xfff;
          height = (end_row < start_row) ? 0 : end_row - start_row /* + 1 */ ;
          mp_msg (MSGT_SPUDEC, MSGL_DBG2,
                  "Coords  col: %d - %d  row: %d - %d  (%dx%d)\n", start_col,
                  end_col, start_row, end_row, width, height);
          off += 6;
          break;
        case 0x06:
          /*
           * Graphic lines 
           */
          current_nibble[0] = 2 * get_be16 (this->packet + off);
          current_nibble[1] = 2 * get_be16 (this->packet + off + 2);
          mp_msg (MSGT_SPUDEC, MSGL_DBG2,
                  "Graphic offset 1: %d  offset 2: %d\n",
                  current_nibble[0] / 2, current_nibble[1] / 2);
          off += 4;
          break;
        case 0xff:
          /*
           * All done, bye-bye 
           */
          mp_msg (MSGT_SPUDEC, MSGL_DBG2, "Done!\n");
          return;
//  break;
        default:
          mp_msg (MSGT_SPUDEC, MSGL_WARN,
                  "spudec: Error determining control type 0x%02x.  Skipping %d bytes.\n",
                  type, next_off - off);
          goto next_control;
      }
    }
  next_control:
    if (display)
    {
      packet_t *packet = calloc (1, sizeof (packet_t));
      int i;
      packet->start_pts = start_pts;
      if (end_pts == UINT_MAX && start_off != next_off)
      {
        start_pts = pts100 + get_be16 (this->packet + next_off) * 1024;
        packet->end_pts = start_pts - 1;
      }
      else
        packet->end_pts = end_pts;
      packet->current_nibble[0] = current_nibble[0];
      packet->current_nibble[1] = current_nibble[1];
      packet->start_row = start_row;
      packet->end_row = end_row;
      packet->start_col = start_col;
      packet->end_col = end_col;
      packet->width = width;
      packet->height = height;
      packet->stride = stride;
      packet->control_start = control_start;
      for (i = 0; i < 4; i++)
      {
        packet->alpha[i] = this->alpha[i];
        packet->palette[i] = this->palette[i];
      }
      packet->packet = malloc (this->packet_size);
      memcpy (packet->packet, this->packet, this->packet_size);
      spudec_queue_packet (this, packet);
    }
  }
}

static void
spudec_decode (spudec_handle_t * this, unsigned int pts100)
{
  /*
   * if(this->hw_spu) {
   * static vo_mpegpes_t packet = { NULL, 0, 0x20, 0 };
   * static vo_mpegpes_t *pkg=&packet;
   * packet.data = this->packet;
   * packet.size = this->packet_size;
   * packet.timestamp = pts100;
   * this->hw_spu->draw_frame((uint8_t**)&pkg);
   * } else
   */
  spudec_process_control (this, pts100);
}

int
spudec_changed (void *this)
{
  spudec_handle_t *spu = (spudec_handle_t *) this;
  return (spu->spu_changed || spu->now_pts > spu->end_pts);
}

void
spudec_assemble (void *this, unsigned char *packet, unsigned int len,
                 unsigned int pts100)
{
  spudec_handle_t *spu = (spudec_handle_t *) this;
//  spudec_heartbeat(this, pts100);
  if (len < 2)
  {
    mp_msg (MSGT_SPUDEC, MSGL_WARN, "SPUasm: packet too short\n");
    return;
  }
#if 0
  if ((spu->packet_pts + 10000) < pts100)
  {
    // [cb] too long since last fragment: force new packet
    spu->packet_offset = 0;
  }
#endif
  spu->packet_pts = pts100;
  if (spu->packet_offset == 0)
  {
    unsigned int len2 = get_be16 (packet);
    // Start new fragment
    if (spu->packet_reserve < len2)
    {
      if (spu->packet != NULL)
        free (spu->packet);
      spu->packet = malloc (len2);
      spu->packet_reserve = spu->packet != NULL ? len2 : 0;
    }
    if (spu->packet != NULL)
    {
      spu->packet_size = len2;
      if (len > len2)
      {
        mp_msg (MSGT_SPUDEC, MSGL_WARN,
                "SPUasm: invalid frag len / len2: %d / %d \n", len, len2);
        return;
      }
      memcpy (spu->packet, packet, len);
      spu->packet_offset = len;
      spu->packet_pts = pts100;
    }
  }
  else
  {
    // Continue current fragment
    if (spu->packet_size < spu->packet_offset + len)
    {
      mp_msg (MSGT_SPUDEC, MSGL_WARN, "SPUasm: invalid fragment\n");
      spu->packet_size = spu->packet_offset = 0;
      return;
    }
    else
    {
      memcpy (spu->packet + spu->packet_offset, packet, len);
      spu->packet_offset += len;
    }
  }
#if 1
  // check if we have a complete packet (unfortunatelly packet_size is bad
  // for some disks)
  // [cb] packet_size is padded to be even -> may be one byte too long
  if ((spu->packet_offset == spu->packet_size) ||
      ((spu->packet_offset + 1) == spu->packet_size))
  {
    unsigned int x = 0, y;
    while (x + 4 <= spu->packet_offset)
    {
      y = get_be16 (spu->packet + x + 2); // next control pointer
      mp_msg (MSGT_SPUDEC, MSGL_DBG2, "SPUtest: x=%d y=%d off=%d size=%d\n",
              x, y, spu->packet_offset, spu->packet_size);
      if (x >= 4 && x == y)
      {                         // if it points to self - we're done!
        // we got it!
        mp_msg (MSGT_SPUDEC, MSGL_DBG2, "SPUgot: off=%d  size=%d \n",
                spu->packet_offset, spu->packet_size);
        spudec_decode (spu, pts100);
        spu->packet_offset = 0;
        break;
      }
      if (y <= x || y >= spu->packet_size)
      {                         // invalid?
        mp_msg (MSGT_SPUDEC, MSGL_WARN,
                "SPUtest: broken packet!!!!! y=%d < x=%d\n", y, x);
        spu->packet_size = spu->packet_offset = 0;
        break;
      }
      x = y;
    }
    // [cb] packet is done; start new packet
    spu->packet_offset = 0;
  }
#else
  if (spu->packet_offset == spu->packet_size)
  {
    spudec_decode (spu, pts100);
    spu->packet_offset = 0;
  }
#endif
}

void
spudec_heartbeat (void *this, unsigned int pts100)
{
  spudec_handle_t *spu = (spudec_handle_t *) this;
  spu->now_pts = pts100;

  while (spu->queue_head != NULL && pts100 >= spu->queue_head->start_pts)
  {
    packet_t *packet = spudec_dequeue_packet (spu);
    spu->start_pts = packet->start_pts;
    spu->end_pts = packet->end_pts;
    if (spu->auto_palette)
      compute_palette (spu, packet);
    spudec_process_data (spu, packet);
    spudec_free_packet (packet);
    spu->spu_changed = 1;
  }
}

void
spudec_set_forced_subs_only (void *const this, const unsigned int flag)
{
  if (this)
  {
    ((spudec_handle_t *) this)->forced_subs_only = flag;
    mp_msg (MSGT_SPUDEC, MSGL_DBG2, "SPU: Display only forced subs now %s\n",
            flag ? "enabled" : "disabled");
  }
}

void
spudec_draw (void *this, draw_func draw_alpha)
{
  spudec_handle_t *spu = (spudec_handle_t *) this;
  if (spu->start_pts <= spu->now_pts && spu->now_pts < spu->end_pts && spu->image)
  {
    if (spu->forced_subs_only && !spu->is_forced_sub)
      return;

    draw_alpha (spu->start_col, spu->start_row, spu->width, spu->height,
                spu->image, spu->aimage, spu->stride);
    spu->spu_changed = 0;
  }
}

/* transform mplayer's alpha value into an opacity value that is linear */
static inline int
canon_alpha (int alpha)
{
  return alpha ? 256 - alpha : 0;
}

void *
spudec_new_scaled (unsigned int *palette, unsigned int frame_width,
                   unsigned int frame_height)
{
  return spudec_new_scaled_vobsub (palette, NULL, 0, frame_width,
                                   frame_height);
}

/* get palette custom color, width, height from .idx file */
void *
spudec_new_scaled_vobsub (unsigned int *palette, unsigned int *cuspal,
                          unsigned int custom, unsigned int frame_width,
                          unsigned int frame_height)
{
  spudec_handle_t *this = calloc (1, sizeof (spudec_handle_t));
  if (this)
  {
    this->packet = NULL;
    this->image = NULL;
    this->scaled_image = NULL;
    /*
     * XXX Although the video frame is some size, the SPU frame is
     * always maximum size i.e. 720 wide and 576 or 480 high 
     */
    this->orig_frame_width = 720;
    this->orig_frame_height = (frame_height == 480
                               || frame_height == 240) ? 480 : 576;
    this->custom = custom;
    // set up palette:
    this->auto_palette = 1;
    if (palette)
    {
      memcpy (this->global_palette, palette, sizeof (this->global_palette));
      this->auto_palette = 0;
    }
    this->custom = custom;
    if (custom && cuspal)
    {
      memcpy (this->cuspal, cuspal, sizeof (this->cuspal));
      this->auto_palette = 0;
    }
    // forced subtitles default: show all subtitles
    this->forced_subs_only = 0;
    this->is_forced_sub = 0;
  }
  else
    mp_msg (MSGT_SPUDEC, MSGL_FATAL, "FATAL: spudec_init: calloc");
  return this;
}

void *
spudec_new (unsigned int *palette)
{
  return spudec_new_scaled (palette, 0, 0);
}

