/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include "ogmdvd.h"

#include "ogmrip-options.h"
#include "ogmrip-helper.h"
#include "ogmrip-plugin.h"
#include "ogmrip-crop.h"
#include "ogmrip-lavc.h"

#include <glib/gi18n.h>
#include <glade/glade.h>

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-options.glade"
#define OGMRIP_GLADE_ROOT "root"

#define ROUND(x) ((gint) ((x) + 0.5) != (gint) (x) ? ((gint) ((x) + 0.5)) : ((gint) (x)))

#define OGMRIP_OPTIONS_DIALOG_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_OPTIONS_DIALOG, OGMRipOptionsDialogPriv))

enum
{
  OGMRIP_SCALE_NONE,
  OGMRIP_SCALE_XSMALL,
  OGMRIP_SCALE_SMALL,
  OGMRIP_SCALE_MEDIUM,
  OGMRIP_SCALE_LARGE,
  OGMRIP_SCALE_XLARGE,
  OGMRIP_SCALE_FULL,
  OGMRIP_SCALE_USER
};

struct _OGMRipOptionsDialogPriv
{
  OGMDvdTitle *title;

  GtkWidget *bitrate_check;
  GtkWidget *bitrate_spin;

  GtkWidget *crop_check;
  GtkWidget *crop_left_label;
  GtkWidget *crop_right_label;
  GtkWidget *crop_top_label;
  GtkWidget *crop_bottom_label;

  GtkWidget *scale_check;
  GtkWidget *scale_combo;
  GtkWidget *scale_user_hbox;
  GtkWidget *scale_width_spin;
  GtkWidget *scale_height_spin;
  GtkWidget *autoscale_button;
  GtkListStore *scale_store;

  GtkWidget *more_exp;
  GtkWidget *ntsc_frame;
  GtkWidget *bpp_spin;
  GtkWidget *cartoon_check;

  GtkWidget *deint_combo;
  GtkWidget *framestep_combo;
  GtkWidget *progressive_check;
  GtkWidget *pullup_check;
};

static void ogmrip_options_dialog_dispose (GObject *gobject);

static gint
ogmrip_options_dialog_get_crop_internal (OGMRipOptionsDialog *dialog, guint *x, guint *y, guint *width, guint *height)
{
  gint retval;

  retval = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->crop_check)) ?
    OGMRIP_OPTIONS_AUTOMATIC : OGMRIP_OPTIONS_MANUAL;

  *x = *y = 0;
  ogmdvd_title_get_size (dialog->priv->title, width, height);

  if (retval == OGMRIP_OPTIONS_MANUAL)
  {
    gint right, bottom;

    *x = gtk_label_get_int (GTK_LABEL (dialog->priv->crop_left_label));
    *y = gtk_label_get_int (GTK_LABEL (dialog->priv->crop_top_label));

    right = gtk_label_get_int (GTK_LABEL (dialog->priv->crop_right_label));
    bottom = gtk_label_get_int (GTK_LABEL (dialog->priv->crop_bottom_label));

    *width -= *x + right;
    *height -= *y + bottom;

    if (*x == 0 && *y == 0 && right == 0 && bottom == 0)
      return OGMRIP_OPTIONS_NONE;
  }

  return retval;
}

static gdouble
ogmrip_options_dialog_get_aspect (OGMRipOptionsDialog *dialog, guint crop_width, guint crop_height)
{
  guint raw_width, raw_height, numerator, denominator;

  ogmdvd_title_get_size (dialog->priv->title, &raw_width, &raw_height);

  ogmdvd_title_get_aspect_ratio (dialog->priv->title, &numerator, &denominator);

  return (crop_width * raw_height * numerator) / (gdouble) (crop_height * raw_width * denominator);
}

static void
ogmrip_options_dialog_scale (guint size, gdouble aspect, guint *width, guint *height)
{
  const struct { guint mult; guint div; } table[] = 
  { 
    { 0, 0 }, { 1, 8 }, { 1, 4 }, { 1, 2 }, { 3, 4 }, { 5, 6 }, { 1, 1 } 
  };

  g_return_if_fail (size != 0);

  *width = *width * table[size].mult / table[size].div;
  *width += *width % 2;

  *width = 16 * ROUND (*width / 16.);
  *height = 16 * ROUND (*width / aspect / 16);
}

static void
ogmrip_options_dialog_scale_update (OGMRipOptionsDialog *dialog)
{
  GtkTreeIter iter;
  guint scale_width, scale_height;
  guint crop_x, crop_y, crop_width, crop_height;
  gfloat aspect;
  gchar text[16];
  gint i, active;

  ogmrip_options_dialog_get_crop_internal (dialog, &crop_x, &crop_y, &crop_width, &crop_height);
  aspect = ogmrip_options_dialog_get_aspect (dialog, crop_width, crop_height);

  gtk_spin_button_set_range (GTK_SPIN_BUTTON (dialog->priv->scale_width_spin), 2, crop_width);
  gtk_spin_button_set_range (GTK_SPIN_BUTTON (dialog->priv->scale_height_spin), 2, crop_height);

  active = gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->priv->scale_combo));

  for (i = 1; i < OGMRIP_SCALE_USER; i++)
  {
    if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (dialog->priv->scale_store), &iter, NULL, i))
    {
      scale_width = crop_width;
      scale_height = crop_height;

      ogmrip_options_dialog_scale (i, aspect, &scale_width, &scale_height);

      snprintf (text, 16, "%u x %u", scale_width, scale_height);
      gtk_list_store_set (dialog->priv->scale_store, &iter, 1, text, -1);

      if (active == i)
      {
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->scale_width_spin), (gdouble) scale_width);
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->scale_height_spin), (gdouble) scale_height);
      }
    }
  }
}

static void
ogmrip_options_dialog_crop_update (OGMRipOptionsDialog *dialog)
{
  gtk_label_set_int (GTK_LABEL (dialog->priv->crop_left_label), 0);
  gtk_label_set_int (GTK_LABEL (dialog->priv->crop_top_label), 0);
  gtk_label_set_int (GTK_LABEL (dialog->priv->crop_right_label), 0);
  gtk_label_set_int (GTK_LABEL (dialog->priv->crop_bottom_label), 0);
}

static void
ogmrip_options_dialog_scale_combo_changed (OGMRipOptionsDialog *dialog, GtkWidget *combo)
{
  gint active;

  active = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

  gtk_widget_set_sensitive (dialog->priv->scale_user_hbox, active == OGMRIP_SCALE_USER);

  if (active > 0 && active < OGMRIP_SCALE_USER)
  {
    guint x, y, width, height;
    gdouble aspect;

    ogmrip_options_dialog_get_crop_internal (dialog, &x, &y, &width, &height);
    aspect = ogmrip_options_dialog_get_aspect (dialog, width, height);

    ogmrip_options_dialog_scale (active, aspect, &width, &height);

    gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->scale_width_spin), width);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->scale_height_spin), height);
  }
}

static void
ogmrip_options_dialog_autoscale_clicked (OGMRipOptionsDialog *dialog)
{
  GType codec;

  codec = ogmrip_plugin_get_nth_video_codec (0);
  if (codec == G_TYPE_NONE)
    ogmrip_message_dialog (GTK_WINDOW (dialog), GTK_MESSAGE_ERROR, "<b>%s</b>\n\n%s",
        _("Could not detect scaling parameters"), _("There is no video codec available"));
  else
  {
    OGMJobSpawn *spawn;
    guint scale_width, scale_height;
    guint crop_x, crop_y, crop_width, crop_height;
    guint bitrate;

    spawn = g_object_new (codec, "input", dialog->priv->title, NULL);

    ogmrip_options_dialog_get_crop_internal (dialog, &crop_x, &crop_y, &crop_width, &crop_height);
    ogmrip_video_set_crop_size (OGMRIP_VIDEO (spawn), crop_x, crop_y, crop_width, crop_height);

    if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->bitrate_check)))
    {
      bitrate = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->priv->bitrate_spin));
      ogmrip_video_set_bitrate (OGMRIP_VIDEO (spawn), bitrate);
    }

    ogmrip_video_autoscale (OGMRIP_VIDEO (spawn));
    ogmrip_video_get_scale_size (OGMRIP_VIDEO (spawn), &scale_width, &scale_height);
    g_object_unref (spawn);

    gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->scale_width_spin), (gdouble) scale_width);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->scale_height_spin), (gdouble) scale_height);

    gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->scale_combo), OGMRIP_SCALE_USER);
  }
}

static void
ogmrip_options_dialog_crop_clicked (OGMRipOptionsDialog *dialog)
{
  GtkWidget *crop_dialog;
  guint old_left, old_top, old_right, old_bottom;
  guint new_left, new_top, new_right, new_bottom;

  old_left = gtk_label_get_int (GTK_LABEL (dialog->priv->crop_left_label));
  old_top = gtk_label_get_int (GTK_LABEL (dialog->priv->crop_top_label));
  old_right = gtk_label_get_int (GTK_LABEL (dialog->priv->crop_right_label));
  old_bottom = gtk_label_get_int (GTK_LABEL (dialog->priv->crop_bottom_label));

  crop_dialog = ogmrip_crop_dialog_new (dialog->priv->title, old_left, old_top, old_right, old_bottom);
  g_signal_connect (crop_dialog, "delete-event", G_CALLBACK (gtk_true), NULL);
  g_signal_connect (crop_dialog, "response", G_CALLBACK (gtk_widget_hide), NULL);

  if (gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->priv->deint_combo)))
    ogmrip_crop_dialog_set_deinterlacer (OGMRIP_CROP_DIALOG (crop_dialog), TRUE);

  if (gtk_dialog_run (GTK_DIALOG (crop_dialog)) == GTK_RESPONSE_OK)
  {
    ogmrip_crop_dialog_get_crop (OGMRIP_CROP_DIALOG (crop_dialog), &new_left, &new_top, &new_right, &new_bottom);
    if (new_left != old_left || new_right != old_right || new_top != old_top || new_bottom != old_bottom)
    {
      gtk_label_set_int (GTK_LABEL (dialog->priv->crop_left_label), new_left);
      gtk_label_set_int (GTK_LABEL (dialog->priv->crop_top_label), new_top);
      gtk_label_set_int (GTK_LABEL (dialog->priv->crop_right_label), new_right);
      gtk_label_set_int (GTK_LABEL (dialog->priv->crop_bottom_label), new_bottom);

      ogmrip_options_dialog_scale_update (dialog);
    }
  }
  gtk_widget_destroy (crop_dialog);
}

static void
ogmrip_options_dialog_autocrop_clicked (OGMRipOptionsDialog *dialog)
{
  GType codec;

  codec = ogmrip_plugin_get_nth_video_codec (0);
  if (codec == G_TYPE_NONE)
    ogmrip_message_dialog (GTK_WINDOW (dialog), GTK_MESSAGE_ERROR, "<b>%s</b>\n\n%s",
        _("Could not detect cropping parameters"), _("There is no video codec available"));
  else
  {
    OGMJobSpawn *spawn;
    GtkWidget *msg_dialog;
    guint old_x, old_y, old_width, old_height;
    guint new_x, new_y, new_width, new_height;
    guint raw_width, raw_height;

    ogmrip_options_dialog_get_crop_internal (dialog, &old_x, &old_y, &old_width, &old_height);

    msg_dialog = gtk_message_dialog_new_with_markup (GTK_WINDOW (dialog), GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
        GTK_MESSAGE_INFO, GTK_BUTTONS_NONE, "<b>%s</b>\n\n%s", _("Detecting cropping parameters"), _("Please wait"));

    gtk_window_set_icon_from_stock (GTK_WINDOW (msg_dialog), GTK_STOCK_DIALOG_INFO);
    gtk_widget_show (msg_dialog);

    spawn = g_object_new (codec, "input", dialog->priv->title, NULL);
    ogmrip_video_set_deinterlacer (OGMRIP_VIDEO (spawn),
        gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->priv->deint_combo)));

    ogmrip_video_autocrop (OGMRIP_VIDEO (spawn), 0);
    ogmrip_video_get_crop_size (OGMRIP_VIDEO (spawn), &new_x, &new_y, &new_width, &new_height);
    g_object_unref (spawn);

    gtk_widget_destroy (msg_dialog);

    ogmdvd_title_get_size (dialog->priv->title, &raw_width, &raw_height);

    if (new_width == 0)
      new_width = raw_width;

    if (new_height == 0)
      new_height = raw_height;

    if (new_x != old_x || new_y != old_y || new_width != old_width || new_height != old_height)
    {
      gtk_label_set_int (GTK_LABEL (dialog->priv->crop_left_label), new_x);
      gtk_label_set_int (GTK_LABEL (dialog->priv->crop_top_label), new_y);

      if (raw_width > new_x + new_width)
        gtk_label_set_int (GTK_LABEL (dialog->priv->crop_right_label), raw_width - new_x - new_width);
      else
        gtk_label_set_int (GTK_LABEL (dialog->priv->crop_right_label), 0);

      if (raw_height > new_y + new_height)
        gtk_label_set_int (GTK_LABEL (dialog->priv->crop_bottom_label), raw_height - new_y - new_height);
      else
        gtk_label_set_int (GTK_LABEL (dialog->priv->crop_bottom_label), 0);

      ogmrip_options_dialog_scale_update (dialog);
    }
  }
}

static void
ogmrip_options_dialog_ccheck_clicked (OGMRipOptionsDialog *dialog)
{
  gtk_dialog_response (GTK_DIALOG (dialog), OGMRIP_OPTIONS_CCHECK);
}

static void
ogmrip_options_dialog_check_toggled (GtkWidget *check, GtkWidget *widget)
{
  gtk_widget_set_sensitive (widget, 
      gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check)));
}

static void
ogmrip_options_dialog_check_untoggled (GtkWidget *check, GtkWidget *widget)
{
  gtk_widget_set_sensitive (widget, 
      !gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check)));
}

static void
ogmrip_options_dialog_bitrate_or_crop_toggled (OGMRipOptionsDialog *dialog)
{
  gboolean active;

  active = (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->bitrate_check)) &
    !gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->crop_check)));

  gtk_widget_set_sensitive (dialog->priv->autoscale_button, active);
}

G_DEFINE_TYPE (OGMRipOptionsDialog, ogmrip_options_dialog, GTK_TYPE_DIALOG);

static void
ogmrip_options_dialog_class_init (OGMRipOptionsDialogClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->dispose = ogmrip_options_dialog_dispose;

  g_type_class_add_private (klass, sizeof (OGMRipOptionsDialogPriv));
}

static void
ogmrip_options_dialog_init (OGMRipOptionsDialog *dialog)
{
  GtkCellRenderer *renderer;
  GtkWidget *widget;
  GladeXML *xml;
  guint i;

  GtkTreeIter iter;
  gchar *size[] = { N_("None"), N_("Extra Small"), N_("Small"), N_("Medium"), 
    N_("Large"), N_("Extra Large"), N_("Full"), N_("User Defined") };

  dialog->priv = OGMRIP_OPTIONS_DIALOG_GET_PRIVATE (dialog);

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, OGMRIP_GLADE_ROOT, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return;
  }

  gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);

  widget = gtk_button_new_with_image (_("E_xtract"), GTK_STOCK_CONVERT);
  gtk_dialog_add_action_widget (GTK_DIALOG (dialog), widget, OGMRIP_OPTIONS_EXTRACT);
  gtk_widget_show (widget);

  gtk_window_set_title (GTK_WINDOW (dialog), _("Options"));
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_PROPERTIES);

  widget = glade_xml_get_widget (xml, OGMRIP_GLADE_ROOT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), widget, TRUE, TRUE, 0);
  gtk_widget_show (widget);

  dialog->priv->bitrate_check = glade_xml_get_widget (xml, "bitrate-check");
  widget = glade_xml_get_widget (xml,"bitrate-vbox");
  g_signal_connect (dialog->priv->bitrate_check, "toggled", 
      G_CALLBACK (ogmrip_options_dialog_check_untoggled), widget);
  g_signal_connect_swapped (dialog->priv->bitrate_check, "toggled", 
      G_CALLBACK (ogmrip_options_dialog_bitrate_or_crop_toggled), dialog);

  dialog->priv->bitrate_spin = glade_xml_get_widget (xml, "bitrate-spin");

  dialog->priv->scale_check = glade_xml_get_widget (xml, "scale-check");
  widget = glade_xml_get_widget (xml,"bpp-hbox");
  g_signal_connect (dialog->priv->scale_check, "toggled", 
      G_CALLBACK (ogmrip_options_dialog_check_toggled), widget);
  widget = glade_xml_get_widget (xml,"scale-vbox");
  g_signal_connect (dialog->priv->scale_check, "toggled", 
      G_CALLBACK (ogmrip_options_dialog_check_untoggled), widget);

  dialog->priv->bpp_spin = glade_xml_get_widget (xml, "bpp-spin");

  dialog->priv->autoscale_button = glade_xml_get_widget (xml, "autoscale-button");
  g_signal_connect_swapped (dialog->priv->autoscale_button, "clicked", 
      G_CALLBACK (ogmrip_options_dialog_autoscale_clicked), dialog);

  dialog->priv->scale_user_hbox = glade_xml_get_widget (xml, "scale-user-hbox");
  dialog->priv->scale_width_spin = glade_xml_get_widget (xml, "scale-width-spin");
  dialog->priv->scale_height_spin = glade_xml_get_widget (xml, "scale-height-spin");

  dialog->priv->scale_store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);

  dialog->priv->scale_combo = glade_xml_get_widget (xml, "scale-combo");
  g_signal_connect_swapped (dialog->priv->scale_combo, "changed",
      G_CALLBACK (ogmrip_options_dialog_scale_combo_changed), dialog);

  for (i = 0; i < G_N_ELEMENTS (size); i++)
  {
    gtk_list_store_append (dialog->priv->scale_store, &iter);
    gtk_list_store_set (dialog->priv->scale_store, &iter, 0, _(size[i]), -1);
  }

  renderer = gtk_cell_renderer_text_new ();
  g_object_set (renderer, "xalign", 0.0, NULL);
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (dialog->priv->scale_combo), renderer, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (dialog->priv->scale_combo), renderer, "text", 0, NULL);

  renderer = gtk_cell_renderer_text_new ();
  g_object_set (renderer, "xalign", 1.0, "xpad", 4, NULL);
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (dialog->priv->scale_combo), renderer, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (dialog->priv->scale_combo), renderer, "text", 1, NULL);

  gtk_combo_box_set_model (GTK_COMBO_BOX (dialog->priv->scale_combo), GTK_TREE_MODEL (dialog->priv->scale_store));

  dialog->priv->crop_check = glade_xml_get_widget (xml, "crop-check");
  widget = glade_xml_get_widget (xml,"crop-hbox");
  g_signal_connect (dialog->priv->crop_check, "toggled", 
      G_CALLBACK (ogmrip_options_dialog_check_untoggled), widget);
  g_signal_connect_swapped (dialog->priv->crop_check, "toggled", 
      G_CALLBACK (ogmrip_options_dialog_bitrate_or_crop_toggled), dialog);

  dialog->priv->crop_left_label = glade_xml_get_widget (xml, "crop-left-label");
  dialog->priv->crop_right_label = glade_xml_get_widget (xml, "crop-right-label");
  dialog->priv->crop_top_label = glade_xml_get_widget (xml, "crop-top-label");
  dialog->priv->crop_bottom_label = glade_xml_get_widget (xml, "crop-bottom-label");

  widget = glade_xml_get_widget (xml, "crop-button");
  g_signal_connect_swapped (widget, "clicked", 
      G_CALLBACK (ogmrip_options_dialog_crop_clicked), dialog);

  widget = glade_xml_get_widget (xml, "autocrop-button");
  g_signal_connect_swapped (widget, "clicked", 
      G_CALLBACK (ogmrip_options_dialog_autocrop_clicked), dialog);

  dialog->priv->more_exp = glade_xml_get_widget (xml, "more-expander");
  dialog->priv->ntsc_frame = glade_xml_get_widget (xml, "ntsc-frame");

  dialog->priv->pullup_check = glade_xml_get_widget (xml, "pullup-check");
  dialog->priv->progressive_check = glade_xml_get_widget (xml, "progressive-check");

  dialog->priv->deint_combo = glade_xml_get_widget (xml, "deint-combo");
  gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->deint_combo), 0);

  dialog->priv->framestep_combo = glade_xml_get_widget (xml, "framestep-combo");
  gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->framestep_combo), 0);

  dialog->priv->cartoon_check = glade_xml_get_widget (xml, "cartoon-check");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->cartoon_check), FALSE);

  widget = glade_xml_get_widget (xml, "ccheck-button");
  g_signal_connect_swapped (widget, "clicked",
      G_CALLBACK (ogmrip_options_dialog_ccheck_clicked), dialog);

  gtk_widget_hide (widget);

  g_object_unref (xml);
}

static void
ogmrip_options_dialog_dispose (GObject *gobject)
{
  OGMRipOptionsDialog *dialog = OGMRIP_OPTIONS_DIALOG (gobject);

  if (dialog->priv->title)
    ogmdvd_title_unref (dialog->priv->title);
  dialog->priv->title = NULL;

  if (dialog->priv->scale_store)
    g_object_unref (dialog->priv->scale_store);
  dialog->priv->scale_store = NULL;

  G_OBJECT_CLASS (ogmrip_options_dialog_parent_class)->dispose (gobject);
}

GtkWidget *
ogmrip_options_dialog_new (void)
{
  return g_object_new (OGMRIP_TYPE_OPTIONS_DIALOG, NULL);
}

GtkWidget *
ogmrip_options_dialog_new_with_title (OGMDvdTitle *title)
{
  GtkWidget *dialog;

  g_return_val_if_fail (title != NULL, NULL);

  dialog = g_object_new (OGMRIP_TYPE_OPTIONS_DIALOG, NULL);
  ogmrip_options_dialog_set_title (OGMRIP_OPTIONS_DIALOG (dialog), title);

  return dialog;
}

OGMDvdTitle *
ogmrip_options_dialog_get_title (OGMRipOptionsDialog *dialog)
{
  g_return_val_if_fail (OGMRIP_IS_OPTIONS_DIALOG (dialog), NULL);

  return dialog->priv->title;
}

void
ogmrip_options_dialog_set_title (OGMRipOptionsDialog *dialog, OGMDvdTitle *title)
{
  g_return_if_fail (OGMRIP_IS_OPTIONS_DIALOG (dialog));
  g_return_if_fail (title != NULL);

  if (title != dialog->priv->title)
  {
    ogmdvd_title_ref (title);
    if (dialog->priv->title)
      ogmdvd_title_unref (dialog->priv->title);
    dialog->priv->title = title;

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->bitrate_check), TRUE);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->bitrate_spin), 800);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->crop_check), TRUE);
    ogmrip_options_dialog_crop_update (dialog);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->scale_check), TRUE);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->bpp_spin), 0.25);
    ogmrip_options_dialog_scale_update (dialog);
    gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->scale_combo), OGMRIP_SCALE_FULL);

    gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->deint_combo), OGMRIP_DEINT_NONE);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->progressive_check), FALSE);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->pullup_check), FALSE);

    gtk_expander_set_expanded (GTK_EXPANDER (dialog->priv->more_exp), FALSE);
    gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->framestep_combo), 0);

    if (ogmdvd_title_get_video_format (title) == OGMDVD_VIDEO_FORMAT_NTSC)
      gtk_widget_show (dialog->priv->ntsc_frame);
    else
      gtk_widget_hide (dialog->priv->ntsc_frame);
  }
}

gint
ogmrip_options_dialog_get_bitrate (OGMRipOptionsDialog *dialog, guint *bitrate)
{
  gint retval;

  g_return_val_if_fail (OGMRIP_IS_OPTIONS_DIALOG (dialog), -1);

  retval = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->bitrate_check)) ?
      OGMRIP_OPTIONS_AUTOMATIC : OGMRIP_OPTIONS_MANUAL;

  if (bitrate)
    *bitrate = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->priv->bitrate_spin)) * 1000;

  return retval;
}

gint
ogmrip_options_dialog_get_scale (OGMRipOptionsDialog *dialog, guint *width, guint *height)
{
  gint retval;

  g_return_val_if_fail (OGMRIP_IS_OPTIONS_DIALOG (dialog), -1);
  g_return_val_if_fail (width != NULL, -1);
  g_return_val_if_fail (height != NULL, -1);

  retval = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->scale_check)) ?
      OGMRIP_OPTIONS_AUTOMATIC : OGMRIP_OPTIONS_MANUAL;

  if (retval == OGMRIP_OPTIONS_MANUAL)
  {
    if (gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->priv->scale_combo)) == 0)
      retval = OGMRIP_OPTIONS_NONE;
    else
    {
      *width = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->priv->scale_width_spin));
      *height = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->priv->scale_height_spin));
    }
  }

  return retval;
}

gint
ogmrip_options_dialog_get_crop (OGMRipOptionsDialog *dialog, guint *x, guint *y, guint *width, guint *height)
{
  g_return_val_if_fail (OGMRIP_IS_OPTIONS_DIALOG (dialog), -1);
  g_return_val_if_fail (x != NULL, -1);
  g_return_val_if_fail (y != NULL, -1);
  g_return_val_if_fail (width != NULL, -1);
  g_return_val_if_fail (height != NULL, -1);

  return ogmrip_options_dialog_get_crop_internal (dialog, x, y, width, height);
}

gint
ogmrip_options_dialog_get_deinterlacer (OGMRipOptionsDialog *dialog)
{
  g_return_val_if_fail (OGMRIP_IS_OPTIONS_DIALOG (dialog), -1);

  return gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->priv->deint_combo));
}

gboolean
ogmrip_options_dialog_get_progressive (OGMRipOptionsDialog *dialog)
{
  g_return_val_if_fail (OGMRIP_IS_OPTIONS_DIALOG (dialog), FALSE);

  if (!GTK_WIDGET_VISIBLE (dialog->priv->ntsc_frame))
    return FALSE;

  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->progressive_check));
}

gboolean
ogmrip_options_dialog_get_pullup (OGMRipOptionsDialog *dialog)
{
  g_return_val_if_fail (OGMRIP_IS_OPTIONS_DIALOG (dialog), FALSE);

  if (!GTK_WIDGET_VISIBLE (dialog->priv->ntsc_frame))
    return FALSE;

  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->pullup_check));
}

gint
ogmrip_options_dialog_get_framestep (OGMRipOptionsDialog *dialog)
{
  gint framestep;

  g_return_val_if_fail (OGMRIP_IS_OPTIONS_DIALOG (dialog), -1);

  framestep = gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->priv->framestep_combo));

  return framestep + 1;
}

gdouble
ogmrip_options_dialog_get_bits_per_pixel (OGMRipOptionsDialog *dialog)
{
  g_return_val_if_fail (OGMRIP_IS_OPTIONS_DIALOG (dialog), -1.0);

  return gtk_spin_button_get_value (GTK_SPIN_BUTTON (dialog->priv->bpp_spin));
}

gboolean
ogmrip_options_dialog_get_cartoon (OGMRipOptionsDialog *dialog)
{
  g_return_val_if_fail (OGMRIP_IS_OPTIONS_DIALOG (dialog), FALSE);

  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->cartoon_check));
}

