/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ogmrip-mp4.h"
#include "ogmrip-mplayer.h"
#include "ogmrip-plugin.h"
#include "ogmrip-fs.h"

#include "ogmjob-exec.h"
#include "ogmjob-queue.h"

#include <string.h>
#include <glib/gi18n-lib.h>

static gint ogmrip_mp4_run (OGMJobSpawn *spawn);

static gchar **
ogmrip_mp4_command (OGMRipContainer *mp4)
{
  GPtrArray *argv;
  OGMRipVideo *video;
  const gchar *output, *filename;

  g_return_val_if_fail (OGMRIP_IS_MP4 (mp4), NULL);

  argv = ogmrip_mencoder_container_command (mp4);

  g_ptr_array_add (argv, g_strdup ("-of"));
  g_ptr_array_add (argv, g_strdup ("lavf"));
  g_ptr_array_add (argv, g_strdup ("-lavfopts"));
  g_ptr_array_add (argv, g_strdup ("format=mp4:i_certify_that_my_video_stream_does_not_use_b_frames"));

  output = ogmrip_container_get_output (mp4);
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));

  video = ogmrip_container_get_video (mp4);
  filename = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_ptr_array_add (argv, g_strdup (filename));

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static gchar **
ogmrip_copy_command (OGMRipContainer *container, const gchar *input, const gchar *output)
{
  GPtrArray *argv;
  gchar *real_output;

  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  if (output)
    real_output = g_strdup (output);
  else
  {
    const gchar *ext;

    output = ogmrip_container_get_output (container);
    g_return_val_if_fail (output != NULL, NULL);

    ext = ogmrip_fs_get_extension (input);

    if (ext && strcmp (ext, "idx") == 0)
      real_output = ogmrip_fs_set_extension (output, ext);
    else
      real_output = ogmrip_fs_set_extension (output, "sub");
  }

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("cp"));
  g_ptr_array_add (argv, g_strdup ("-f"));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, real_output);
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

G_DEFINE_TYPE (OGMRipMp4, ogmrip_mp4, OGMRIP_TYPE_CONTAINER)

static void
ogmrip_mp4_class_init (OGMRipMp4Class *klass)
{
  OGMJobSpawnClass *spawn_class;
  OGMRipContainerClass *container_class;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);
  container_class = OGMRIP_CONTAINER_CLASS (klass);

  spawn_class->run = ogmrip_mp4_run;
}

static void
ogmrip_mp4_init (OGMRipMp4 *mp4)
{
}

static void
ogmrip_mp4_foreach_subp (OGMRipContainer *avi, 
    OGMRipCodec *codec, guint demuxer, gint language, OGMJobContainer *queue)
{
  OGMJobSpawn *child;

  const gchar *filename;
  gchar *input, **argv = NULL;

  filename = ogmrip_codec_get_output (codec);

  if (demuxer == OGMRIP_SUBP_DEMUXER_VOBSUB)
    input = g_strconcat (filename, ".sub", NULL);
  else
    input = g_strdup (filename);

  argv = ogmrip_copy_command (avi, input, NULL);
  if (argv)
  {
    child = ogmjob_exec_newv (argv);
    ogmjob_container_add (OGMJOB_CONTAINER (queue), child);
    g_object_unref (child);
  }
  g_free (input);

  if (demuxer == OGMRIP_SUBP_DEMUXER_VOBSUB)
  {
    input = g_strconcat (filename, ".idx", NULL);

    argv = ogmrip_copy_command (avi, input, NULL);
    if (argv)
    {
      child = ogmjob_exec_newv (argv);
      ogmjob_container_add (OGMJOB_CONTAINER (queue), child);
      g_object_unref (child);
    }
    g_free (input);
  }
}

static gint
ogmrip_mp4_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *queue, *child;
  gchar **argv;
  gint result;

  result = OGMJOB_RESULT_ERROR;

  queue = ogmjob_queue_new ();
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), queue);
  g_object_unref (queue);

  argv = ogmrip_mp4_command (OGMRIP_CONTAINER (spawn));
  if (argv)
  {
    child = ogmjob_exec_newv (argv);
    ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_mencoder_container_watch, spawn, TRUE, FALSE, FALSE);
    ogmjob_container_add (OGMJOB_CONTAINER (queue), child);
    g_object_unref (child);

    ogmrip_container_foreach_subp (OGMRIP_CONTAINER (spawn), 
        (OGMRipContainerCodecFunc) ogmrip_mp4_foreach_subp, queue);

    result = OGMJOB_SPAWN_CLASS (ogmrip_mp4_parent_class)->run (spawn);
  }

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), queue);

  return result;
}

/**
 * ogmrip_mp4_new:
 * @output: The output file
 *
 * Creates a new #OGMRipMp4.
 *
 * Returns: The new #OGMRipMp4
 */
OGMJobSpawn *
ogmrip_mp4_new (const gchar *output)
{
  g_return_val_if_fail (output && *output, NULL);

  return g_object_new (OGMRIP_TYPE_MP4, "output", output, NULL);
}

static OGMRipPluginContainer mp4_plugin =
{
  NULL,
  G_TYPE_NONE,
  "mp4",
  N_("Mpeg-4 Media (MP4)"),
  FALSE,
  1,
  1,
  NULL
};

static gint formats[] = 
{
  OGMRIP_FORMAT_MPEG4,
  OGMRIP_FORMAT_H264,
  OGMRIP_FORMAT_THEORA,
  OGMRIP_FORMAT_AAC,
  OGMRIP_FORMAT_AC3,
  OGMRIP_FORMAT_DTS,
  OGMRIP_FORMAT_MP3,
  OGMRIP_FORMAT_PCM,
  OGMRIP_FORMAT_SRT,
  OGMRIP_FORMAT_VOBSUB,
  -1
};

OGMRipPluginContainer *
ogmrip_init_plugin (void)
{
  mp4_plugin.type = OGMRIP_TYPE_MP4;
  mp4_plugin.formats = formats;

  return &mp4_plugin;
}

