/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * SECTION:ogmrip-preferences
 * @title: Preferences
 * @include: ogmrip-preferences.h
 * @short_description: Functions to manage preferences
 */

#include "ogmrip-preferences.h"

#define OGMRIP_PREF_KEY "__ogmrip_preferences_key__"

typedef struct
{
  gchar *key;
  OGMRipPrefGetFunc get_func;
  OGMRipPrefSetFunc set_func;
} OGMRipPrefData;

static GConfClient *client;

/**
 * ogmrip_preferences_init:
 * @dir: directory to add to the list
 *
 * Initialize the preferences system and add add a directory to the list of
 * directories to watch.
 */
void
ogmrip_preferences_init (const gchar *dir)
{
  if (client)
    g_object_unref (client);

  client = gconf_client_get_default ();

  if (dir)
    gconf_client_add_dir (client, dir, GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
}

/**
 * ogmrip_preferences_uninit:
 *
 * Uninitialize the preferences system.
 */
void
ogmrip_preferences_uninit (void)
{
  if (client)
  {
    g_object_unref (client);
    client = NULL;
  }
}

/**
 * ogmrip_preferences_get:
 * @key: key to get
 *
 * Gets the value of a configuration key.
 *
 * Returns: newly-allocated #GConfValue, or NULL if unset and no default exists
 */
GConfValue *
ogmrip_preferences_get (const gchar *key)
{
  return gconf_client_get (client, key, NULL);
}

/**
 * ogmrip_preferences_get_bool:
 * @key: key to get
 * @def: default value
 *
 * Requests the boolean value stored at @key.
 *
 * Returns: the value of @key, or @def if no value was stored
 */
gboolean
ogmrip_preferences_get_bool (const gchar *key, gboolean def)
{
  GConfValue* val;

  val = gconf_client_get (client, key, NULL);
  if (!val)
    gconf_client_set_bool (client, key, def, NULL);
  else
  {
    def = gconf_value_get_bool (val);
    gconf_value_free (val);
  }

  return def;
}

/**
 * ogmrip_preferences_set_bool:
 * @key: key you want to set the value of
 * @value: new value of @key
 *
 * Change the value of @key to @val. Automatically creates the key if it didn't
 * exist before.
 */
void
ogmrip_preferences_set_bool (const gchar *key, gboolean value)
{
  gconf_client_set_bool (client, key, value, NULL);
}

/**
 * ogmrip_preferences_get_int:
 * @key: key to get
 * @def: default value
 *
 * Requests the integer value stored at @key.
 *
 * Returns: the value of @key, or @def if no value was stored
 */
gint
ogmrip_preferences_get_int (const gchar *key, gint def)
{
  GConfValue* val;

  val = gconf_client_get (client, key, NULL);
  if (!val)
    gconf_client_set_int (client, key, def, NULL);
  else
  {
    def = gconf_value_get_int (val);
    gconf_value_free (val);
  }

  return def;
}

/**
 * ogmrip_preferences_set_int:
 * @key: key you want to set the value of
 * @value: new value of @key
 *
 * Change the value of @key to @val. Automatically creates the key if it didn't
 * exist before.
 */
void
ogmrip_preferences_set_int (const gchar *key, gint value)
{
  gconf_client_set_int (client, key, value, NULL);
}

/**
 * ogmrip_preferences_get_string:
 * @key: key to get
 * @def: default value
 *
 * Requests the string value stored at @key.
 *
 * Returns: the value of @key, or @def if no value was stored
 */
gchar *
ogmrip_preferences_get_string (const gchar *key, const gchar *def)
{
  GConfValue* val;
  gchar *str;

  val = gconf_client_get (client, key, NULL);
  if (!val)
  {
    gconf_client_set_string (client, key, def, NULL);
    str = g_strdup (def);
  }
  else
  {
    str = g_strdup (gconf_value_get_string (val));
    gconf_value_free (val);
  }

  return str;
}

/**
 * ogmrip_preferences_set_string:
 * @key: key you want to set the value of
 * @value: new value of @key
 *
 * Change the value of @key to @val. Automatically creates the key if it didn't
 * exist before.
 */
void
ogmrip_preferences_set_string (const gchar *key, const gchar *value)
{
  gconf_client_set_string (client, key, value, NULL);
}

/**
 * ogmrip_preferences_get_filename:
 * @key: key to get
 * @def: default value
 *
 * Requests the filename value stored at @key.
 *
 * Returns: the value of @key, or @def if no value was stored
 */
gchar *
ogmrip_preferences_get_filename (const gchar *key, const gchar *def)
{
  gchar *utf8, *path;

  utf8 = ogmrip_preferences_get_string (key, def);
  path = g_filename_from_utf8 (utf8, -1, NULL, NULL, NULL);
  g_free (utf8);

  return path;
}

/**
 * ogmrip_preferences_set_filename:
 * @key: key you want to set the value of
 * @value: new value of @key
 *
 * Change the value of @key to @val. Automatically creates the key if it didn't
 * exist before.
 */
void
ogmrip_preferences_set_filename (const gchar *key, const gchar *value)
{
  gconf_client_set_string (client, key, value, NULL);
}

/*
 * GConf
 */

static void
gconf_client_notify_while_alive_disconnector (gpointer handler_id, GObject *gobject)
{
  gconf_client_notify_remove (client, GPOINTER_TO_UINT (handler_id));
}

static guint
gconf_client_notify_add_while_alive (GConfClient *client, const gchar *key,
    GConfClientNotifyFunc func, gpointer user_data, GFreeFunc destroy_notify, GtkWidget *widget)
{
  guint handler_id;

  handler_id = gconf_client_notify_add (client, key, func, user_data, destroy_notify, NULL);

  g_object_weak_ref (G_OBJECT (widget),
      gconf_client_notify_while_alive_disconnector,
      GUINT_TO_POINTER (handler_id));

  return handler_id;
}

typedef struct
{
  GFunc func;
  gpointer user_data;
} OGMRipPrefNotification;

static void
ogmrip_preferences_notified (GConfClient *client, guint cnxn_id, GConfEntry *entry, OGMRipPrefNotification *notification)
{
  (* notification->func) (entry->value, notification->user_data);
}

/**
 * ogmrip_preferences_add_notify_while_alive:
 * @key: where to listen for changes
 * @func: function to call when changes occur
 * @data: user data to pass to @func
 * @widget: the widget whose destruction signals the removal of this notification
 *
 * Request notification of changes to namespace_section. This includes the key
 * namespace_section itself, and any keys below it. The notification is removed
 * when @widget is destroyed.
 *
 * Returns: a connection ID for removing the notification
 */
guint
ogmrip_preferences_add_notify_while_alive (const gchar *key, GFunc func, gpointer data, gpointer widget)
{
  OGMRipPrefNotification *notification;
  guint handler_id;

  g_return_val_if_fail (key != NULL, 0);
  g_return_val_if_fail (func != NULL, 0);

  notification = g_new0 (OGMRipPrefNotification, 1);

  notification->func = func;
  notification->user_data = data;

  if (GTK_IS_WIDGET (widget))
    handler_id = gconf_client_notify_add_while_alive (client, key,
        (GConfClientNotifyFunc) ogmrip_preferences_notified, notification, (GFreeFunc) g_free, widget);
  else
    handler_id = gconf_client_notify_add (client, key,
        (GConfClientNotifyFunc) ogmrip_preferences_notified, notification, (GFreeFunc) g_free, NULL);

  if (!handler_id)
    g_free (notification);

  return handler_id;
}

/**
 * ogmrip_preferences_add_notify:
 * @key: where to listen for changes
 * @func: function to call when changes occur
 * @data: user data to pass to @func
 *
 * Request notification of changes to namespace_section. This includes the key
 * namespace_section itself, and any keys below it.
 *
 * Returns: a connection ID for removing the notification
 */
guint
ogmrip_preferences_add_notify (const gchar *key, GFunc func, gpointer data)
{
  return ogmrip_preferences_add_notify_while_alive (key, func ,data, NULL);
}

/*
 * Data
 */

static OGMRipPrefData *
ogmrip_preferences_data_new (const gchar *key, OGMRipPrefGetFunc get_func, OGMRipPrefSetFunc set_func)
{
  OGMRipPrefData *data;

  data = g_new0 (OGMRipPrefData, 1);
  data->key = g_strdup (key);
  data->get_func = get_func;
  data->set_func = set_func;

  return data;
}

static void
ogmrip_preferences_data_free (OGMRipPrefData *data)
{
  g_free (data->key);
  g_free (data);
}

/*
 * Generic
 */

static void
ogmrip_preferences_widget_changed (GtkWidget *widget, OGMRipPrefData *data)
{
  GConfValue *value;

  value = data->get_func (widget);
  if (value)
  {
    gconf_client_set (client, data->key, value, NULL);
    gconf_value_free (value);
  }
}

static void
ogmrip_preferences_widget_update (GConfValue *value, GtkWidget *widget)
{
  OGMRipPrefData *data;
  GConfValue *val;

  data = g_object_get_data (G_OBJECT (widget), OGMRIP_PREF_KEY);

  val = value ? value : gconf_client_get (client, data->key, NULL);

  g_signal_handlers_block_by_func (widget, ogmrip_preferences_widget_changed, data);

  if (!data->set_func (widget, val))
    ogmrip_preferences_widget_changed (widget, data);

  g_signal_handlers_unblock_by_func (widget, ogmrip_preferences_widget_changed, data);

  if (!value && val)
    gconf_value_free (val);
}

/*
 * Toggle
 */

/**
 * ogmrip_preferences_connect_toggle_custom:
 * @toggle: A #GtkToggleButton
 * @key: The associated key
 * @get_func: The function to get the state of @toggle
 * @set_func: The function to set the state of @toggle
 *
 * Connects @toggle to @key. When the state of @toggle changes, @get_func is
 * called and returns the new value for @key. Accordingly, when @key changes,
 * @set_func is called and sets the new state of @toggle.
 */
void
ogmrip_preferences_connect_toggle_custom (GtkWidget *toggle, const gchar *key,
      OGMRipPrefGetFunc get_func, OGMRipPrefSetFunc set_func)
{
  OGMRipPrefData *data;

  g_return_if_fail (GTK_IS_TOGGLE_BUTTON (toggle));
  g_return_if_fail (key != NULL);
  g_return_if_fail (get_func != NULL);
  g_return_if_fail (set_func != NULL);

  data = ogmrip_preferences_data_new (key, get_func, set_func);

  g_object_set_data_full (G_OBJECT (toggle),
      OGMRIP_PREF_KEY, data, (GDestroyNotify) ogmrip_preferences_data_free);

  ogmrip_preferences_add_notify_while_alive (key,
      (GFunc) ogmrip_preferences_widget_update, toggle, toggle);

  g_signal_connect (toggle, "toggled",
      G_CALLBACK (ogmrip_preferences_widget_changed), data);

  ogmrip_preferences_widget_update (NULL, toggle);
}

static GConfValue *
ogmrip_preferences_toggle_get_value (GtkWidget *toggle)
{
  GConfValue *value;

  value = gconf_value_new (GCONF_VALUE_BOOL);
  gconf_value_set_bool (value, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggle)));

  return value;
}

static gboolean
ogmrip_preferences_toggle_set_value (GtkWidget *toggle, GConfValue *value)
{
  if (!value)
    return FALSE;

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggle),
      gconf_value_get_bool (value));

  return TRUE;
}

/**
 * ogmrip_preferences_connect_toggle:
 * @toggle: A #GtkToggleButton
 * @key: The associated key
 *
 * Connects @toggle to @key. @key must be associated to a boolean value.
 * See ogmrip_preferences_connect_toggle_custom().
 */
void
ogmrip_preferences_connect_toggle (GtkWidget *toggle, const gchar *key)
{
  ogmrip_preferences_connect_toggle_custom (toggle, key,
      ogmrip_preferences_toggle_get_value, ogmrip_preferences_toggle_set_value);
}

/*
 * Combo
 */

/**
 * ogmrip_preferences_connect_combo_custom:
 * @combo: A #GtkComboBox
 * @key: The associated key
 * @get_func: The function to get the selected item of @combo
 * @set_func: The function to set the selected item of @combo
 *
 * Connects @combo to @key. When a new item in @combo is selected, @get_func is
 * called and returns the new value for @key. Accordingly, when @key changes,
 * @set_func is called and sets the new selected item of @combo.
 */
void
ogmrip_preferences_connect_combo_custom (GtkWidget *combo, const gchar *key,
      OGMRipPrefGetFunc get_func, OGMRipPrefSetFunc set_func)
{
  OGMRipPrefData *data;

  g_return_if_fail (GTK_IS_COMBO_BOX (combo));
  g_return_if_fail (key != NULL);
  g_return_if_fail (get_func != NULL);
  g_return_if_fail (set_func != NULL);

  data = ogmrip_preferences_data_new (key, get_func, set_func);

  g_object_set_data_full (G_OBJECT (combo),
      OGMRIP_PREF_KEY, data, (GDestroyNotify) ogmrip_preferences_data_free);

  ogmrip_preferences_add_notify_while_alive (key,
      (GFunc) ogmrip_preferences_widget_update, combo, combo);

  g_signal_connect (combo, "changed",
      G_CALLBACK (ogmrip_preferences_widget_changed), data);
  
  ogmrip_preferences_widget_update (NULL, combo);
}

static GConfValue *
ogmrip_preferences_combo_get_value (GtkWidget *combo)
{
  GConfValue *value;

  value = gconf_value_new (GCONF_VALUE_INT);
  gconf_value_set_int (value, gtk_combo_box_get_active (GTK_COMBO_BOX (combo)));
  
  return value;
}

static gboolean
ogmrip_preferences_combo_set_value (GtkWidget *combo, GConfValue *value)
{
  GtkTreeModel *model;
  gint high, val;

  if (!value)
    return FALSE;

  model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo));
  high = gtk_tree_model_iter_n_children (model, NULL) - 1;

  val = gconf_value_get_int (value);
  if (val < 0 || val > high)
    return FALSE;

  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), val);

  return TRUE;
}

/**
 * ogmrip_preferences_connect_combo:
 * @combo: A #GtkComboBox
 * @key: The associated key
 *
 * Connects @combo to @key. @key must be associated to an integer value.
 * See ogmrip_preferences_connect_combo_custom().
 */
void
ogmrip_preferences_connect_combo (GtkWidget *combo, const gchar *key)
{
  ogmrip_preferences_connect_combo_custom (combo, key,
      ogmrip_preferences_combo_get_value, ogmrip_preferences_combo_set_value);
}

/*
 * Spin
 */

/**
 * ogmrip_preferences_connect_spin_custom:
 * @spin: A #GtkSpinButton
 * @key: The associated key
 * @get_func: The function to get the value of @spin
 * @set_func: The function to set the value of @spin
 *
 * Connects @spin to @key. When a new value in @spin is selected, @get_func is
 * called and returns the new value for @key. Accordingly, when @key changes,
 * @set_func is called and sets the new value of @spin.
 */
void
ogmrip_preferences_connect_spin_custom (GtkWidget *spin, const gchar *key,
      OGMRipPrefGetFunc get_func, OGMRipPrefSetFunc set_func)
{
  OGMRipPrefData *data;

  g_return_if_fail (GTK_IS_SPIN_BUTTON (spin));
  g_return_if_fail (key != NULL);
  g_return_if_fail (get_func != NULL);
  g_return_if_fail (set_func != NULL);

  data = ogmrip_preferences_data_new (key, get_func, set_func);

  g_object_set_data_full (G_OBJECT (spin),
      OGMRIP_PREF_KEY, data, (GDestroyNotify) ogmrip_preferences_data_free);

  ogmrip_preferences_add_notify_while_alive (key,
      (GFunc) ogmrip_preferences_widget_update, spin, spin);

  g_signal_connect (spin, "value-changed",
      G_CALLBACK (ogmrip_preferences_widget_changed), data);
  
  ogmrip_preferences_widget_update (NULL, spin);
}

static GConfValue *
ogmrip_preferences_spin_get_value (GtkWidget *spin)
{
  GConfValue *value;

  value = gconf_value_new (GCONF_VALUE_INT);
  gconf_value_set_int (value, gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin)));
  
  return value;
}

static gboolean
ogmrip_preferences_spin_set_value (GtkWidget *widget, GConfValue *value)
{
  GtkSpinButton *spin;
  gint val;

  if (!value)
    return FALSE;

  spin = GTK_SPIN_BUTTON (widget);

  val = gconf_value_get_int (value);
  if (val < spin->adjustment->lower || val > spin->adjustment->upper)
    return FALSE;

  gtk_spin_button_set_value (spin, val);

  return TRUE;
}

/**
 * ogmrip_preferences_connect_spin:
 * @spin: A #GtkSpinButton
 * @key: The associated key
 *
 * Connects @spin to @key. @key must be associated to an integer value.
 * See ogmrip_preferences_connect_spin_custom().
 */
void
ogmrip_preferences_connect_spin (GtkWidget *spin, const gchar *key)
{
  ogmrip_preferences_connect_spin_custom (spin, key,
      ogmrip_preferences_spin_get_value,
      ogmrip_preferences_spin_set_value);
}

/*
 * Radio
 */

static void
ogmrip_preferences_radio_changed (GtkWidget *radio, OGMRipPrefData *data)
{
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio)))
  {
    GConfValue *value;

    value = data->get_func (radio);
    if (value)
    {
      gconf_client_set (client, data->key, value, NULL);
      gconf_value_free (value);
    }
  }
}

static void
ogmrip_preferences_radio_update (GConfValue *value, GtkWidget *widget)
{
  OGMRipPrefData *data;
  GConfValue *val;

  data = g_object_get_data (G_OBJECT (widget), OGMRIP_PREF_KEY);

  val = value ? value : gconf_client_get (client, data->key, NULL);
  g_signal_handlers_block_by_func (widget, ogmrip_preferences_widget_changed, data);

  if (!data->set_func (widget, val))
  {
    GSList *group;

    group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (widget));
    while (group)
    {
      ogmrip_preferences_radio_changed (group->data, data);
      group = group->next;
    }
  }

  g_signal_handlers_unblock_by_func (widget, ogmrip_preferences_widget_changed, data);

  if (!value && val)
    gconf_value_free (val);
}

/**
 * ogmrip_preferences_connect_radio_custom:
 * @radio: A #GtkRadioButton
 * @key: The associated key
 * @get_func: The function to get the selected @radio
 * @set_func: The function to set the selected @radio
 *
 * Connects @radio to @key. When @radio is selected, @get_func is
 * called and returns the new value for @key. Accordingly, when @key changes,
 * @set_func is called and selects @radio if necessary.
 */
void
ogmrip_preferences_connect_radio_custom (GtkWidget *radio, const gchar *key,
    OGMRipPrefGetFunc get_func, OGMRipPrefSetFunc set_func)
{
  OGMRipPrefData *data;

  g_return_if_fail (GTK_IS_RADIO_BUTTON (radio));
  g_return_if_fail (key != NULL);
  g_return_if_fail (get_func != NULL);
  g_return_if_fail (set_func != NULL);

  data = ogmrip_preferences_data_new (key, get_func, set_func);

  g_object_set_data_full (G_OBJECT (radio),
      OGMRIP_PREF_KEY, data, (GDestroyNotify) ogmrip_preferences_data_free);

  ogmrip_preferences_add_notify_while_alive (key,
      (GFunc) ogmrip_preferences_radio_update, radio, radio);

  g_signal_connect (radio, "toggled",
      G_CALLBACK (ogmrip_preferences_radio_changed), data);
  
  ogmrip_preferences_radio_update (NULL, radio);
}

static GConfValue *
ogmrip_preferences_radio_get_value (GtkWidget *radio)
{
  GConfValue *value;
  GSList *group;
  gint i;

  value = gconf_value_new (GCONF_VALUE_INT);

  group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));
  for (i = 0; group; group = group->next, i++)
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (group->data)))
      break;

  gconf_value_set_int (value, i);

  return value;
}

static gboolean
ogmrip_preferences_radio_set_value (GtkWidget *radio, GConfValue *value)
{
  GSList *group;
  gint i, active;

  if (!value)
    return FALSE;

  group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));
  active = gconf_value_get_int (value);

  for (i = 0; group; group = group->next, i++)
     if (i == active)
        break;

  if (!group)
    return FALSE;

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (group->data), TRUE);

  return TRUE;
}

/**
 * ogmrip_preferences_connect_radio:
 * @radio: A #GtkRadioButton
 * @key: The associated key
 *
 * Connects @radio to @key. @key must be associated to an integer value.
 * See ogmrip_preferences_connect_radio_custom().
 */
void
ogmrip_preferences_connect_radio (GtkWidget *radio, const gchar *key)
{
  ogmrip_preferences_connect_radio_custom (radio, key,
        ogmrip_preferences_radio_get_value,
        ogmrip_preferences_radio_set_value);
}

/*
 * Chooser
 */

/**
 * ogmrip_preferences_connect_chooser_custom:
 * @chooser: A #GtkChooserFile
 * @key: The associated key
 * @get_func: The function to get the selected file of @chooser
 * @set_func: The function to set the selected file of @chooser
 *
 * Connects @chooser to @key. When a new file is selected, @get_func is
 * called and returns the new value for @key. Accordingly, when @key changes,
 * @set_func is called and selects the file.
 */
void
ogmrip_preferences_connect_chooser_custom (GtkWidget *chooser, const gchar *key,
    OGMRipPrefGetFunc get_func, OGMRipPrefSetFunc set_func)
{
  OGMRipPrefData *data;

  g_return_if_fail (GTK_IS_FILE_CHOOSER (chooser));
  g_return_if_fail (key != NULL);
  g_return_if_fail (get_func != NULL);
  g_return_if_fail (set_func != NULL);

  data = ogmrip_preferences_data_new (key, get_func, set_func);

  g_object_set_data_full (G_OBJECT (chooser),
      OGMRIP_PREF_KEY, data, (GDestroyNotify) ogmrip_preferences_data_free);

  ogmrip_preferences_add_notify_while_alive (key,
      (GFunc) ogmrip_preferences_widget_update, chooser, chooser);

  g_signal_connect (chooser, "current-folder-changed",
      G_CALLBACK (ogmrip_preferences_widget_changed), data);
  
  ogmrip_preferences_widget_update (NULL, chooser);
}

static GConfValue *
ogmrip_preferences_chooser_get_value (GtkWidget *chooser)
{
  GConfValue *value;
  gchar *str;

  value = gconf_value_new (GCONF_VALUE_STRING);

  str = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
  if (!str)
    return NULL;

  gconf_value_set_string (value, str);
  g_free (str);
  
  return value;
}

static gboolean
ogmrip_preferences_chooser_set_value (GtkWidget *chooser, GConfValue *value)
{
  const gchar *new_path;
  gchar *old_path;

  if (!value)
    return FALSE;

  new_path = gconf_value_get_string (value);
  if (!g_file_test (new_path, G_FILE_TEST_IS_DIR))
    return FALSE;

  old_path = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER (chooser));
  if (!old_path || !g_str_equal (new_path, old_path))
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (chooser), new_path);

  return TRUE;
}

/**
 * ogmrip_preferences_connect_chooser:
 * @chooser: A #GtkFileChooser
 * @key: The associated key
 *
 * Connects @chooser to @key. @key must be associated to an integer value.
 * See ogmrip_preferences_connect_chooser_custom().
 */
void
ogmrip_preferences_connect_chooser (GtkWidget *chooser, const gchar *key)
{
  ogmrip_preferences_connect_chooser_custom (chooser, key,
      ogmrip_preferences_chooser_get_value,
      ogmrip_preferences_chooser_set_value);
}

