/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

/**
 * SECTION:ogmrip-helper
 * @title: Helper
 * @include: ogmrip-source-chooser.h
 * @short_description: A list of helper functions
 */

#include "ogmrip-helper.h"

#include <stdlib.h>
#include <locale.h>

/**
 * g_get_locale:
 * @category: A pointer to store the type of the chooser
 *
 * Returns the active source and its type.
 *
 * Returns: The active #OGMRipSource
 */
gchar *
g_get_locale (gint category)
{
  gchar *locale;

  locale = setlocale (category, NULL);
  if (locale)
    return g_strdup (locale);

  return NULL;
}

/**
 * gconf_client_get_int_default:
 * @client: A #GConfClient
 * @key: A gconf key
 * @def: A fallback value
 *
 * Returns the integer value associated with @key, or @def if no such data
 * exists.
 *
 * Returns: An integer
 */
gint
gconf_client_get_int_default (GConfClient *client, const gchar *key, gint def)
{
  GConfValue* val;

  val = gconf_client_get (client, key, NULL);
  if (!val)
    gconf_client_set_int (client, key, def, NULL);
  else
  {
    def = gconf_value_get_int (val);
    gconf_value_free (val);
  }

  return def;
}

/**
 * gconf_client_get_bool_default:
 * @client: A #GConfClient
 * @key: A gconf key
 * @def: A fallback value
 *
 * Returns the boolean value associated with @key, or @def if no such data
 * exists.
 *
 * Returns: A boolean
 */
gboolean
gconf_client_get_bool_default (GConfClient *client, const gchar *key, gboolean def)
{
  GConfValue* val;

  val = gconf_client_get (client, key, NULL);
  if (!val)
    gconf_client_set_bool (client, key, def, NULL);
  else
  {
    def = gconf_value_get_bool (val);
    gconf_value_free (val);
  }

  return def;
}

/**
 * gconf_client_get_string_default:
 * @client: A #GConfClient
 * @key: A gconf key
 * @def: A fallback value
 *
 * Returns the string associated with @key, or @def if no such data exists.
 *
 * Returns: A newly allocated string
 */
gchar *
gconf_client_get_string_default (GConfClient *client, const gchar *key, const gchar *def)
{
  GConfValue* val;
  gchar *str;

  val = gconf_client_get (client, key, NULL);
  if (!val)
  {
    gconf_client_set_string (client, key, def, NULL);
    str = g_strdup (def);
  }
  else
  {
    str = g_strdup (gconf_value_get_string (val));
    gconf_value_free (val);
  }

  return str;
}

/**
 * gconf_client_get_filename_default:
 * @client: A #GConfClient
 * @key: A gconf key
 * @def: A fallback value
 *
 * Returns the string associated with @key in the encoding GLib uses for
 * filenames, or @def if no such data exists.
 *
 * Returns: A newly allocated string
 */
gchar *
gconf_client_get_filename_default (GConfClient *client, const gchar *key, const gchar *def)
{
  gchar *utf8, *path;

  utf8 = gconf_client_get_string_default (client, key, def);
  path = g_filename_from_utf8 (utf8, -1, NULL, NULL, NULL);
  g_free (utf8);

  return path;
}

/**
 * gtk_window_set_parent:
 * @window: A #GtkWindow
 * @parent: The parent window
 *
 * Sets a parent window for a window. This is equivalent to calling
 * gtk_window_set_transient_for(),
 * gtk_window_set_position(),
 * gtk_window_set_gravity(), and
 * gtk_window_set_destroy_with_parent() on @window.
 */
void
gtk_window_set_parent (GtkWindow *window, GtkWindow *parent)
{
  g_return_if_fail (GTK_IS_WINDOW (window));
  g_return_if_fail (GTK_IS_WINDOW (parent));
  g_return_if_fail (window != parent);

  gtk_window_set_transient_for (window, parent);
  gtk_window_set_position (window, GTK_WIN_POS_CENTER_ON_PARENT);
  gtk_window_set_gravity (window, GDK_GRAVITY_CENTER);
  gtk_window_set_destroy_with_parent (window, TRUE);
}

/**
 * gtk_window_set_icon_from_stock:
 * @window: A #GtkWindow
 * @stock_id: the name of the stock item
 *
 * Sets the icon of @window from a stock item.
 */
void
gtk_window_set_icon_from_stock (GtkWindow *window, const gchar *stock_id)
{
  GdkPixbuf *pixbuf;

  g_return_if_fail (GTK_IS_WINDOW (window));
  g_return_if_fail (stock_id && *stock_id);
  
  pixbuf = gtk_widget_render_icon (GTK_WIDGET (window), stock_id, GTK_ICON_SIZE_DIALOG, NULL);
  gtk_window_set_icon (window, pixbuf);
  gdk_pixbuf_unref (pixbuf);
}

/**
 * gtk_button_new_with_image:
 * @text: The text you want the #GtkLabel to hold
 * @stock_id: the name of the stock item
 *
 * Creates a new #GtkButton containing the given text and the image from a stock item.
 *
 * Returns: A new #GtkButton
 */
GtkWidget *
gtk_button_new_with_image (const gchar *text, const gchar *stock_id)
{
  GtkWidget *button, *alignment, *box, *image, *label;

  g_return_val_if_fail (text != NULL, NULL);
  g_return_val_if_fail (stock_id != NULL, NULL);

  button = gtk_button_new ();

  alignment = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (button), alignment);
  gtk_widget_show (alignment);

  box = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (alignment), box);
  gtk_widget_show (box);

  image = gtk_image_new_from_stock (stock_id, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start (GTK_BOX (box), image, FALSE, FALSE, 0);
  gtk_widget_show (image);

  label = gtk_label_new (text);
  gtk_label_set_use_underline (GTK_LABEL (label), TRUE);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  return button;
}

/**
 * gtk_radio_button_get_active:
 * @radio: Any #GtkRadioButton of the group
 *
 * Returns the index of the active #GtkRadioButton.
 *
 * Returns: An integer, or -1
 */
gint
gtk_radio_button_get_active (GtkRadioButton *radio)
{
  GSList *link;
  gint i;

  g_return_val_if_fail (GTK_IS_RADIO_BUTTON (radio), -1);

  link = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));

  for (i = g_slist_length (link) - 1; link; i--, link = link->next)
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (link->data)))
      return i;

  return -1;
}

/**
 * gtk_radio_button_set_active:
 * @radio: Any #GtkRadioButton of the group
 * @index: The index of the active item
 *
 * Sets the active item of the radio group.
 */
void
gtk_radio_button_set_active (GtkRadioButton *radio, guint index)
{
  GSList *link;
  guint i;

  g_return_if_fail (GTK_IS_RADIO_BUTTON (radio));

  link = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));
  for (i = g_slist_length (link) - 1;  link; i--, link = link->next)
    if (i == index)
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (link->data), TRUE);
}

/**
 * gtk_tree_model_iter_prev:
 * @tree_model: A #GtkTreeModel
 * @iter: The #GtkTreeIter
 *
 * Sets @iter to point to the node preceding it at the current level.
 * If there is no previous @iter, %FALSE is returned and @iter is set to be invalid.
 *
 * Returns: %TRUE, if @iter has been changed to the previous node
 */
gboolean
gtk_tree_model_iter_prev (GtkTreeModel *tree_model, GtkTreeIter *iter)
{
  gboolean retval = FALSE;
  GtkTreePath *path;

  path = gtk_tree_model_get_path (tree_model, iter);
  if (path)
  {
    if (gtk_tree_path_prev (path))
      retval = gtk_tree_model_get_iter (tree_model, iter, path);
    gtk_tree_path_free (path);
  }

  return retval;
}

/**
 * gtk_label_set_int:
 * @label: A #GtkLabel
 * @value: An integer
 *
 * Sets the value of a #GtkLabel widget.
 */
void
gtk_label_set_int (GtkLabel *label, gint value)
{
  gchar *text;

  g_return_if_fail (GTK_IS_LABEL (label));

  text = g_strdup_printf ("%d", value);
  gtk_label_set_text (label, text);
  g_free (text);
}

/**
 * gtk_label_get_int:
 * @label: A #GtkLabel
 *
 * Gets the value of the @label represented as an integer.
 *
 * Returns: The value of the @label widget
 */
gint
gtk_label_get_int (GtkLabel *label)
{
  const gchar *text;

  g_return_val_if_fail (GTK_IS_LABEL (label), G_MININT);

  text = gtk_label_get_text (label);
  
  return atoi (text);
}

/**
 * gtk_box_get_nth_child:
 * @box: A #GtkBox
 * @n: The index of the desired child
 *
 * Returns the @n'th item in @box.
 *
 * Returns: The nth #GtkWidget, or NULL
 */
GtkWidget *
gtk_box_get_nth_child (GtkBox *box, gint n)
{
  GList *link;

  g_return_val_if_fail (GTK_IS_BOX (box), NULL);

  if (n < 0)
    link = g_list_last (box->children);
  else
    link = g_list_nth (box->children, n);

  if (!link)
    return NULL;

  return ((GtkBoxChild *)(link->data))->widget;
}

/**
 * ogmrip_statusbar_push:
 * @statusbar: A #GtkStatusbar
 * @text: The message to add to the statusbar
 *
 * Pushes a new message onto a statusbar's stack using the default
 * context identifier.
 */
void
ogmrip_statusbar_push (GtkStatusbar *statusbar, const gchar *text)
{
  guint id;

  g_return_if_fail (GTK_IS_STATUSBAR (statusbar));
  g_return_if_fail (text != NULL);

  id = gtk_statusbar_get_context_id (statusbar, "__menu_hint__");
  gtk_statusbar_push (statusbar, id, text);
}

/**
 * ogmrip_statusbar_pop:
 * @statusbar: A #GtkStatusbar
 *
 * Removes the message at the top of a GtkStatusBar's stack using the
 * default context identifier.
 */
void
ogmrip_statusbar_pop (GtkStatusbar *statusbar)
{
  guint id;

  g_return_if_fail (GTK_IS_STATUSBAR (statusbar));

  id = gtk_statusbar_get_context_id (statusbar, "__menu_hint__");
  gtk_statusbar_pop (statusbar, id);
}

GQuark
ogmrip_error_quark (void)
{
  static GQuark quark = 0;

  if (quark == 0)
    quark = g_quark_from_static_string ("ogmrip-error-quark");

  return quark;
}

/**
 * ogmrip_message_dialog_newv:
 * @parent: A #GtkWindow
 * @type: A #GtkMessageType
 * @format: printf()-style format string, or NULL
 * @args: Arguments for @format
 *
 * Creates a new message dialog, which is a simple dialog with an icon
 * indicating the dialog type (error, warning, etc.) and some text the user may
 * want to see.
 *
 * Returns: A new #GtkMessageDialog
 */
GtkWidget *
ogmrip_message_dialog_newv (GtkWindow *parent, GtkMessageType type, const gchar *format, va_list args)
{
  GtkWidget *dialog = NULL;
  GtkButtonsType buttons = GTK_BUTTONS_NONE;
  const gchar *stock_id = NULL;
  gchar *message;

  g_return_val_if_fail (parent == NULL || GTK_IS_WINDOW (parent), NULL);

  switch (type)
  {
    case GTK_MESSAGE_ERROR:
      buttons = GTK_BUTTONS_CLOSE;
      stock_id = GTK_STOCK_DIALOG_ERROR;
      break;
    case GTK_MESSAGE_QUESTION:
      buttons = GTK_BUTTONS_YES_NO;
      stock_id = GTK_STOCK_DIALOG_QUESTION;
      break;
    case GTK_MESSAGE_INFO:
      buttons = GTK_BUTTONS_CLOSE;
      stock_id = GTK_STOCK_DIALOG_INFO;
      break;
    case GTK_MESSAGE_WARNING:
      buttons = GTK_BUTTONS_CLOSE;
      stock_id = GTK_STOCK_DIALOG_WARNING;
      break;
    default:
      break;
  }

  message = g_strdup_vprintf (format, args);
  dialog = gtk_message_dialog_new (parent,
      GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT, type, buttons, message);
  g_free (message);

  if (dialog)
  {
    if (stock_id)
      gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), stock_id);

    gtk_window_set_gravity (GTK_WINDOW (dialog), GDK_GRAVITY_CENTER);
    gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER_ON_PARENT);
  }

  return dialog;
}

/**
 * ogmrip_message_dialog_new:
 * @parent: A #GtkWindow
 * @type: A #GtkMessageType
 * @format: printf()-style format string, or NULL
 * @...: Arguments for @format
 *
 * Creates a new message dialog, which is a simple dialog with an icon
 * indicating the dialog type (error, warning, etc.) and some text the user may
 * want to see.
 *
 * Returns: A new #GtkMessageDialog
 */
GtkWidget *
ogmrip_message_dialog_new (GtkWindow *parent, GtkMessageType type, const gchar *format, ...)
{
  GtkWidget *dialog;
  va_list args;

  g_return_val_if_fail (parent == NULL || GTK_IS_WINDOW (parent), NULL);

  va_start (args, format);
  dialog = ogmrip_message_dialog_newv (parent, type, format, args);
  va_end (args);

  return dialog;
}

/**
 * ogmrip_message_dialog:
 * @parent: A #GtkWindow
 * @type: A #GtkMessageType
 * @format: printf()-style format string, or NULL
 * @...: Arguments for @format
 *
 * Creates and displays a new message dialog, which is a simple dialog with an
 * icon indicating the dialog type (error, warning, etc.) and some text the user
 * may want to see.
 *
 * Returns: The response ID
 */
gint
ogmrip_message_dialog (GtkWindow *parent, GtkMessageType type, const gchar *format, ...)
{
  GtkWidget *dialog;
  va_list args;

  g_return_val_if_fail (parent == NULL || GTK_IS_WINDOW (parent), GTK_RESPONSE_NONE);

  va_start (args, format);
  dialog = ogmrip_message_dialog_newv (parent, type, format, args);
  va_end (args);

  if (dialog)
  {
    gint response;

    gtk_label_set_use_markup (GTK_LABEL (GTK_MESSAGE_DIALOG (dialog)->label), TRUE);
    response = gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);

    return response;
  }

  return GTK_RESPONSE_NONE;
}

