/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2009 Olivier Rolland <billl@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <ogmrip-container.h>
#include <ogmrip-mplayer.h>
#include <ogmrip-plugin.h>
#include <ogmrip-fs.h>

#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#define OGMRIP_TYPE_OGGZ          (ogmrip_oggz_get_type ())
#define OGMRIP_OGGZ(obj)          (G_TYPE_CHECK_INSTANCE_CAST ((obj), OGMRIP_TYPE_OGGZ, OGMRipOggz))
#define OGMRIP_OGGZ_CLASS(klass)  (G_TYPE_CHECK_CLASS_CAST ((klass), OGMRIP_TYPE_OGGZ, OGMRipOggzClass))
#define OGMRIP_IS_OGGZ(obj)       (G_TYPE_CHECK_INSTANCE_TYPE ((obj), OGMRIP_TYPE_OGGZ))
#define OGMRIP_IS_OGGZ_CLASS(obj) (G_TYPE_CHECK_CLASS_TYPE ((klass), OGMRIP_TYPE_OGGZ))

typedef struct _OGMRipOggz      OGMRipOggz;
typedef struct _OGMRipOggzClass OGMRipOggzClass;

struct _OGMRipOggz
{
  OGMRipContainer parent_instance;
};

struct _OGMRipOggzClass
{
  OGMRipContainerClass parent_class;
};

static GType ogmrip_oggz_get_type (void);
static gint  ogmrip_oggz_run      (OGMJobSpawn *spawn);

static void
ogmrip_oggz_append_audio_file (OGMRipContainer *oggz, const gchar *filename, GPtrArray *argv)
{
  struct stat buf;

  if (g_stat (filename, &buf) == 0 && buf.st_size > 0)
    g_ptr_array_add (argv, g_strdup (filename));
}

static void
ogmrip_oggz_foreach_audio (OGMRipContainer *oggz, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  const gchar *input;

  input = ogmrip_codec_get_output (codec);
  ogmrip_oggz_append_audio_file (oggz, input, argv);
}

static void
ogmrip_oggz_foreach_audio_file (OGMRipContainer *oggz, OGMRipFile *file, GPtrArray *argv)
{
  gchar *filename;

  filename = ogmrip_file_get_filename (file);
  if (filename && ogmrip_file_get_type (file) == OGMRIP_FILE_TYPE_AUDIO)
    ogmrip_oggz_append_audio_file (oggz, filename, argv);
  g_free (filename);
}

static gchar **
ogmrip_oggz_command (OGMRipContainer *oggz)
{
  GPtrArray *argv;
  OGMRipVideoCodec *video;
  const gchar *filename;

  g_return_val_if_fail (OGMRIP_IS_OGGZ (oggz), NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("oggzmerge"));

  filename = ogmrip_container_get_output (oggz);
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (filename));

  video = ogmrip_container_get_video (oggz);

  filename = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_ptr_array_add (argv, g_strdup (filename));

  ogmrip_container_foreach_audio (oggz, 
      (OGMRipContainerCodecFunc) ogmrip_oggz_foreach_audio, argv);
  ogmrip_container_foreach_file (oggz, 
      (OGMRipContainerFileFunc) ogmrip_oggz_foreach_audio_file, argv);

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

G_DEFINE_TYPE (OGMRipOggz, ogmrip_oggz, OGMRIP_TYPE_CONTAINER)

static void
ogmrip_oggz_class_init (OGMRipOggzClass *klass)
{
  OGMJobSpawnClass *spawn_class;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);
  spawn_class->run = ogmrip_oggz_run;
}

static void
ogmrip_oggz_init (OGMRipOggz *oggz)
{
}

static gint
ogmrip_oggz_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *child;
  gchar **argv;
  gint result;

  argv = ogmrip_oggz_command (OGMRIP_CONTAINER (spawn));
  if (!argv)
    return OGMJOB_RESULT_ERROR;

  child = ogmjob_exec_newv (argv);
  ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) NULL, spawn, TRUE, FALSE, FALSE);
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), child);
  g_object_unref (child);

  result = OGMJOB_SPAWN_CLASS (ogmrip_oggz_parent_class)->run (spawn);

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), child);

  return result;
}

static OGMRipContainerPlugin oggz_plugin =
{
  NULL,
  G_TYPE_NONE,
  "ogg",
  N_("Ogg Container (OGG)"),
  TRUE,
  FALSE,
  G_MAXINT,
  0,
  NULL
};

static gint formats[] = 
{
  OGMRIP_FORMAT_VORBIS,
  OGMRIP_FORMAT_THEORA,
  -1
};

OGMRipContainerPlugin *
ogmrip_init_plugin (GError **error)
{
  gchar *fullname;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif /* ENABLE_NLS */

  fullname = g_find_program_in_path ("oggzmerge");
  if (!fullname)
    return NULL;
  g_free (fullname);

  oggz_plugin.type = OGMRIP_TYPE_OGGZ;
  oggz_plugin.formats = formats;

  return &oggz_plugin;
}

