/* ============================================================
 * File  : streambrowser.h
 * Author: Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date  : 2003-12-22
 * Description : av stream repository handling
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// development snapshot, many todo's

#ifndef STORAGE_H
#define STORAGE_H

#include <iostream>
#include <qsqldatabase.h>
#include <qwidget.h>
#include <qdatetime.h>
#include <qprocess.h>
#include <qhttp.h>
#include <qvaluevector.h>

#include "storagehandlers.h"

/*
  Absolutely NOT thread safe

  Storage constraints:
    1) url is unique
    2) (folder, name) is unique

    --> on insertRecord of unique url with non-unique (folder,name),
        name is changed to url (variable name is passed by reference).

    bool loadList()
      clears memory list, reloads records from resource

    bool storeList()
      clear current selected storage and insert records from memory list
      -> selectDbStorage + loadList + selectFileStorage + storeList copies db records to file.
      -> In case of file storage: selectFileStorage + removeRecord + storeList compacts file
*/

class Storage : public QObject
{
    Q_OBJECT

  public:
    enum AccessType
    {
      none = 0,
      database,
      file,
      web
    };

    // copied in GenStorage class
    enum EventType
    {
      selected = 0,
      loaded,
      saved
    };

    Storage(QString table = "", QString name = "",
            int key_pri = 2, int key_mul1 = 0, int key_mul2 = 1);
    ~Storage();

    // select storage resource (without reading records)
    bool selectDefaultDb  (int ident); // select db passed in constructor
    bool selectDbStorage  (int ident, QString name, QString dbDriver, QString hostName,
                           unsigned int port, QString dbName, QString table,
                           QString login, QString password);
    bool selectFileStorage(int ident, QString name, QString fileName);
    void selectWebStorage (int ident, QString name, QString url, QString login = "", QString password = "");

    QString    getStorageDescription();
    AccessType getAccessType();
    QString    getStorageName();
    bool       getSynchronized();
    bool       storageReadOnly();
    QString    getLastError();
    QString    getDefaultName()   { return defaultName;      }

    // list manipulation
    bool loadList(int ident, QString& error);
    bool storeList(int ident, QString& error); // copy from "loaded" into "selected" (not yet loaded) storage

    // record manipulation
    bool insertRecord(int ident, ValueList& values, QString& error);
    bool updateRecord(int ident, ValueList& oldValues, ValueList& values, QString& error);
    bool removeRecord(int ident, ValueList values, QString& error);

    // check functions
    bool itemKeyExists(ValueList& values);

    // list walk
    void resetRecordList();
    bool getNextRecord(ValueList& values);

  signals:
    void storageEvent(int ident, int eventType, bool error); // ident=0: internal use
    void recordInserted(ChangedRecord* rec);
    void recordUpdated(ChangedRecord* rec);
    void recordRemoved(ChangedRecord* rec);

  private:
    GenStorage *currentStorage, *oldStorage;
    QString defaultTable;    //
    QString defaultName;     //
    int key_pri, key_mul1, key_mul2;

    void linkNewStorage(GenStorage *storage);
};


#endif /* STORAGE_H */
