/* Acyclic.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.simple;

import org.grinvin.GraphBundleView;
import org.grinvin.invariants.BooleanValue;
import org.grinvin.invariants.computers.AbstractInvariantComputer;

/**
 * Computes whether the graph is acyclic. This uses the following theorem:
 * a graph with <i>n</i> vertices and <i>k</i> components is acyclic if and
 * only if it has <i>n-k</i> edges.
 */
public class Acyclic extends AbstractInvariantComputer {
    
    public BooleanValue compute(GraphBundleView bundle) {
        int nrOfVertices = bundle.getGraph().getNumberOfVertices();
        int nrOfEdges = bundle.getGraph().getNumberOfEdges();
        
        //first count components
        int[][] adjlist = bundle.adjacencyList();
        //boolean array: true if component containing this vertex has been counted
        boolean[] vertices = new boolean[nrOfVertices];
        int nrOfComponents = 0;
        
        for(int i=0; i < nrOfVertices; i++) {
            if(!vertices[i]){
                vertices[i]=true;
                nrOfComponents++;
                checkComponent(adjlist, vertices, i);
            }
        }
        
        //check if graph is acyclic        
        return new BooleanValue(nrOfEdges == nrOfVertices - nrOfComponents, this);
    }
    
    private void checkComponent(int[][] adjacencyList, boolean[] vertices, int currentVertex) {
        for(int i=0; i < adjacencyList[currentVertex].length; i++){
            if(!vertices[adjacencyList[currentVertex][i]]){
                vertices[adjacencyList[currentVertex][i]]=true;
                checkComponent(adjacencyList, vertices, adjacencyList[currentVertex][i]);
            }
        }
    }
    
    public String getInvariantId() {
        return "org.grinvin.invariants.Acyclic";
    }
    
}
