/* AbstractListAction.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list.actions;

import be.ugent.caagt.swirl.actions.SimpleAction;
import be.ugent.caagt.swirl.lists.TypedListModel;

import java.util.ArrayList;
import java.util.List;
import java.util.ResourceBundle;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

/**
 * Abstract implementation for list actions. These actions work on a <code>TypedListModel</code>
 * and use a <code>ListSelectionModel</code> to know on which elements to apply the action. The
 * <code>TypedListModel</code> has to contain elements of type <code>E</code>. This action will
 * listen to the <code>ListSelectionModel</code> to know if it's enabled.
 */
public abstract class AbstractListAction<E> extends SimpleAction implements ListSelectionListener {
    
    //
    protected TypedListModel<E> list;
    
    //
    protected ListSelectionModel selection;
    
    //
    public AbstractListAction(TypedListModel<E> list, ListSelectionModel selection, ResourceBundle bundle, String resourceID) {
        super(bundle, resourceID, null);
        this.list = list;
        this.selection = selection;
        selection.addListSelectionListener(this);
        setEnabled();
    }
    
    /**
     * Returns the selected elements in the <code>TypedListModel</code> based on the <code>ListSelectionModel</code>.
     */
    protected List<E> getSelectedElements(){
        List<E> selected = new ArrayList<E>();
        int iMin = selection.getMinSelectionIndex();
        int iMax = selection.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin)
            return selected; // no selection
        
        for (int i = iMin; i <= iMax; i++)
            if (selection.isSelectedIndex(i))
                selected.add(list.get(i));
        
        return selected;
    }
    
    //
    public void valueChanged(ListSelectionEvent e){
        setEnabled();
    }
    
    /**
     * Decides the enabled status of this action, based on the selection. This method is called every
     * time the selection changes. By default the action will be disabled when there are no elements 
     * selected and enabled in all other cases. This method can be overridden to alter this behaviour.
     */
    public void setEnabled(){
        setEnabled(getSelectedElements().size()>0);
    }
}
