/* AbstractInvariant.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants;

import org.grinvin.expr.Expression;

/**
 * Abstract implementation of an {@link org.grinvin.invariants.Invariant}.
 */
public abstract class AbstractInvariant implements InvariantExpressionNode {
    
    /**
     * Universally unique identifier for this invariant.
     */
    public abstract String getId ();
    
    /**
     * Returns the localized name of the invariant.
     */
    public abstract String getName ();

    /**
     * Returns null. An invariant cannot have other invariant nodes as
     * children.
     */
    public Iterable<InvariantNode> getChildren() {
        return null;
    }
    
    /**
     * Returns the class to which all values of this invariant belong.
     * This class must be an extension of {@link InvariantValue}.
     */
    public abstract Class<? extends InvariantValue> getType ();
    
    
    //
    public void accept(InvariantNode.Visitor visitor) {
        visitor.visitInvariant(this);
    }
    
    //
    public void accept(Expression.Visitor visitor) {
        visitor.visitInvariant(this);
    }
    
    /**
     * Two invariants are equal if and only if their identifiers are
     * the same.
     */
    @Override
    public boolean equals (Object obj) {
        if (this == obj) {
            return true;
        } else if (obj instanceof Invariant) {
            return ((Invariant)obj).getId ().equals (getId());
        } else {
            return false;
        }
    }
    
    /**
     * The hash value of an invariant is the hash value of its identifier.
     */
    @Override
    public int hashCode () {
        return getId().hashCode ();
    }
    
    /**
     * Return the localized name of the invariant.
     */
    @Override
    public abstract String toString (); // TODO: change InvariantList to use getName
      
}
