/* BullsEye.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred.guides;

import java.awt.Graphics2D;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;

import org.grinvin.gred.guides.AbstractGuides.ButtonIcon;

/**
 * Guides consisting of concentric circles divided into sectors. The number of circles
 * and sectors is configurable.
 */
public class BullsEye extends AbstractGuides {
    
    //
    private int numberOfCircles;
    
    //
    private int numberOfSectors;
    
    /**
     * Set the number of divisions used by this grid.
     * @param numberOfCircles Number of circles in the bull's eye.
     * @param numberOfSectors
     *
     */
    public void setNumberOfDivisions (int numberOfCircles, int numberOfSectors) {
        if (numberOfCircles < 1 || numberOfSectors < 2)
            throw new IllegalArgumentException ("Illegal number of grid divisions");
        this.numberOfCircles = numberOfCircles;
        this.numberOfSectors = numberOfSectors;
    }
    
    /**
     * Create a new object of this type.
     * @param numberOfCircles Number of rows in the grid, i.e., one less than the
     * number of horizontal grid lines.
     * @param numberOfSectors Number of columns in the grid, i.e., one less than the
     * number of vertical grid lines.
     */
    public BullsEye (int numberOfCircles, int numberOfSectors) {
        this.numberOfCircles = numberOfCircles;
        this.numberOfSectors = numberOfSectors;
        this.buttonIcon = new ButtonIcon ();
    }
    
    
    /**
     * Paint the bull's eye.
     */
    public void paint (Graphics2D g2, double scale, double maxX, double maxY) {
        g2.setStroke (GUIDE_STROKE);
        g2.setPaint (GUIDE_COLOR);
        
        double halfSize = scale + 2;
        
        // horizontal lines
        Line2D line = new Line2D.Double (); // reusable line
        Ellipse2D circle = new Ellipse2D.Double (); // reusable circle
        
        // circles
        for (int i = 1; i <= numberOfCircles; i++) {
            double r = 1.0*i*scale/numberOfCircles;
            circle.setFrame (-r,-r,2*r,2*r);
            g2.draw (circle);
        }
        
        // radii
        for (int i = 0; i <= numberOfSectors; i++) {
            double a = Math.PI * 2.0 * i/ numberOfSectors;
            line.setLine (0, 0, halfSize*Math.cos (a), halfSize*Math.sin (a));
            g2.draw (line);
        }
        
    }
    
    /**
     * Snap coordinates to the closest grid position.
     */
    public void snap (double[] coords) {
        // radius
        double r = Math.sqrt (coords[0]*coords[0] + coords[1]*coords[1]);
        if (r <= 0.0)
            return;
        else if (r > 1.0)
            r = 1.0;
        else
            r = Math.floor ( r*numberOfCircles + 0.5) / numberOfCircles;
        
        // angle
        double a = Math.atan2 (coords[1], coords[0]);
        if(numberOfSectors!=0)
            a = Math.floor ( a * numberOfSectors / 2.0 / Math.PI + 0.5)
                * Math.PI * 2.0 / numberOfSectors;
        
        coords[0] = r*Math.cos (a);
        coords[1] = r*Math.sin (a);
        
    } 
    
}
