/* StateEntryDelegate.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.conjecture.history;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;

import org.grinvin.io.list.FilterListLoader;
import org.grinvin.io.list.FilterListSaver;
import org.grinvin.io.list.GraphGeneratorInstanceListLoader;
import org.grinvin.io.list.GraphGeneratorInstanceListSaver;
import org.grinvin.io.list.GraphListLoader;
import org.grinvin.io.list.GraphListSaver;
import org.grinvin.io.list.InvariantListLoader;
import org.grinvin.io.list.InvariantListSaver;
import org.grinvin.io.SectionLoader;
import org.grinvin.io.SectionSaver;
import org.grinvin.list.DefaultFilterListModel;
import org.grinvin.list.generators.DefaultGraphGeneratorInstanceListModel;
import org.grinvin.list.graphs.DefaultGraphListModel;
import org.grinvin.list.invariants.DefaultInvariantListModel;
import org.grinvin.list.FilterListModel;
import org.grinvin.list.generators.GraphGeneratorInstanceListModel;
import org.grinvin.list.graphs.GraphListModel;
import org.grinvin.list.invariants.InvariantListModel;

import org.jdom.Element;

/**
 *
 */
public class StateEntryDelegate implements ConjectureHistoryEntryDelegate<StateEntry> {
    
    public Element toElement(StateEntry entry) {
        Element element = new Element("entry");
        element.setAttribute("type", entry.getType());
        element.setAttribute("time", Long.toString(entry.getTimeStamp()));
        
        Element graphList = new Element("graphList");
        graphList.setAttribute("uri", entry.getGraphListModel().getURI().toString());
        element.addContent(graphList);

        Element invariantList = new Element("invariantList");
        invariantList.setAttribute("uri", entry.getInvariantListModel().getURI().toString());
        element.addContent(invariantList);
        
        Element graphGeneratorInstanceList = new Element("graphGeneratorInstanceList");
        graphGeneratorInstanceList.setAttribute("uri", entry.getGeneratorInstanceListModel().getURI().toString());
        element.addContent(graphGeneratorInstanceList);

        Element filterList = new Element("filterList");
        filterList.setAttribute("uri", entry.getFilterListModel().getURI().toString());
        element.addContent(filterList);
        
        return element;
    }
    
    public void saveIntoWorkspace(SectionSaver ssaver, StateEntry entry) throws IOException {
        GraphListSaver.saveIntoWorkspace(entry.getGraphListModel(), ssaver);
        InvariantListSaver.saveIntoWorkspace(entry.getInvariantListModel(), ssaver);
        GraphGeneratorInstanceListSaver.saveIntoWorkspace(entry.getGeneratorInstanceListModel(), ssaver);
        FilterListSaver.saveIntoWorkspace(entry.getFilterListModel(), ssaver);
    }
    
    public StateEntry fromElement(Element element, SectionLoader sloader) {
        GraphListModel graphListModel = new DefaultGraphListModel();
        InvariantListModel invariantListModel = new DefaultInvariantListModel();
        GraphGeneratorInstanceListModel generatorInstanceListModel = new DefaultGraphGeneratorInstanceListModel();
        FilterListModel filterListModel = new DefaultFilterListModel();
        try {
            
            URI graphlistURI = new URI(element.getChild("graphList").getAttributeValue("uri"));
            InputStream input = sloader.openSection(graphlistURI.getSchemeSpecificPart());
            GraphListLoader.load(graphListModel, input, sloader);
            graphListModel.setURI(graphlistURI);
            
            URI invariantlistURI = new URI(element.getChild("invariantList").getAttributeValue("uri"));
            InputStream invariantlist = sloader.openSection(invariantlistURI.getSchemeSpecificPart());
            InvariantListLoader.load(invariantListModel, invariantlist);
            invariantListModel.setURI(invariantlistURI);
                
            URI generatorURI = new URI(element.getChild("graphGeneratorInstanceList").getAttributeValue("uri"));
            InputStream graphgeneratorinstancelist = sloader.openSection(generatorURI.getSchemeSpecificPart());
            GraphGeneratorInstanceListLoader.load(generatorInstanceListModel, graphgeneratorinstancelist);
            generatorInstanceListModel.setURI(generatorURI);
                
            URI filterURI = new URI(element.getChild("filterList").getAttributeValue("uri"));
            InputStream filterlist = sloader.openSection(filterURI.getSchemeSpecificPart());
            FilterListLoader.load(filterListModel, filterlist);
            filterListModel.setURI(filterURI);

        } catch (IOException ex) {
            //
        } catch (URISyntaxException ex) {
            //
        }
        StateEntry result = new StateEntry();
        result.setGraphListModel(graphListModel);
        result.setInvariantListModel(invariantListModel);
        result.setGeneratorInstanceListModel(generatorInstanceListModel);
        result.setFilterListModel(filterListModel);
        result.setTimeStamp(Long.parseLong(element.getAttributeValue("time")));
        return result;
        
    }

}
