/* Wrapper.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace;

import java.io.File;
import java.io.IOException;

import org.jdom.Element;

/**
 * An object of this type manages the life cycle of one or more subcomponents
 * of the GrInVin system, of type <C>. Wrappers are typically stateless objects
 * one instance of which can be reused to manage different component objects 
 * of the same type <C>.
 */
public interface Wrapper<C> {
    
    /**
     * Create the subcomponent(s) managed by this wrapper. This is typically
     * called when the system is started with no persisted workspace available.
     * @return the newly created subcomponent
     */
    public C create ();
    
    /**
     * Dispose of the given subcomponent. Typically
     * disposes windows, clears caches, etc.
     */
    public void dispose (C component);
    
    /**
     * Load the subcomponent(s) managed by this wrapper from a persistent
     * work space. Will only be called after {@link #dispose} was called on
     * the same object.
     * @param element JDOM element which stores extra information about
     * this wrapper.
     * @param directory Workspace directory
     * @return the object which was loaded
     */
    public C load (Element element, File directory) throws IOException;
    
    /**
     * Save the given subcomponent.
     * @param component Subcomponent to be saved
     * @param parent parent JDOM element into which the JDOM element(s) for
     * this wrapper must be stored.
     * @param directory Workspace directory
     */
    public void save (C component, Element parent, File directory) throws IOException;
    
    
    /**
     * Return the XML-element name by which this type of wrapper is identified
     * within its parent.
     */
    public String getElementName ();
    
    /**
     * Return the major version of the format used by this wrapper. If non-zero
     * a version string will be stored with the XML-element.
     */
    public int getMajorVersion ();
    
    /**
     * Return the minor version of the format used by this wrapper.
     */
    public int getMinorVersion ();
    
}
