/* Workspace.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.Preference;
import org.grinvin.util.IOHelper;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

/**
 * Handles session persistence. The singleton object of this type
 * allows sessions to be persisted to a workspace directory on disk.
 */
public class Workspace {
    
    //
    private static final Logger LOGGER
            = Logger.getLogger("org.grinvin.workspace", "org.grinvin.workspace.resources");
    
    //
    private File directory;
    
    //
    private List<Subsystem<?>> subsystems;
    
    //
    private Workspace() {
        this.directory = null;
        this.subsystems = new ArrayList<Subsystem<?>> ();
    }
    
    //
    private static Workspace SINGLETON = new Workspace();
    
    /**
     * Get the singleton instance of this class
     * @return the single instance of this class
     */
    public static Workspace getInstance() {
        return SINGLETON;
    }
    
    // changes directory and remembers it as a preference
    private void setDirectory(File directory) {
        this.directory = directory;
        GrinvinPreferences.INSTANCE.setStringPreference(
                Preference.WORKSPACE_LOCATION,
                directory.getAbsolutePath());
    }
    
    /**
     * Register (the wrapper) of a subsystem. The subsystem that is registered
     * first is also loaded and initialized first, but disposed and saved last.
     * If a subsystem A depends on a subsystem B, then B should be registered
     * before A.
     * @throws IllegalStateException when called after
     * a workspace directory has already been set
     */
    public<C> void registerSubsystem(Wrapper<C> wrapper) {
        if (directory != null)
            throw new IllegalStateException("Cannot register a subsystem in a running system");
        subsystems.add(new Subsystem<C> (wrapper));
    }
    
    //
    private void dispose() {
        if (this.directory != null) {
            // in reverse order
            for (ListIterator<Subsystem<?>> iter = subsystems.listIterator(subsystems.size());
            iter.hasPrevious(); )
                iter.previous().dispose();
        }
        this.directory = null;
    }
    
    /**
     * Initialize all subsystems and perform an initial save into the
     * given workspace directory. If there is an active workspace already,
     * it is disposed of first. If the new directory already exists,
     * all its contents are removed.
     */
    public void newWorkspace(File directory) throws WorkspaceException {
        dispose();
        // in order of registration
        for (Subsystem<?> subsystem : subsystems)
            subsystem.init();
        saveAs(directory);
    }
    
    /**
     * Save the workspace into the current workspace directory.
     */
    public void save() throws WorkspaceException {
        assert directory != null : "save called without directory";
        try {
            Element root = new Element("workspace");
            // in reverse order
            for (ListIterator<Subsystem<?>> iter = subsystems.listIterator(subsystems.size());
            iter.hasPrevious(); )
                iter.previous().save(root, directory);
            File file = new File(directory, "meta.xml");
            new XMLOutputter(Format.getPrettyFormat()).output(
                    new Document(root), new FileOutputStream(file)
                    );
            LOGGER.log(Level.INFO, "log.saved.workspace", directory);
        } catch (IOException ex) {
            throw new WorkspaceException("I/O error while saving workspace",
                    directory, ex);
        }
    }
    
    /**
     * Save the workspace into a new workspace directory. If the new directory
     * already exists, all its contents are removed.
     */
    public void saveAs(File directory) throws WorkspaceException {
        if (directory.exists()) {
            if (! IOHelper.deleteDirectory(directory))
                throw new WorkspaceException("Directory could not be removed", directory);
        }
        if (! directory.mkdir())
            throw new WorkspaceException("Could not create directory", directory);
        
        setDirectory(directory);
        save();
        LOGGER.log(Level.INFO, "log.created.workspace", directory);
    }
    
    /**
     * Load the workspace from the given workspace directory. If there
     * is an active workspace already, it is first saved and then disposed of.
     */
    public void load(File directory) throws WorkspaceException {
        if (this.directory != null) {
            save();
            dispose();
        }
        try {
            File file = new File(directory, "meta.xml");
            InputStream input = new FileInputStream(file);
            Element root = new SAXBuilder().build(input).getRootElement();
            // in order of registration
            for (Subsystem<?> subsystem : subsystems)
                subsystem.load(root, directory);
            setDirectory(directory);
            LOGGER.log(Level.INFO, "log.loaded.workspace", directory);
        } catch (IOException ex) {
            throw new WorkspaceException("I/O error while loading workspace",
                    directory, ex);
        } catch (JDOMException ex) {
            throw new WorkspaceException("I/O format error in meta.xml",
                    directory, ex);
        }
    }
    
    /**
     * Combines a wrapper and the component for which it is responsible.
     */
    private static class Subsystem<C> {
        
        private Wrapper<C> wrapper;
        
        private C component;
        
        public Subsystem(Wrapper<C> wrapper) {
            this.wrapper = wrapper;
        }
        
        //
        public void init() {
            component = wrapper.create();
        }
        
        //
        public void dispose() {
            wrapper.dispose(component);
            component = null;
        }
        
        //
        public void load(Element parent, File directory) throws IOException {
            Element element = parent.getChild(wrapper.getElementName());
            if (element != null)
                component = wrapper.load(element, directory);
            else
                component = wrapper.create();
        }
        
        //
        public void save(Element parent, File directory) throws IOException {
            wrapper.save(component, parent, directory);
        }
    }
    
}
