/* EditorWindowWrapper.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace;


import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import org.grinvin.gred.GraphEditor;
import org.grinvin.io.EmbeddingLoader;
import org.grinvin.io.EmbeddingSaver;
import org.grinvin.io.GraphListLoader;
import org.grinvin.io.GraphListSaver;
import org.grinvin.io.GraphLoader;
import org.grinvin.io.GraphSaver;
import org.grinvin.io.IOFormatException;
import org.grinvin.list.DefaultGraphListModel;
import org.grinvin.list.GraphList;
import org.grinvin.list.GraphListElement;
import org.grinvin.list.GraphListModel;

import org.jdom.DataConversionException;
import org.jdom.Element;

/**
 * Wrapper for the EditorWindow
 */
public class EditorWindowWrapper extends DefaultWindowWrapper<GraphEditor> {
    
    @Override
    public GraphEditor create() {
        return new GraphEditor(new DefaultGraphListModel());
    }
    
    public String getElementName() {
        return "editorWindow";
    }
    
    /**
     * Dispose of the editor window and the associated guides tool
     */
    @Override
    public void dispose(GraphEditor window) {
        window.getGuidesTool().dispose();
        super.dispose(window);
    }
    
    /**
     * Load the editor window, its graph list, the graph being edited
     * and the guides tool configuration.
     */
    @Override
    public GraphEditor load(Element element, File directory) throws IOException {
        try {
            // Load the corresponding graph list
            URI uri = new URI(element.getAttributeValue("graphlistURI"));
            GraphListModel list = new DefaultGraphListModel();
            File file = new File(directory, uri.getSchemeSpecificPart());
            GraphListLoader.load(list, null, file, directory);
            list.setURI(uri);
            
            // create and initialize the window
            GraphEditor window = new GraphEditor(list);
            
            // load the graph
            // TODO: store graph on file
            GraphLoader graphloader = new GraphLoader(window.getGraph());
            EmbeddingLoader embeddingloader = new EmbeddingLoader(window.getEmbedding());
            graphloader.toGraph(element.getChild("graph"));
            embeddingloader.toEmbedding(element.getChild("embedding"));
            
            // initialize properties of this window and of its guides tool
            loadWindowProperties(element, window);
            window.setDirty(element.getAttribute("dirty").getBooleanValue());
            loadWindowProperties(element.getChild("guidestool"),
                    window.getGuidesTool());
            return window;
            
        } catch (URISyntaxException ex) {
            throw new IOFormatException("Invalid URI in meta.xml", ex);
        } catch (DataConversionException ex) {
            throw new IOFormatException("Incorrect boolean attribute value in meta.xml",
                    ex);
        }
    }
    
    /**
     * Load the editor window, its graph list, the graph being edited
     * and the guides tool configuration.
     */
    @Override
    public void save(GraphEditor window, Element parent, File directory) throws IOException {
        
        // save properties of window and of guides tool
        Element result = baseElement ();
        saveWindowProperties(result, window);
        Element child = new Element("guidestool");
        saveWindowProperties(child, window.getGuidesTool());
        result.addContent(child);
        result.setAttribute("dirty", "" + window.isDirty());
        
        // save graph being edited
        EmbeddingSaver embeddingsaver = new EmbeddingSaver(window.getEmbedding());
        GraphSaver graphsaver = new GraphSaver(window.getGraph());
        result.addContent(embeddingsaver.toElement());
        result.addContent(graphsaver.toElement());
        
        // save graph list (including contents)
        GraphList list = window.getGraphList();
        for (GraphListElement gle: list)
            GraphListSaver.saveIntoWorkspace(gle, directory);
        GraphListSaver.saveIntoWorkspace(list, null, directory);
        result.setAttribute("graphlistURI", list.getURI().toString());
        
        parent.addContent(result);
        
    }
    
}
