/* InternationalizedProperties.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.Preference;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

/**
 * Internationalized version of the standard Java
 * {@link java.util.Properties} class. Bundles a
 * set of properties with internationalized values. A property can be retrieved
 * for a given key and locale. When no locale is explicitly indicated,
 * a default locale is used. This default corresponds to the default locale of the
 * application, but this may be changed by means of the method
 * {@link #setDefaultLocale}.<p>
 * Properties are stored in a hierarchy based on locale. When searching for a
 * property a list of subsequently less specific locales is used until a match is
 * found or no match exists. For example, when the current locale is
 * <tt>nl_be_UNIX</tt> properties are looked for in locales
 * <tt>nl_be_UNIX</tt>, <tt>nl_be</tt>, <tt>nl</tt> and finally the
 * default 'empty' locale.<p>
 * This class has an interface similar to that of
 * the standard class {@link java.util.Properties}
 * but does not extend it. Files saved by the one class cannot be loaded
 * by the other.<p>
 * The methods {@link #load(InputStream)} and {@link #save(OutputStream)}
 * load and save internationalized
 * properties in a simple XML-format that conforms to the following DTD.
 * <pre>
 *    &lt;!ELEMENT resources ( properties, node* ) &gt;
 *
 *    &lt;!ELEMENT node ( properties, node* ) &gt;
 *    &lt;!ATTLIST node name CDATA #REQUIRED &gt;
 *
 *    &lt;!ELEMENT properties ( entry* ) &gt;
 *    &lt;!ELEMENT entry (#PCDATA) &gt;
 *    &lt;!ATTLIST entry key CDATA #REQUIRED &gt;
 * </pre>
 * Each <tt>node</tt> corresponds to a locale with a given <tt>name</tt>.
 * A child
 * node corresponds to a locale that is more specific than that of its parent.
 * The <tt>properties</tt> element has the same structure as the one
 * used in the standard Java {@link java.util.Properties} class.<p>
 * Methods {@link #toElement} and {@link #fromElement} provide conversion
 * to and from JDOM elements, using a similar format, except that
 * instead of <tt>resources</tt> a different
 * top level element can be used.
 */
public class InternationalizedProperties {
    
    /**
     * Default constructor. Creates an empty property list.
     */
    public InternationalizedProperties() {
        this.defaultLocale = Locale.getDefault();
        this.root = new Node(null, "_");
    }
    
    //
    private Node root;
    
    /**
     * Default locale used by this list.
     */
    private Locale defaultLocale;
    
    /**
     * Set the default locale to be used for this list.
     */
    public void setDefaultLocale(Locale locale) {
        this.defaultLocale = locale;
    }
    
    /**
     * Searches for the property with the specified key in this property list,
     * starting with the given locale.
     * The method returns the given default value if the property is not found.
     *
     * @param   key   the property key.
     * @param   locale the locale to be searched
     * @param   defaultValue a default value to be returned when the property
     *          is not found
     */
    public String getProperty(String key, Locale locale, String defaultValue) {
        Node node = root.findDescendant(locale);
        String value = node.properties.get(key);
        while (value == null && node.parent != null) {
            node = node.parent;
            value = node.properties.get(key);
        }
        return value == null ? defaultValue : value;
    }
    
    /**
     * Searches for the property with the specified key in this property list,
     * starting with the given locale.
     * The method returns {@code null} if the property is not found.
     *
     * @param   key   the property key.
     * @param   locale the locale to be searched
     */
    public String getProperty(String key, Locale locale) {
        return getProperty(key, locale, null);
    }
    
    /**
     * Searches for a property with the specified key. Starts the
     * search with the default locale. Returns the goven default value
     * when the property is not found.
     * @param   key   the property key.
     * @param   defaultValue a default value to be returned when the property
     *          is not found
     */
    public String getProperty(String key, String defaultValue) {
        return getProperty(key,defaultLocale,defaultValue);
    }
    
    /**
     * Searches for a property with the specified key. Starts the
     * search with the default locale. Returns {@code null} when the
     * property is not found.
     * @param   key   the property key.
     */
    public String getProperty(String key) {
        return getProperty(key,defaultLocale);
    }
    
    
    /**
     * Adds a given property/value pair for the given locale.
     * The property is only stored with the locale if that locale was previously
     * registered with the list, otherwise a less specific locale is used.
     */
    public void setProperty(String key, String value, Locale locale) {
        root.findDescendant(locale).properties.put(key, value);
    }
    
    /**
     * Adds a given property/value pair for the default locale.
     * If the default locale is not registered with this list, then the property
     * is stored with a less specific locale.
     */
    public void setProperty(String key, String value) {
        setProperty(key, value, defaultLocale);
    }
    
    /**
     * Registers a locale with this property list. Properties can only be
     * stored with locales that have been registered. If an ancestor
     * of this locale was not yet registered, it is registerd automatically.
     */
    public void registerLocale(Locale locale) {
        if (locale != null)
            register(locale.toString()+"_");
    }
    
    /**
     * Find the node with the given name or add it to the tree if necessary.
     */
    private Node register(String name) {
        Node n = root.findDescendant(name);
        while (! n.lName.equals(name)) {
            int end = name.indexOf('_', n.lName.length())+1;
            Node child = new Node(n, name.substring(0,end));
            n.children.add(child);
            n = child;
        }
        return n;
    }
    
    /**
     * Load properties from a JDOM node element with the given locale name.
     */
    private void load(Element element, String name) {
        register(name).propertiesFromElement(element.getChild("properties"));
        for (Object obj: element.getChildren("node")) {
            Element el = (Element)obj;
            load(el, el.getAttributeValue("name") + "_");
        }
    }
    
    /**
     * Loads the properties from the given input stream
     * and adds them to this list. Uses the same data format as {@link #save}.
     */
    public void load(InputStream input) throws IOException {
        try {
            Document document = new SAXBuilder().build(input);
            Element root = document.getRootElement(); // probably closes the stream
            if (! "resources".equals(root.getName()))
                throw new IOException("Invalid input format");
            fromElement(root);
        } catch (JDOMException ex) {
            throw new IOException("Invalid input format");
        }
    }
    
    /**
     * Load properties from the given JDOM-element and adds them to the list.
     */
    public void fromElement(Element element) {
        load(element, "_");
    }
    
    /**
     * Saves the contents of the given node into the given element.
     */
    private void save(Element element, Node node) {
        element.addContent(node.propertiesToElement());
        for (Node child : node.children) {
            Element childElement = new Element("node");
            save(childElement, child);
            String name = child.lName.substring(0, child.lName.length()-1);
            childElement.setAttribute("name", name);
            element.addContent(childElement);
        }
    }
    
    /**
     * Convert the current list of properties to a JDOM-element of the given name.
     */
    public Element toElement(String name) {
        Element element = new Element(name);
        save(element, root);
        return element;
    }
    
    /**
     * Convert the current list of properties to a JDOM-element with the
     * default name <tt>resources</tt>.
     */
    public Element toElement() {
        return toElement("resources");
    }
    
    /**
     * Writes this property list to the given output stream.
     * Uses the same data format as {@link #save}. Note that the generated data
     * will contain information on all locales.
     */
    public void save(OutputStream out) throws IOException {
        XMLOutputter outputter;
        if (GrinvinPreferences.INSTANCE.getStringPreference(Preference.XMLOUTPUT_FORMAT).equals("pretty"))
            outputter = new XMLOutputter(Format.getPrettyFormat());
        else
            outputter = new XMLOutputter(Format.getCompactFormat());
        
        outputter.output(new Document(toElement()), out);
    }
    
    /**
     * Node in the property tree. Each node corresponds to a locale
     * where children of a node correspond to a more specific locale
     * than their parents. Locales are represented by their names. Properties
     * are stored in hash maps.
     */
    private static class Node {
        
        // name of the locale, suffixed by an underscore
        public String lName;
        
        // List of children
        public List<Node> children;
        
        // Properties
        public Map<String,String> properties;
        
        // parent node, or <code>null</code>
        public Node parent;
        
        
        /**
         * Create a new node with the given parent and locale.
         */
        public Node(Node parent, String name) {
            this.parent = parent;
            this.lName = name;
            this.children = new ArrayList<Node> ();
            this.properties = new HashMap <String, String> ();
        }
        
        /**
         * Find the index of the child that best matches the given
         * locale string.
         */
        private int findChild(String str) {
            int i = children.size() - 1;
            while (i >= 0 && ! str.startsWith(children.get(i).lName))
                i--;
            return i;
        }
        
        /**
         * Find the descendant that best matches the given locale.
         */
        public Node findDescendant(Locale locale) {
            if (locale == null)
                return this;
            return findDescendant(locale.toString()+"_");
        }
        
        /**
         * Find the descendant that best matches the given name.
         */
        public Node findDescendant(String str) {
            Node n = this;
            int index = n.findChild(str);
            while (index >= 0) {
                n = n.children.get(index);
                index = n.findChild(str);
            }
            return n;
        }
        
        /**
         * Create a JDOM element that corresponds to the properties stored
         * in this node.
         */
        public  Element propertiesToElement() {
            Element element = new Element("properties");
            for (Map.Entry<String,String> entry: properties.entrySet()) {
                String value = entry.getValue();
                if (value != null) {
                    Element entryElement = new Element("entry");
                    entryElement.setAttribute("key",entry.getKey());
                    entryElement.setText(value);
                    element.addContent(entryElement);
                }
            }
            return element;
        }
        
        /**
         * Load the properties from a JDOM properties element into the given node.
         */
        public  void propertiesFromElement(Element element) {
            for (Object obj: element.getChildren("entry")) {
                Element el = (Element)obj;
                properties.put(el.getAttributeValue("key"), el.getTextNormalize());
            }
        }
        
    }
}
