/* GraphSaver.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io;

import java.io.IOException;
import java.io.OutputStream;
import java.util.IdentityHashMap;
import java.util.Map;
import org.grinvin.Edge;
import org.grinvin.GraphView;
import org.grinvin.Vertex;
import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.Preference;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

/**
 * Saves a {@link GraphView} as a <tt>graph.xml</tt>-section onto an output
 * stream.
 */
public class GraphSaver {
    
    // map which provides IDs for all vertices
    private Map<Vertex,Integer> map;
    
    //
    private GraphView graph;
    
    // number of vertices
    private int numberOfVertices;

    /**
     * Return the vertex map for this graph. This map can be used by other
     * savers to translate actual vertex objects to vertex numbers.
     */
    public Map<Vertex,Integer> getVertexMap () {
        return this.map;
    }
    
    /** Creates a new instance of this saver for the given abstract graph. 
     * Initializes the vertex map.
     * @see #getVertexMap
     */
    public GraphSaver(GraphView graph) {
        this.graph = graph;
        this.map = new IdentityHashMap<Vertex,Integer> ();
        int count = 0;
        for (Vertex v: graph.vertices()) {
            map.put(v,new Integer(count));
            count ++;
        }
        numberOfVertices = count;
    }
    
    /**
     * Create a JDOM-element that represents the abstract graph
     * for this initializer.
     */
    public Element toElement() {
        Element result = new Element("graph");
        result.setAttribute("nrofvertices", "" + numberOfVertices);
        
        // vertices (annotated only)
        for (Vertex v: graph.vertices()) {
            if (v.getAnnotation() != null) {
                Element vertex = new Element("vertex");
                vertex.setAttribute("id", map.get(v).toString());
                vertex.setAttribute("annotation", v.getAnnotation().toString());
                result.addContent(vertex);
            }
        }
        
        // edges (all edges)
        for (Edge e: graph.edges()) {
            Element edge = new Element("edge");
            edge.setAttribute("from", map.get(e.getFirstEndpoint()).toString());
            edge.setAttribute("to", map.get(e.getSecondEndpoint()).toString());
            if (e.getAnnotation() != null) {
                edge.setAttribute("annotation", e.getAnnotation().toString());
            }
            result.addContent(edge);
        }
        return result;
    }
    
    /**
     * Save the graph onto the output stream.
     * @param out Output stream to which the result shall be written.
     */
    public void save(OutputStream out) throws IOException{
        XMLOutputter outputter;
        if (GrinvinPreferences.INSTANCE.getStringPreference(Preference.XMLOUTPUT_FORMAT).equals("pretty"))
            outputter = new XMLOutputter(Format.getPrettyFormat());
        else
            outputter = new XMLOutputter(Format.getCompactFormat());
        
        Document doc = new Document (toElement());
        outputter.output(doc, out);
    }
}
