/* InvariantValue.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants;

import org.grinvin.io.IOFormatException;
import org.jdom.Element;

/**
 * Wraps the result of an invariant computation and provides a
 * means to represent this result both as a string and an XML element.
 */
public abstract class InvariantValue {
    
    //
    private Invariant invariant;
    
    //
    private String computerId;
    
    //
    private String computerVersion;
    
    //
    private String type;
    
    /**
     * Return the invariant of which this is a value.
     */
    public Invariant getInvariant () {
        return invariant;
    }
    
    /**
     * Create a value of the given type which was computed by the given invariant computer.
     */
    protected InvariantValue (String type, InvariantComputer computer) {
        this.computerId = computer.getId ();
        this.computerVersion = computer.getVersion ();
        this.invariant = InvariantManager.getInstance ().
                getInvariantForComputer (computer);
        this.type = type;
    }
    
    /**
     * Create a new unitialized value of the given type. Used by the loader.
     */
    protected InvariantValue (String type) {
        this.type = type;
    }
    
    /**
     * Initialize the components of this value from a JDOM-element.
     * The element is known to be of type 'invariantvalue'.
     * Delegates to {@link #loadValue} to load the actual value.
     */
    protected  void load (Element element) throws 
            IOFormatException, UnknownInvariantException, UnknownInvariantTypeException {
        
        Element child = element.getChild ("invariant");
        if (child == null)
            throw new IOFormatException ("Missing 'invariant' element in 'invariantvalue'");
        this.invariant = InvariantManager.getInstance ().getInvariant (child.getTextTrim ());
        
        child = element.getChild ("computer");
        if (child == null)
            throw new IOFormatException ("Missing 'computer' element in 'invariantvalue'");
        this.computerId = child.getTextTrim ();
        this.computerVersion = child.getAttributeValue ("version");
        
        child = element.getChild ("value");
        if (child == null)
            throw new IOFormatException ("Missing 'value' element in 'invariantvalue'");
        loadValue (child);
        
    }
    
    /**
     * Should be overridden to load the actual value from a JDOM-element.
     * Clients should throw an IOFormatException when element is not
     * in the correct format.
     * @param element JDOM element, which is known to be of type
     * 'value'
     * @see #load
     */
    public abstract void loadValue (Element element) throws IOFormatException;
    
    /**
     * Converts this value to a JDOM element. Delegates saving of the actual value to
     * {@link #saveValue}.
     */
    public Element toElement () {
        Element element = new Element ("invariantvalue").setAttribute ("type", type);
        element.addContent (new Element ("invariant").setText (invariant.getId ()));
        element.addContent (new Element ("computer").setText (computerId)
        .setAttribute ("version", computerVersion));
        Element child = new Element ("value");
        saveValue (child);
        return element.addContent (child);
    }
    
    
    /**
     * Should be overridden to save the actual value into a JDOM-element.
     * @param element JDOM element, which is of type 'value'
     */
    public abstract void saveValue (Element element);
    
    // TODO: load (see InvariantValuesLoader)
    
    /**
     * Convert this value to a string, so it can be displayed
     * as part of a GUI.
     */
    public abstract String toText () ;
    
    //
    public abstract double asDouble ();
    
    public String toString () {
        return invariant.getId() + " computed by " + computerId + " (" + computerVersion + ") : " + toText();
    }
    
    /**
     * Convert a JDOM-element to an invariant value of the correct class.
     */
    public static InvariantValue fromElement (Element element) throws
            IOFormatException, UnknownInvariantException, UnknownInvariantTypeException {
        if (! "invariantvalue".equals (element.getName ()))
            throw new IOFormatException ("Expected <invariantvalue> element");
        String type = element.getAttributeValue ("type");
        if (type == null)
            throw new IOFormatException ("Missing type attribute for invariant value");
        
        try {
            Class<? extends InvariantValue> typeClass =
                InvariantTypes.classForType (type);
            InvariantValue value = typeClass.newInstance ();
            value.load (element);
            return value;
        } catch (InstantiationException ex) {
            throw new UnknownInvariantTypeException (
                    "Could not instantiate invariant type", type, ex);
        } catch (IllegalAccessException ex) {
            throw new  UnknownInvariantTypeException (
                    "Could not instantiate invariant type", type, ex);
        }
    }
    
    public String getComputerVersion () {
        return computerVersion;
    }
}
