/* DefaultInvariant.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants;

import java.io.IOException;

import org.grinvin.io.IOFormatException;
import org.grinvin.util.InternationalizedProperties;
import org.grinvin.xml.XMLUtils;

import org.jdom.Element;

/**
 * Default implementation of {@link Invariant}.
 */
class DefaultInvariant extends Invariant {
    
    //
    protected String id;
    
    //
    protected InternationalizedProperties iprops;
    
    //
    protected Class type;
    
    /**
     * Initialize this invariant from the given JDOM element.
     * @param id Expected value of the id element.
     * @param rootElement Expected name of a root element.
     */
    protected void init (Element root, String id) throws 
            IOFormatException, UnknownInvariantTypeException {
        if (root == null)
            throw new IOFormatException ("Top level element is null");

        if (! getRootElement ().equals (root.getName ()))
            throw new IOFormatException ("Top level element should be \'" + getRootElement () + "\'");
        
        // identifier
        Element el = root.getChild ("id");
        if (!id.equals (el.getTextTrim ()))
            throw new IOFormatException ("Incorrect value for element 'id'");
        this.id = id;
        
        // type
        el = root.getChild ("type");
        if (el == null)
            throw new IOFormatException ("Missing 'type' element");
        this.type = InvariantTypes.classForType (el.getTextTrim ());
        
        // internationalized properties
        el = root.getChild ("resources");
        if (el == null)
            throw new IOFormatException ("Cannot find 'resources' element");
        this.iprops = new InternationalizedProperties ();
        iprops.fromElement (el);
    }
    
    /**
     * Top level root element for the XML-file.
     */
    protected String getRootElement () {
        return "invariant";
    }
    
    //
    private final String path;
    
    /**
     * Return the path which contains the definition file for this invariant.
     */
    public String getPath () {
        return this.path;
    }
    
    /**
     * Create a default invariant with given identifier.
     *
     * Information about this invariant is loaded from
     * an XML-file in the class path
     *
     * @param id Identifier of this invariant
     * @param path Path for the XML-file of the invariant
     * @param class loader to be used for loading the XML file
     * @throws UnknownInvariantException when an invariant with this id
     * could not be instantiated
     */
    public DefaultInvariant (String id, String path, ClassLoader classLoader) throws 
            UnknownInvariantException {
        
        try {
            this.path = path;
            Element root = XMLUtils.loadFromClassPath(classLoader, path);
            init (root, id);
        } catch (IOFormatException ex) {
            throw new UnknownInvariantException ("Invalid format of XML-file", id, ex);
        } catch (IOException ex) {
            throw new UnknownInvariantException ("Could not read XML-file", id, ex);
        } catch (UnknownInvariantTypeException ex) {
            throw new UnknownInvariantException ("Invariant of unknown type", id, ex);
        }
    }
    
    
    // implements Invariant
    public String getName () {
        return iprops.getProperty ("name");
    }
    
    // implements Invariant
    public String getId () {
        return id;
    }
    
    //
    public Class getType () {
        return type;
    }
    
    //
    public String toString () {
        return getName ();
    }
    
}
