/* InvariantTOCView.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.help;

import java.util.Hashtable;
import java.util.Locale;
import javax.help.BadIDException;
import javax.help.HelpSet;
import javax.help.Map.ID;
import javax.help.TOCItem;
import javax.help.TOCView;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.MutableTreeNode;

import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantFactory;
import org.grinvin.list.InvariantTreeModel;

/**
 * Loads an overview of the available invariants and places them in a TOC tree
 * that is mergeable with the GrInvIn help TOC tree.
 */
public class InvariantTOCView extends TOCView {
    
    private InvariantTreeModel tree = InvariantTreeModel.getStandardTree();
      
    /**
     * Creates a new instance of InvariantTOCView
     */
    public InvariantTOCView(){
        super(HelpManager.getHelpSet(), HelpManager.getResourceBundle().getString("TOC.name"), HelpManager.getResourceBundle().getString("TOC.label"), HelpManager.getHelpSet().getLocale(), new Hashtable());
   }
    
    public InvariantTOCView(HelpSet hs, String name, String Label, Hashtable params){
        this();
    }
    
    public InvariantTOCView(HelpSet hs, String name, String Label, Locale locale, Hashtable params){
        this();
    }
    
    public String getMergeType(){
        return "javax.help.UniteAppendMerge";
    }
    
    public DefaultMutableTreeNode getDataAsTree() {
        //grinvin
        TOCItem itemgrinvin = new TOCItem();
        itemgrinvin.setName(HelpManager.getResourceBundle().getString("TOC.item.grinvin"));
        DefaultMutableTreeNode grinvin = new DefaultMutableTreeNode(itemgrinvin);
        // -- mainwindow
        TOCItem itemmainwindow = new TOCItem(ID.create("org.grinvin.window.main", HelpManager.getHelpSet()), null, null);
        itemmainwindow.setName(HelpManager.getResourceBundle().getString("TOC.item.mainwindow"));
        DefaultMutableTreeNode mainWindow = new DefaultMutableTreeNode(itemmainwindow);
        // -- -- invariants tab
        TOCItem itemInvariantsTab = new TOCItem(ID.create("org.grinvin.window.main.tab.invariants", HelpManager.getHelpSet()), null, null);
        itemInvariantsTab.setName(HelpManager.getResourceBundle().getString("TOC.item.invariantstab"));
        DefaultMutableTreeNode invariantsTab = new DefaultMutableTreeNode(itemInvariantsTab);
        // -- -- -- (invariants)
        for(int i=0; i < tree.getChildCount(tree.getRoot()); i++) {
            MutableTreeNode childNode = load(tree.getChild(tree.getRoot(), i));
            if(childNode!=null)
                invariantsTab.add(childNode);
        }
        mainWindow.add(invariantsTab);
        grinvin.add(mainWindow);
        DefaultMutableTreeNode root = new DefaultMutableTreeNode(null);
        root.add(grinvin);
        return root;
    }
    
    private MutableTreeNode load(Object invariantTreeNode){
        TOCItem item = null;
        if(tree.isLeaf(invariantTreeNode)) {
            try {
                Object nodeItem = ((DefaultMutableTreeNode)invariantTreeNode).getUserObject();
                //should be Invariant or InvariantFactory
                if(nodeItem instanceof Invariant){
                    item = new TOCItem(ID.create(((Invariant)nodeItem).getId(), HelpManager.getHelpSet()), ID.create("org.grinvin.icons.invariant", HelpManager.getHelpSet()), null, null);
                    item.setName(((Invariant)nodeItem).getName());
                    return new DefaultMutableTreeNode(item, false);
                } else if(nodeItem instanceof InvariantFactory){
                    item = new TOCItem(ID.create(((InvariantFactory)nodeItem).getId(), HelpManager.getHelpSet()), ID.create("org.grinvin.icons.invariantfactory", HelpManager.getHelpSet()), null, null);
                    item.setName(((InvariantFactory)nodeItem).getName());
                    return new DefaultMutableTreeNode(item, false);
                } else
                    return null;
            } catch (ClassCastException ex) {
                //something went wrong: too bad, no help for this item
                return null;
            }  catch (BadIDException ex) {
                //There is no help page for this item
                //so it shouldn't be in the TOC.
                return null;
            }
        } else {
            item = new TOCItem();
            item.setName(invariantTreeNode.toString());
            DefaultMutableTreeNode node = new DefaultMutableTreeNode(item);
            for(int i=0; i < tree.getChildCount(invariantTreeNode); i++) {
                MutableTreeNode childNode = load(tree.getChild(invariantTreeNode, i));
                if(childNode!=null)
                    node.add(childNode);
            }
            return node;
        }
        
    }
}
