/* ChemicalIconFactory.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.icons;

import javax.swing.Icon;

import org.grinvin.GraphBundleView;
import org.grinvin.GraphView;

/**
 * Graph icon factory that uses {@link ChemicalIcon} for graphs with 
 * a small number of vertices.
 */
public class ChemicalIconFactory extends DefaultGraphIconFactory {

    /** 
     * Default constructor. Can be overridden, but should not
     * be called directly. 
     * @see #getInstance
     */
    protected ChemicalIconFactory() { }
    
    //
    private static final GraphIconFactory SINGLETON = new ChemicalIconFactory();
    
    /**
     * Return the sole instance of this class.
     */
    public static GraphIconFactory getInstance() {
        return SINGLETON;
    }
    
    //
    protected int getVertexLimitFactor() {
        return 600;
    }

    protected GraphIcon createIconForSmallGraph(GraphBundleView bundle, int size) {
        return new ChemicalIcon (bundle, size);
    }
    
    protected GraphIcon createIconForMediumGraph(GraphBundleView bundle, int size) {
        return new EdgesOnlyIcon (bundle, size);
    }
    
    /**
     * Return the background icon for the large graph icon.
     */
    protected Icon getBackgroundIcon(int size) {
        return SvgIconManager.getInstance().getIcon("/org/grinvin/icons/eograph.svg", size);
    }
    
    //
    public GraphIcon createIcon(GraphBundleView bundle, int size) {
        GraphView graph = bundle.getGraph();
        if (graph != null) {
            if (isSmallGraph(bundle, size))
                return createIconForSmallGraph(bundle, size);
            else if (isMediumGraph(bundle, size))
                return createIconForMediumGraph(bundle, size);
            else
                return createIconForLargeGraph(bundle, size);
        } else {
            return createIconForMissingGraph(bundle, size);
        }
    }

    private boolean isMediumGraph(GraphBundleView bundle, int size) {
        return bundle.getGraph().getNumberOfVertices()*getVertexLimitFactor()/4
        <= size*size;
    }
}
