/* GrinvinWindow.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.grinvin;

import be.ugent.caagt.swirl.commands.AttributedCommandManager;
import be.ugent.caagt.swirl.commands.ShowWindow;
import be.ugent.caagt.swirl.lists.ArrayListModel;
import be.ugent.caagt.swirl.lists.TypedListModel;

import java.awt.Component;
import java.awt.Window;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.help.CSH;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.ListSelectionModel;
import javax.swing.WindowConstants;
import javax.swing.event.ChangeListener;

import org.grinvin.factories.graphs.GraphFactory;
import org.grinvin.factories.graphs.GraphFactoryManager;
import org.grinvin.graphbrowser.GraphBrowserWindow;
import org.grinvin.gui.icons.SvgIconManager;
import org.grinvin.help.FactoryListHelpManager;
import org.grinvin.help.HelpManager;
import org.grinvin.help.InvariantTreeHelpManager;
import org.grinvin.list.DefaultGraphListsModel;
import org.grinvin.list.FactoryList;
import org.grinvin.list.GraphListsComponent;
import org.grinvin.list.GraphListsModel;
import org.grinvin.list.InvariantTree;
import org.grinvin.list.InvariantTreeModel;
import org.grinvin.workspace.Workspace;
import org.grinvin.workspace.WorkspaceException;

import org.pietschy.command.CommandGroup;
import org.pietschy.command.LoadException;

/**
 * This class represents the GrInvIn main window.
 */
public class GrinvinWindow extends JFrame {
    
    //
    private static final ResourceBundle BUNDLE
            = ResourceBundle.getBundle("org.grinvin.grinvin.resources");
    
    
    private AttributedCommandManager commandManager;
    
    //
    private JTabbedPane tabs;
    
    /**
     * Creates a new instance of GrinvinWindow
     */
    public GrinvinWindow() {
        super(BUNDLE.getString("window.main.title"));
        setLayout(new BoxLayout(this.getContentPane(), BoxLayout.LINE_AXIS));
        
        // window map
        windowMap = new HashMap<String,Window> ();
        
        // tabs map
        tabsMap = new HashMap<String,JComponent> ();
        reverseTabsMap = new HashMap<JComponent,String> ();
        
        
        tabs = new JTabbedPane();
        
        // add logo
        ImageIcon icon = SvgIconManager.getInstance().getIcon("/org/grinvin/icons/grinvin.svg", 96);
        JLabel intro =  new JLabel(icon);
        intro.setAlignmentY(Component.TOP_ALIGNMENT);
        add(intro);
        //tabs.addTab(BUNDLE.getString("tab.intro.title"), null, intro, BUNDLE.getString("tab.intro.tooltip"));
        //tabsMap.put("intro", intro);
        //reverseTabsMap.put(intro, "intro");
        
        //add invariants
        InvariantTree tree = new InvariantTree(InvariantTreeModel.getStandardTree());
        tree.setDragEnabled(true);
        JScrollPane treePane = new JScrollPane(tree);
        tabs.addTab(BUNDLE.getString("tab.invariants.title"), null, treePane, BUNDLE.getString("tab.invariants.tooltip"));
        tabsMap.put("invariants", treePane);
        reverseTabsMap.put(treePane, "invariants");
        
        //add factories
        FactoryList factoryList = new FactoryList(getStandardGraphFactories());
        JScrollPane listPane = new JScrollPane(factoryList);
        tabs.addTab(BUNDLE.getString("tab.factories.title"), null, listPane, BUNDLE.getString("tab.factories.tooltip"));
        tabsMap.put("factories", listPane);
        reverseTabsMap.put(listPane, "factories");
        
        // add graphlists
        this.graphListsModel = new DefaultGraphListsModel(new ArrayListModel<GraphBrowserWindow> ());
        JList graphListsList = new GraphListsComponent(graphListsModel.getListModel());
        graphListsListSelectionModel = graphListsList.getSelectionModel();
        JScrollPane graphListsPane = new JScrollPane(graphListsList);
        tabs.addTab(BUNDLE.getString("tab.graphlists.title"), null, graphListsPane, BUNDLE.getString("tab.graphlists.tooltip"));
        tabsMap.put("graphlists", graphListsPane);
        reverseTabsMap.put(graphListsPane, "graphlists");
        
        tabs.setAlignmentY(Component.TOP_ALIGNMENT);
        add(tabs);
        
        // command manager
        this.commandManager = new AttributedCommandManager();
        commandManager.setResourceBundle(BUNDLE);
        commandManager.setAttribute(getClass().getName(), this);
        
        // load the xml command definition and initialize the manager.
        URL myCommandURL = getClass().getResource("grinvin.cmds");
        try {
            commandManager.load(myCommandURL);
        } catch (LoadException e) {
            throw new RuntimeException("Failed to load grinvin.cmds", e);
        }
        
        //CommandManager.setLoggerFactory(new PrintStreamLoggerFactory(System.out));
        
        CommandGroup mainMenuGroup = commandManager.getGroup("main.menu");
        
        setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
        addWindowListener(new GrinvinWindowListener());
        // menus
        setJMenuBar(mainMenuGroup.createMenuBar());
        
        // icon
        setIconImage(SvgIconManager.getInstance().getIcon("/org/grinvin/icons/grinvin-favicon.svg", 16).getImage());
        
        // pack
        pack();
        
        // help
        HelpManager.setHelpIDString(this, "org.grinvin.window.main");
        HelpManager.setHelpIDString(commandManager.getGroup("menu.file").getButtonIn(this), "org.grinvin.window.main.menu.file");
        HelpManager.setHelpIDString(commandManager.getGroup("menu.window").getButtonIn(this), "org.grinvin.window.main.menu.window");
        HelpManager.setHelpIDString(commandManager.getGroup("menu.help").getButtonIn(this), "org.grinvin.window.main.menu.help");
        HelpManager.enableHelpKey(this, "org.grinvin.window.main");
        HelpManager.enableHelpKey(tabs, "org.grinvin.window.main");
        HelpManager.enableHelpKey(tree, "org.grinvin.window.main.tab.invariants");
        CSH.addManager(new InvariantTreeHelpManager(tree));
        HelpManager.enableHelpKey(factoryList, "org.grinvin.window.main.tab.factory");
        CSH.addManager(new FactoryListHelpManager(factoryList));
        HelpManager.enableHelpKey(graphListsList, "org.grinvin.window.main.tab.graphlists");
    }
    
    //
    private final Map<String,Window> windowMap;
    
    /**
     * Retrieve the application window with the given key. Current
     * available keys are
     * <ul>
     * <li><tt>AvailableInvariantsList</tt> for the invariant tree window</li>
     * <li><tt>EditorWindow</tt> for the graph editor</li>
     * <li><tt>FactoryWindow</tt> for the list of graph factories</li>
     * <li><tt>ConjecturingWindow</tt> for the conjecturing window</li>
     * </ul>
     */
    public Window getWindow(String key) {
        return windowMap.get(key);
    }
    
    /**
     * Set the application window for the given key. See {@link #getWindow}
     * for possible keys. Also installs a menu command which enables the
     * window to be made visible.
     */
    public void setWindow(String key, Window window) {
        windowMap.put(key, window);
        new ShowWindow(commandManager, key + "Command", window).
                export();
    }
    
    private GraphListsModel graphListsModel;
    
    public GraphListsModel getGraphListsModel(){
        return graphListsModel;
    }
    
    public void setGraphListsModel(GraphListsModel model) {
        this.graphListsModel = model;
        loadGraphListWindowsInMenu();
    }
    
    private ListSelectionModel graphListsListSelectionModel;
    
    public ListSelectionModel getGraphListsListSelectionModel(){
        return graphListsListSelectionModel;
    }
    
    /**
     * Install the menu for the graphs lists window and its associated
     * window group.
     */
    public void loadGraphListWindowsInMenu(){
        CommandGroup windowMenuGroup = commandManager.getGroup("menu.window");
        WindowGroup windowsMenuGroup = new WindowGroup(commandManager, "menu.openwindows", getGraphListsModel().getListModel());
        windowMenuGroup.addInline(windowsMenuGroup);
    }
    
    //
    private final Map<String,JComponent> tabsMap;
    private final Map<JComponent,String> reverseTabsMap;
    
    public String getSelectedTab(){
        return reverseTabsMap.get(tabs.getSelectedComponent());
    }
    
    public void setSelectedTab(String key){
        tabs.setSelectedComponent(tabsMap.get(key));
    }
    
    public void addTabsChangeListener(ChangeListener l){
        tabs.addChangeListener(l);
    }
    
    public void removeTabsChangeListener(ChangeListener l){
        tabs.removeChangeListener(l);
    }
    
    private TypedListModel<GraphFactory> getStandardGraphFactories() {
        TypedListModel<GraphFactory> model = new ArrayListModel<GraphFactory>();
        GraphFactoryManager.getInstance().addToList(model);
        return model;
    }
    
    /**
     * Simple WindowAdapter that saves the Workspace when GrInvIn is closed.
     */
    private static class GrinvinWindowListener extends WindowAdapter {
        
        //
        public void windowClosing(WindowEvent e) {
            try {
                Workspace.getInstance().save();
            } catch (WorkspaceException ex) {
                Logger.getLogger("org.grinvin.io").log(Level.SEVERE, BUNDLE.getString("workspace.savefailed.text"), ex);
            }
            System.exit(0);
        }
        
    }
    
}
