/* AffineTransformation.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred.transformations;


/**
 * Affine transformation from n-space to 2-space. Such a
 * transformation is represented by a matrix
 * <pre>
 *    [ a00 a01 a02 ... b0 ]
 *    [ a10 a11 a12 ... b1 ]
 * </pre>
 * and transforms source coordinates (c0,c1,...) to destination
 * coordinates (x,y) as follows:
 * <pre>
 *    x = a00 c0 + a01 c1 + a02 c2 + ... + b0
 *    y = a10 c0 + a11 c1 + a12 c2 + ... + b1
 * </pre>
 */
public class AffineTransformation extends AbstractTransformation {
    
    // transformation matrix
    private double[][] matrix;
    
    // dimension of the source space
    private int dimension;
    
    // extends Transformation
    @Override public int getDimension() {
        return dimension;
    }

    // extends Transformation
    @Override public double[] transform(double[] src, double[] dest) {
        if (src.length != dimension)
            throw new IllegalArgumentException ("Incorrect source dimension");
        for (int r=0; r < 2; r++) {
            double result = matrix[r][dimension];
            for (int c=0; c < dimension; c++)
                result += src[c]*matrix[r][c];
            dest[r] = result;
        }
        return dest;
    }

    /**
     * Perform a rotation over the given angle around the origin in destination
     * space.
     */
    public void rotate(double angle) {
        if (angle == 0.0)
            return;
        rotateImpl(angle);
        fireTransformationChanged();
    }

    // performs the computations for rotate, but does not signal observers
    private void rotateImpl(final double angle) {
        double[][] result = new double[2][3];
        
        for (int i=0; i < 2; i++) {
            result[i][0] = matrix[i][0]*Math.cos(angle)
            - matrix[i][1]*Math.sin(angle);
            result[i][1] = matrix[i][0]*Math.sin(angle)
            + matrix[i][1]*Math.cos(angle);
            result[i][2] = matrix[i][2];
        }
        matrix = result;
    }    
    
    /**
     * Zoom by the given factor in destination space.
     */
    public void zoom (double factor) {
        if (factor != 1.0) {
            for (int r=0; r < 2; r++)
                for (int c=0; c < dimension; c++)
                    matrix[r][c] *= factor;
            fireTransformationChanged ();
        }
    }
    
    /**
     * Concatenate a translation (in destination space) with this transformation.
     * @param tx Delta x
     * @param ty Delta y
     */
    public void moveOrigin (double tx, double ty) {
        if (tx != 0.0 || ty != 0.0) {
            matrix[0][dimension] += tx;
            matrix[1][dimension] += ty;
            fireTransformationChanged ();
        }
    }
    
    /**
     * Is this transformation invertible?
     * @return true if invertible, false if not
     */
    @Override public boolean isInvertible () {
        if (matrix[0].length != 3)
            return false;
        double det = matrix[0][0]*matrix[1][1] - matrix[1][0]*matrix[0][1];
        return det != 0.0;
    }
    

    /**
     * Perform an inverse transformation from destination space back to
     * source space.<p>
     * @param x {@inheritDoc}
     * @param y {@inheritDoc}
     * @param dest {@inheritDoc}
     */
    @Override public void inverseTransform (double x, double y, double[] dest) throws IllegalStateException {
        if (matrix[0].length != 3)
            throw new IllegalStateException ("Transformation cannot be inverted");
        double det = matrix[0][0]*matrix[1][1] - matrix[1][0]*matrix[0][1];
        if (det == 0.0)
            throw new IllegalStateException ("Transformation cannot be inverted");
        x -= matrix[0][2];
        y -= matrix[1][2];
        dest[0] = (x*matrix[1][1] - y*matrix[0][1]) /det;
        dest[1] = (y*matrix[0][0] - x*matrix[1][0]) /det;
    }

    /**
     * Create an identity transform for two-dimensional space.
     */
    public AffineTransformation () {
        this.dimension = 2;
        this.matrix = new double[2][3];
        this.matrix[0][0] = 1.0;
        this.matrix[1][1] = 1.0;
    }

    /**
     * Create a new affine transformation with given transformation matrix.
     * @param matrix transformation matrix
     */
    public AffineTransformation (double[][] matrix) {
        if (matrix.length != 2)
            throw new IllegalArgumentException ("Transformation matrix should have 2 rows");
        if (matrix[1].length != matrix[0].length)
            throw new IllegalArgumentException ("Transformation matrix should be rectangular");
        this.dimension = matrix[0].length-1;
        this.matrix = new double[2][];
        this.matrix[0] = matrix[0].clone();
        this.matrix[1] = matrix[1].clone();        
    }
    
}
