/* VerticalIsometricGrid.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred.guides;

import java.awt.Graphics2D;
import java.awt.geom.Line2D;

/**
 * Guides based on a isometric grid with vertical lines. The number of
 * triangles in a column is configurable.
 */
public class VerticalIsometricGrid extends AbstractGuides {
    
    //
    private int numberOfTriangles;
    
    /**
     * Set the number of divisions used by this gridtriangles in a column.
     * @param numberOfTriangles Number of triangles in a column.
     *
     */
    public void setNumberOfTriangles(int numberOfTriangles) {
        if (numberOfTriangles < 1)
            throw new IllegalArgumentException("Illegal number of triangles");
        this.numberOfTriangles = numberOfTriangles;
    }
    
    /**
     * Create a new object of this type.
     * Set the number of divisions used by this gridtriangles in a column.
     * @param numberOfTriangles Number of triangles in a column.
     *
     */
    public VerticalIsometricGrid(int numberOfTriangles) {
        this.numberOfTriangles = numberOfTriangles;
    }
    
    /**
     * Paint the grid
     */
    public void paint(Graphics2D g2, double scale, double maxX, double maxY) {
        g2.setStroke(GUIDE_STROKE);
        g2.setPaint(GUIDE_COLOR);
        
        // vertical lines
        Line2D line = new Line2D.Double(); // reusable line
        for (int i = 0; i <= numberOfTriangles; i++) {
            double x = (-1.0 + 2.0*i/numberOfTriangles)*Math.cos(Math.PI/6)*scale;
            double y = (1.0 - (i%2)*1.0/numberOfTriangles)*scale;
            line.setLine(x, -y, x, y);
            g2.draw(line);
        }
        
        // diagonal lines
        int bound = numberOfTriangles - (numberOfTriangles/2) - (numberOfTriangles%2) + 1;
        for(int i = 0; i < bound; i++) {
            double x = Math.cos(Math.PI/6)*scale;
            double y = (-1.0 + 2.0*i/numberOfTriangles)*scale;
            line.setLine(-x,y,x,y+scale);
            g2.draw(line);
            line.setLine(-x,-y,x,-y-scale);
            g2.draw(line);
        }
        for(int i = bound; i < numberOfTriangles; i++){
            double x1 = -Math.cos(Math.PI/6)*scale;
            double x2 = (-1.0+2.0/numberOfTriangles*2*(i-bound+1))*Math.cos(Math.PI/6)*scale;
            double y = (-1.0 + (i - bound + 1)*2.0/numberOfTriangles)*scale;
            line.setLine(x1,y,x2,-scale);
            g2.draw(line);
            line.setLine(x1,-y,x2,scale);
            g2.draw(line);
            y = (1.0 + i - bound)/numberOfTriangles*scale*2;
            line.setLine(x2,scale,-x1,y);
            g2.draw(line);
            line.setLine(x2,-scale,-x1,-y);
            g2.draw(line);
        }
    }
    
    /**
     * Snap coordinates to the closest grid position.
     */
    public void snap(double[] coords) {
        // first snap X-coordinate to closest vertical line
        double x = coords[0];
        if (x < -Math.cos(Math.PI/6)*(1.0-1.0/numberOfTriangles)){
            x = 0;
            coords[0] = -Math.cos(Math.PI/6);
        } else if (x > Math.cos(Math.PI/6)*(1.0-1.0/numberOfTriangles)) {
            x = numberOfTriangles;
            coords[0] = Math.cos(Math.PI/6);
        } else {
            x = Math.floor((x/Math.cos(Math.PI/6)+1.0)*numberOfTriangles/2.0 + 0.5);
            coords[0] = (2.0*x/numberOfTriangles -1.0)*Math.cos(Math.PI/6);
        }
        // snap Y-coordinate to closest intersection
        double y = coords[1];
        if (y < -1.0 + (1 + x%2)/numberOfTriangles)
            coords[1] = -1.0 + (x%2)/numberOfTriangles;
        else if (y > 1.0 - (1 + x%2)/numberOfTriangles)
            coords[1] = 1.0 - (x%2)/numberOfTriangles;
        else {
            if(x%2==0.0){
                y = Math.floor((y+1.0)*numberOfTriangles/2.0 + 0.5);
                coords[1] = 2.0*y/numberOfTriangles -1.0;
            } else {
                y = Math.floor((y+1.0-1.0/numberOfTriangles)*(numberOfTriangles-1)/(2.0 - 2.0/numberOfTriangles) + 0.5);
                coords[1] = (2.0 - 2.0/numberOfTriangles)*y/(numberOfTriangles-1) - 1.0 + 1.0/numberOfTriangles;
            }
        }
    }
    
}
