/* GraphEditor.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred;

import be.ugent.caagt.swirl.commands.AttributedCommandManager;
import be.ugent.caagt.swirl.commands.ShowWindow;
import be.ugent.caagt.swirl.mouse.MouseTool;
import be.ugent.caagt.swirl.mouse.MouseToolCommand;
import be.ugent.caagt.swirl.undoredo.UndoManager;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Rectangle;
import java.io.InputStream;
import java.util.ResourceBundle;
import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;
import javax.swing.ListSelectionModel;

import org.grinvin.DefaultEmbeddingModel;
import org.grinvin.DefaultGraphModel;
import org.grinvin.Embedding;
import org.grinvin.EmbeddingView;
import org.grinvin.Graph;
import org.grinvin.MutableEmbeddingModel;
import org.grinvin.graphbrowser.GraphCellListComponent;
import org.grinvin.graphbrowser.HasGraphList;
import org.grinvin.gred.guides.GridLines;
import org.grinvin.gred.guides.GuidesTool;
import org.grinvin.gred.mouse.AddEdges;
import org.grinvin.gred.mouse.AddVertices;
import org.grinvin.gred.mouse.MoveSelection;
import org.grinvin.gred.mouse.RollOverOverlay;
import org.grinvin.gred.mouse.SplitEdges;
import org.grinvin.gred.mouse.SelectElements;
import org.grinvin.gred.mouse.SelectRectangle;
import org.grinvin.help.HelpManager;
import org.grinvin.list.GraphList;
import org.grinvin.list.GraphListModel;
import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.Preference;
import org.grinvin.render.AnnotatedRenderer;
import org.grinvin.render.Renderer;

import org.pietschy.command.LoadException;

/**
 * Graph editor window.
 */
public class GraphEditor extends JFrame implements HasGraphList {
    
    //
    private static final ResourceBundle BUNDLE =
            ResourceBundle.getBundle ("org.grinvin.gred.editor");
            
    //
    private GraphBundleWithModels bundle;
    
    //
    private GraphContext context;
    
    //
    private MutableGraphPanel panel;
    
    //
    private MutableEmbeddingModel embedding;
    
    //
    private AttributedCommandManager commandManager;
    
    //
    private GuidesTool guidesTool;
    
    //
    private GraphCellListComponent cellList;
    
    /**
     * Return the command manager of this editor window.
     */
    public AttributedCommandManager getCommandManager () {
        return commandManager;
    }
    
    // Returns the panel for this editor, mainly for testing purposes
    MutableGraphPanel getPanel () {
        return panel;
    }
    
    /**
     * Default constructor.
     */
    public GraphEditor (GraphListModel list) {
        this (new DefaultEmbeddingModel (new DefaultGraphModel (), 2), 
                new GraphContext (), list);
    }
    
    /**
     * Create an editor that uses the given embedding model and context.
     * This constructor is mainly useful when models and contexts should be
     * shared between editors. In most cases users should use the
     * no argument constructor.
     */
    public GraphEditor (MutableEmbeddingModel model, GraphContext context, GraphListModel list) {
        super (BUNDLE.getString ("editor.title"));
        this.embedding = model;
        
        setDefaultCloseOperation (HIDE_ON_CLOSE);
        
        Renderer renderer = new AnnotatedRenderer ();
        this.context = context;
        
        this.panel = new MutableGraphPanel (model, renderer, context, 140);
        panel.setGuides (new GridLines (5,5));
        
        RollOverOverlay roo = new RollOverOverlay (panel);
        
        // the order of these handlers is important
        MouseTool creationTool = new MouseTool ();
        creationTool.addHandler (AddEdges.getHandler ());
        creationTool.addHandler (SplitEdges.getHandler ());
        creationTool.addHandler (AddVertices.getHandler ());
        
        MouseTool selectionTool = new MouseTool ();
        selectionTool.addHandler (MoveSelection.getHandler ());
        selectionTool.addHandler (SelectElements.getHandler ());
        selectionTool.addHandler (SelectRectangle.getHandler ());
        
        commandManager = new AttributedCommandManager ();
        commandManager.setAttribute (GraphPanel.class.getName (), panel);
        commandManager.setAttribute (UndoManager.class.getName (), panel.getUndoManager ());
        commandManager.setResourceBundle (BUNDLE);
        commandManager.setAttribute (GraphList.class.getName (),list);
        commandManager.setAttribute(GraphEditor.class.getName(), this);
        
        int graphIconWidth = GrinvinPreferences.INSTANCE.getIntPreference(Preference.GRAPHICON_WIDTH);
        int graphIconHeigth = GrinvinPreferences.INSTANCE.getIntPreference(Preference.GRAPHICON_HEIGHT);
        this.cellList = new GraphCellListComponent (list, graphIconWidth, graphIconHeigth, 1);
        
        commandManager.setAttribute (ListSelectionModel.class.getName (),cellList.getSelectionModel());
        
        // background color of the cell list
        Color bg = panel.getBackground ();
        int red = Math.min (bg.getRed ()+15, 255);
        int green = Math.min (bg.getGreen ()+15, 255);
        int blue = Math.min (bg.getBlue ()+15, 255);
        cellList.setBackground (new Color (red, green, blue));
        
        
        try {
            InputStream ins = GraphEditor.class.getResourceAsStream ("editor.cmds");
            commandManager.load (ins);
        } catch (LoadException lex) {
            throw new RuntimeException ("Could not initialize menus", lex);
        }
        
        commandManager.getGroup ("toolbar").add (commandManager.getCommand ("Snapshot"));
        
        // mouse tools
        new MouseToolCommand (commandManager, "tool.selection", roo, selectionTool).export ();
        MouseToolCommand creationToolCommand = new MouseToolCommand (commandManager, "tool.creation", roo, creationTool);
        creationToolCommand.export ();
        creationToolCommand.setSelected (true);

        add (panel);
        
        JToolBar toolbar = commandManager.getGroup ("toolbar").createToolBar ();
        add (toolbar, BorderLayout.NORTH);
        
        JToolBar operationToolbar = commandManager.getGroup ("operation.toolbar").createToolBar ();
        operationToolbar.setOrientation(JToolBar.VERTICAL);
        add (operationToolbar, BorderLayout.WEST);
        
        add (new JScrollPane (cellList), BorderLayout.EAST);
        
        setJMenuBar (commandManager.getGroup ("main.menu").createMenuBar ());
        
        pack ();
        
        // create and position guides tool
        guidesTool = new GuidesTool (this, panel, BUNDLE);
        Rectangle rect = getBounds ();
        int x = rect.x + rect.width - guidesTool.getWidth ()/2;
        int y = rect.y + rect.height - guidesTool.getHeight ()/2;
        guidesTool.setLocation (x, y);
        
        new ShowWindow (commandManager, "showGuidesTool", guidesTool).export ();
        
        // help
        HelpManager.setHelpIDString(this, "org.grinvin.window.editor");
        HelpManager.setHelpIDString(toolbar, "org.grinvin.window.editor.toolbar");
        HelpManager.setHelpIDString(commandManager.getCommand("Snapshot").getButtonIn(this), "org.grinvin.window.editor.snapshot");
        HelpManager.setHelpIDString(commandManager.getCommand("tool.selection").getButtonIn(this), "org.grinvin.window.editor.selection");
        HelpManager.setHelpIDString(commandManager.getCommand("tool.creation").getButtonIn(this), "org.grinvin.window.editor.creation");
        HelpManager.setHelpIDString(commandManager.getCommand("undo").getButtonIn(this), "org.grinvin.window.editor.undo");
        HelpManager.setHelpIDString(commandManager.getCommand("redo").getButtonIn(this), "org.grinvin.window.editor.redo");
        HelpManager.setHelpIDString(commandManager.getCommand("AutoSnap").getButtonIn(this), "org.grinvin.window.editor.autosnap");
        HelpManager.setHelpIDString(commandManager.getCommand("SnapSelection").getButtonIn(this), "org.grinvin.window.editor.snapselection");
        HelpManager.setHelpIDString(commandManager.getCommand("SnapAll").getButtonIn(this), "org.grinvin.window.editor.snapall");
        HelpManager.setHelpIDString(commandManager.getCommand("Clear").getButtonIn(this), "org.grinvin.window.editor.clear");
        HelpManager.setHelpIDString(commandManager.getCommand("SpringEmbedder").getButtonIn(this), "org.grinvin.window.editor.springembedder");
        HelpManager.setHelpIDString(commandManager.getGroup("file.menu").getButtonIn(this), "org.grinvin.window.editor.menu.file");
        HelpManager.setHelpIDString(commandManager.getGroup("edit.menu").getButtonIn(this), "org.grinvin.window.editor.menu.edit");
        HelpManager.setHelpIDString(commandManager.getGroup("selection.menu").getButtonIn(this), "org.grinvin.window.editor.menu.selection");
        HelpManager.setHelpIDString(commandManager.getGroup("guides.menu").getButtonIn(this), "org.grinvin.window.editor.menu.guides");
        HelpManager.setHelpIDString(operationToolbar, "org.grinvin.window.editor.operationtoolbar");
        HelpManager.setHelpIDString(commandManager.getCommand("MergeSelection").getButtonIn(this), "org.grinvin.window.editor.mergeselection");
        HelpManager.setHelpIDString(commandManager.getCommand("SplitSelection").getButtonIn(this), "org.grinvin.window.editor.splitselection");
        HelpManager.setHelpIDString(commandManager.getCommand("ConnectSelection").getButtonIn(this), "org.grinvin.window.editor.connectselection");
        HelpManager.setHelpIDString(commandManager.getCommand("VertexInducedGraph").getButtonIn(this), "org.grinvin.window.editor.vertexinducedgraph");
        HelpManager.setHelpIDString(commandManager.getCommand("EdgeInducedGraph").getButtonIn(this), "org.grinvin.window.editor.edgeinducedgraph");
        HelpManager.setHelpIDString(commandManager.getCommand("Complement").getButtonIn(this), "org.grinvin.window.editor.complement");
        HelpManager.setHelpIDString(commandManager.getCommand("Closure").getButtonIn(this), "org.grinvin.window.editor.closure");
        HelpManager.setHelpIDString(commandManager.getCommand("LineGraph").getButtonIn(this), "org.grinvin.window.editor.linegraph");
        HelpManager.setHelpIDString(guidesTool, "org.grinvin.window.editor.guides");
        HelpManager.enableHelpKey(this.getRootPane(), "org.grinvin.window.editor");
    }
    
    /**
     * Load (a copy of) the given embedding into the editor.
     */
    public void setEmbedding (EmbeddingView newEmbedding) {
        panel.loadEmbedding (newEmbedding);
    }
    /**
     * Get the embedding for the current graph in the editor.
     */
    public Embedding getEmbedding() {
        return panel.getEmbedding();
    }
    
    /**
     * Get the current graph in the editor.
     */
    public Graph getGraph() {
        return panel.getGraph();
    }

    public GraphList getGraphList() {
        return cellList.getGraphList();
    }
    
    /**
     * Set the 'dirty' flag of the editor.
     */
    public void setDirty(boolean dirty) {
        panel.setDirty(dirty);
    }
    
    /**
     * Check whether the editor is 'dirty'.
     */
    public boolean isDirty() {
        return panel.isDirty();
    }
    
    /**
     * Get the guidestool for this editor.
     */
    public GuidesTool getGuidesTool() {
        return guidesTool;
    }
}
