/* RowHeaderView.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.graphbrowser;

import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ResourceBundle;
import javax.swing.AbstractAction;
import javax.swing.JList;
import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.KeyStroke;

import org.grinvin.list.DefaultGraphList;
import org.grinvin.list.GraphList;
import org.grinvin.list.GraphListElement;
import org.grinvin.list.GraphListModel;
import org.grinvin.list.GraphListRenderer;
import org.grinvin.list.GraphPropertiesWindow;
import org.grinvin.list.commands.GraphRenamePanel;

/**
 * List that is used as row header of a {@link GraphTableComponent}. Provides
 * drag and drop support, rename and delete.
 */
public class RowHeaderView extends JList implements HasSelectableGraphList, MouseListener {
    
    //
    private GraphListModel model;
    
    //
    private static final ResourceBundle BUNDLE = ResourceBundle.getBundle("org.grinvin.graphbrowser.resources");
    
    
    /** Creates a new instance of RowHeaderView */
    public RowHeaderView(GraphListModel model, JTable table) {
        super(model);
        this.model = model;
        setFixedCellHeight(table.getRowHeight());
        setCellRenderer(new GraphListRenderer(table));
        setSelectionModel(table.getSelectionModel());
        setTransferHandler(CommonGraphListTransferHandler.TRANSFER_HANDLER);
        setDragEnabled(true);
        
        // Keyboard interaction
        getInputMap(WHEN_FOCUSED).put(KeyStroke.getKeyStroke("DELETE"), "deleteSelectedRows");
        getActionMap().put("deleteSelectedRows", new AbstractAction() {
            public void actionPerformed(ActionEvent ev) {
                deleteSelectedGraphs();
            }
        });
        getInputMap(WHEN_FOCUSED).put(KeyStroke.getKeyStroke("F2"), "startEditing");
        getActionMap().put("startEditing", new AbstractAction() {
            public void actionPerformed(ActionEvent ev) {
                int index = RowHeaderView.this.getSelectionModel().getMinSelectionIndex();
                if (index < 0)
                    return; // none selected
                if (index != RowHeaderView.this.getSelectionModel().getMaxSelectionIndex())
                    return; // more than one selection
                GraphListElement gle = getGraphList().get(index);
                
                if (gle.isNameEditable()) {
                    new GraphRenamePanel().showDialog(gle, RowHeaderView.this);
                }
            }
        });
        addMouseListener(this);
    }
    
    public GraphList getGraphList() {
        return model;
    }
    
    /**
     * Delete currently selected graphs.
     */
    public void deleteSelectedGraphs() {
        int[] indices = getSelectedIndices();
        
        GraphList graphList = getGraphList();
        GraphList list = new DefaultGraphList();
        
        for(int i : indices)
            list.add(graphList.get(i));
        
        graphList.removeAll(list);
    }
    
    public void mouseClicked(MouseEvent e) {
        final GraphListElement gle = (GraphListElement)(RowHeaderView.this.getSelectedValue());
        if (gle == null)
            return;
        if (e.getClickCount() == 2) {
            new GraphPropertiesWindow(gle).setVisible(true);
        } else if (e.getButton() == MouseEvent.BUTTON3) {
            JPopupMenu menu = new JPopupMenu();
            menu.add(new AbstractAction(BUNDLE.getString("Popupmenu.properties")) {
                public void actionPerformed(ActionEvent ev) {
                    new GraphPropertiesWindow(gle).setVisible(true);
                }
            });
            menu.add(new AbstractAction(BUNDLE.getString("Popupmenu.rename")) {
                public void actionPerformed(ActionEvent ev) {
                    if (gle.isNameEditable()) {
                        new GraphRenamePanel().showDialog(gle, RowHeaderView.this);
                    }
                }
                public boolean isEnabled() {
                    return gle.isNameEditable();
                }
            });
            menu.show(this,e.getX(),e.getY());
        }
    }
    
    public void mousePressed(MouseEvent e) {
    }
    
    public void mouseReleased(MouseEvent e) {
    }
    
    public void mouseEntered(MouseEvent e) {
    }
    
    public void mouseExited(MouseEvent e) {
    }
    
}
