/* GraphListComponent.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.graphbrowser;

import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ResourceBundle;
import javax.swing.AbstractAction;
import javax.swing.DefaultListSelectionModel;
import javax.swing.JList;
import javax.swing.JPopupMenu;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;

import org.grinvin.list.DefaultGraphList;
import org.grinvin.list.GraphList;
import org.grinvin.list.GraphListElement;
import org.grinvin.list.GraphListElementRenderer;
import org.grinvin.list.GraphListModel;
import org.grinvin.list.GraphPropertiesWindow;
import org.grinvin.list.commands.GraphRenamePanel;

/**
 *
 * @author adpeeter
 */
public class GraphListComponent extends JList implements HasSelectableGraphList, MouseListener {

    //
    private GraphListModel model;
    
    //
    private static final ResourceBundle BUNDLE = ResourceBundle.getBundle ("org.grinvin.graphbrowser.resources");
    
    //
    public GraphListComponent(GraphListModel model) {
        this(model, null);
    }
    
    //
    public GraphListComponent(GraphListModel model, ListSelectionModel selectionModel) {
        super(model);
        if (selectionModel == null)
            setSelectionModel(createListSelectionModel());
        else
            setSelectionModel(selectionModel);
        
        this.model = model;
        
        setDragEnabled(true);
        setTransferHandler(CommonGraphListTransferHandler.TRANSFER_HANDLER);
        setCellRenderer(new GraphListElementRenderer());
        
        // Keyboard interaction
        getInputMap(WHEN_FOCUSED).put(KeyStroke.getKeyStroke("DELETE"), "deleteSelectedElements");
        getActionMap().put("deleteSelectedElements", new AbstractAction() {
            public void actionPerformed(ActionEvent ev) {
                deleteSelectedElements();
            }
        });
        
        // mouse interaction
        addMouseListener(this);

    }
    
    //
    protected ListSelectionModel createListSelectionModel() {
        return new DefaultListSelectionModel();
    }
    
    //
    public GraphList getGraphList() {
        return model;
    }
    
     /* ============================================================
      * ACTIONS
      * ============================================================ */
    
    /**
     * Delete currently selected elements.
     */
    public void deleteSelectedElements() {
        // iterate over selected indices
        ListSelectionModel selectionModel = getSelectionModel();
        
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin)
            return ; // no selection
        
        // gather all elements to be deleted
        GraphList list = new DefaultGraphList();
        for(int i = iMax; i >= iMin; i--)
            if (selectionModel.isSelectedIndex(i))
                list.add(model.get(i));
        
        // and remove them
        model.removeAll(list);
        //TODO: make sure changed messages are sent
    }
    
    //
    public void mouseClicked(MouseEvent e) {
        final GraphListElement gle = (GraphListElement)(GraphListComponent.this.getSelectedValue());
        if (gle == null)
            return;
        if (e.getClickCount() == 2) {
            new GraphPropertiesWindow(gle).setVisible(true);
        } else if (e.getButton() == MouseEvent.BUTTON3) {
            JPopupMenu menu = new JPopupMenu();
            menu.add(new AbstractAction(BUNDLE.getString("Popupmenu.properties")) {
               public void actionPerformed(ActionEvent ev) {
                   new GraphPropertiesWindow(gle).setVisible(true);
               } 
            });
            menu.add(new AbstractAction(BUNDLE.getString("Popupmenu.rename")) {
                public void actionPerformed(ActionEvent ev) {
                    if (gle.isNameEditable()) {
                        new GraphRenamePanel().showDialog(gle, GraphListComponent.this);
                    }
                }
                public boolean isEnabled() {
                    return gle.isNameEditable();
                }
            });
            menu.show(this,e.getX(),e.getY());
        }
    }

    public void mousePressed(MouseEvent e) {
    }

    public void mouseReleased(MouseEvent e) {
    }

    public void mouseEntered(MouseEvent e) {
    }

    public void mouseExited(MouseEvent e) {
    }
    
}
