/* ConjecturingWindow.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.graphbrowser;

import be.ugent.caagt.swirl.commands.AttributedCommandManager;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.io.InputStream;
import java.util.ResourceBundle;
import javax.swing.AbstractButton;
import javax.swing.AbstractListModel;
import javax.swing.ComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListDataEvent;

import org.grinvin.engine.APEngine;
import org.grinvin.engine.EngineRunner;
import org.grinvin.engine.EngineRunner.Status;
import org.grinvin.help.HelpManager;
import org.grinvin.invariants.Invariant;
import org.grinvin.list.GraphInvariantListModel;
import org.grinvin.list.GraphList;
import org.grinvin.list.InvariantList;
import org.grinvin.list.InvariantListModelListener;

import org.pietschy.command.LoadException;

/**
 * Window associated to a conjecturing engine. Consists of
 * <ul>
 * <li>An invariant table which contains the data to be used by the engine.</li>
 * <li>A status and configuration panel for the engine.</li>
 * <li>A history panel displaying the generated conjectures.</li>
 * </ul>
 *
 */
public class ConjecturingWindow extends JFrame 
        implements HasGraphList, HasInvariantList {
    
    //
    private ResourceBundle rbundle;
        
    //
    private GraphInvariantListModel model;
    
    //
    private EngineRunner runner;
    
    //
    public EngineRunner getEngineRunner() {
        return runner;
    }
    
    //
    private JTextArea textArea;
    
    //
    private JLabel statusLabel;
    
    //
    private BoxModel boxModel;
    
    //
    private GraphBrowserMultiPanel multipanel;
    
    //
    private AttributedCommandManager commandManager;
    
    //
    private JSplitPane sp;
    
    //
    public Invariant getSelectedInvariant() {
        return boxModel.getSelectedItem();
    }
    
    //
    public int getUsableInvariantsCount(){
        return boxModel.getSize();
    }

    //
    public GraphList getGraphList() {
        return model.getGraphList();
    }

    public InvariantList getInvariantList() {
        return model.getInvariantList();
    }
    
    //
    public void addResult(String result) {
        textArea.append(result + "\n");
    }
    
    //
    public void setStatus(Status status) {
        statusLabel.setText(rbundle.getString("enginestatus."+status));
    }
    
    //
    public ConjecturingWindow(GraphInvariantListModel list) {
        this.rbundle = ResourceBundle.getBundle("org.grinvin.graphbrowser.conjecturingwindow");
        this.model = list;
        this.runner = new EngineRunner (new APEngine(), model);
        //this.runner = new EngineRunner(new QDEngine(), model);
        boxModel = new BoxModel(); //create boxModel early (at least before menu's are build)
        
        setTitle(rbundle.getString("conjecturingwindow.title"));
        //model.addGraphListModelListener (this);
        
        setLayout(new BorderLayout());
        
        multipanel = new GraphBrowserMultiPanel(model);
        multipanel.setPreferredSize(new Dimension(500,200));
        
        JPanel bottomPanel = new JPanel(new BorderLayout());
        
        textArea = new JTextArea();
        textArea.setEditable(false);
        JScrollPane scrollPane = new JScrollPane(textArea);
        scrollPane.setPreferredSize(new Dimension(300,100));
        
        statusLabel = new JLabel(rbundle.getString("enginestatus." + Status.STOPPED));
        statusLabel.setHorizontalAlignment(JLabel.TRAILING);
        
        bottomPanel.add(scrollPane, BorderLayout.CENTER);
        
        sp = new JSplitPane(JSplitPane.VERTICAL_SPLIT, multipanel, bottomPanel);
        getContentPane().add(sp, BorderLayout.CENTER);
        
        commandManager = new AttributedCommandManager();
        commandManager.setAttribute(ConjecturingWindow.class.getName(), this);
        commandManager.setAttribute(GraphBrowserMultiPanel.class.getName(), multipanel);
        commandManager.setResourceBundle(rbundle);
        
        try {
            InputStream ins = ConjecturingWindow.class.getResourceAsStream("conjecturingwindow.cmds");
            commandManager.load(ins);
        } catch (LoadException lex) {
            throw new RuntimeException("Could not initialize menus", lex);
        }
        
        //add (commandManager.getGroup ("toolbar").createToolBar (), BorderLayout.NORTH);
        
        JPanel configPanel = new JPanel(new GridBagLayout());
        bottomPanel.add(configPanel, BorderLayout.NORTH);
        GridBagConstraints gbc = new GridBagConstraints();
        
        gbc.insets = new Insets(5,10,5,10);
        gbc.anchor = GridBagConstraints.WEST;
        gbc.fill = GridBagConstraints.NONE;
        gbc.gridheight = 2;
        
        configPanel.add(new JLabel(rbundle.getString("config.maininvariant") + ": "), gbc);
        
        gbc.weightx = 1.0;
        gbc.gridx = 1;
        JComboBox comboBox = new JComboBox(boxModel);
        comboBox.setPrototypeDisplayValue("Maximum degree ...");
        configPanel.add(comboBox, gbc);
        
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 0.0;
        gbc.gridheight = 1;
        gbc.gridx = 3;
        gbc.anchor = GridBagConstraints.EAST;
        configPanel.add(new ProgressBar(), gbc);
        
        gbc.fill = GridBagConstraints.NONE;
        gbc.weightx = 1.0;
        gbc.gridy = 1;
        gbc.gridx = 2;
        gbc.gridwidth = 2;
        configPanel.add(statusLabel, gbc);
        
        gbc.weightx = 0.0;
        gbc.gridy = 0;
        gbc.gridheight = 2;
        gbc.gridwidth = 1;
        gbc.gridx = 4;
        AbstractButton button = commandManager.getCommand("Conjecture").createButton();
        configPanel.add(button, gbc);
        
        //make sure the InvariantTable view is the default view
        setCurrentComponent("graphInvariantTable");
        
        setJMenuBar(commandManager.getGroup("main.menu").createMenuBar());
        pack();

        //set divider in center (has to be done after pack())
        setInvariantTableDividerLocation(multipanel.getHeight()/2);
        
        // help
        HelpManager.setHelpIDString(this, "org.grinvin.window.conjecturing");
        HelpManager.setHelpIDString(button, "org.grinvin.window.conjecturing.conjecturebutton");
        HelpManager.setHelpIDString(comboBox, "org.grinvin.window.conjecturing.maininvariant");
        HelpManager.setHelpIDString(commandManager.getGroup("menu.file").getButtonIn(this),"org.grinvin.window.conjecturing.menu.file");
        HelpManager.setHelpIDString(commandManager.getGroup("menu.edit").getButtonIn(this),"org.grinvin.window.conjecturing.menu.edit");
        HelpManager.setHelpIDString(commandManager.getGroup("toolbar.view").getButtonIn(this),"org.grinvin.window.conjecturing.menu.view");
        HelpManager.enableHelpKey(this.getRootPane(), "org.grinvin.window.conjecturing");
        
    }
    
    public String getCurrentComponent() {
        return multipanel.getCurrentComponent();
    }
    
    public void setCurrentComponent(String component) throws RuntimeException {
        // TODO: the above should be solved in a different way
        if ("graphInvariantTable".equals(component)) {
            commandManager.getCommand("view.invarianttable").execute();
        } else if ("graphCellList".equals(component)) {
            commandManager.getCommand("view.graphcellist").execute();
        } else if ("graphList".equals(component)) {
            commandManager.getCommand("view.graphlist").execute();
        } else {
            throw new RuntimeException("invalid component");
        }
    }

    public void setDividerLocation(int location){
        sp.setDividerLocation(location);
    }
    
    public int getDividerLocation(){
        return sp.getDividerLocation();
    }
    
    public void setInvariantTableDividerLocation(int location){
        multipanel.setInvariantTableDividerLocation(location);
    }
    
    public int getInvariantTableDividerLocation(){
        return multipanel.getInvariantTableDividerLocation();
    }
    
    public void setGraphListDividerLocation(int location){
        multipanel.setGraphListDividerLocation(location);
    }
    
    public int getGraphListDividerLocation(){
        return multipanel.getGraphListDividerLocation();
    }

    public void setGraphPropertiesComponentDividerLocation(int location){
        multipanel.setGraphPropertiesComponentDividerLocation(location);
    }
    
    public int getGraphPropertiesComponentDividerLocation(){
        return multipanel.getGraphPropertiesComponentDividerLocation();
    }
    
    public void addConjecturingWindowListener(ConjecturingWindowListener l){
        model.getInvariantListModel().addInvariantListModelListener(l);
        boxModel.addListDataListener(l);
    }

    public void removeConjecturingWindowListener(ConjecturingWindowListener l){
        model.getInvariantListModel().removeInvariantListModelListener(l);
        boxModel.removeListDataListener(l);
    }

    //
    private class ProgressBar extends JProgressBar implements ChangeListener {
        
        public ProgressBar() {
            super();
            runner.addChangeListener(this);
            setPreferredSize(new Dimension(60, getPreferredSize().height));
        }
        
        public void stateChanged(ChangeEvent e) {
            if(runner.getStatus() != Status.STOPPED) {
                try {
                    Thread.sleep(500);
                    setIndeterminate(true);
                } catch (InterruptedException ex) {
                    //TODO: document this
                }
            } else {
                setIndeterminate(false);
            }
        }
        
    }
    
    /**
     * Model for the combo box. Delegates to the invariant table model.
     */
    private class BoxModel extends AbstractListModel implements ComboBoxModel, InvariantListModelListener {
        
        //
        public BoxModel() {
            model.getInvariantListModel().addInvariantListModelListener(this);
        }
        
        //
        private Invariant selected;
        
        //
        public Invariant getSelectedItem() {
            return selected;
        }

        //
        public void setSelectedItem(Object anItem) {
            if (anItem == selected)
                return;
            else if (anItem instanceof Invariant) {
                selected = (Invariant)anItem;
                fireContentsChanged(this, -1, -1);
            } else if (selected != null) {
                selected = null;
                fireContentsChanged(this, -1, -1);
            }
        }
        
        public Object getElementAt(int index) {
            return runner.getUsableInvariants().get(index);
        }
        
        public int getSize() {
            return runner.getUsableInvariants().size();
        }
        
        private void invariantListModelChanged() {
            fireContentsChanged(this, 0, getSize());
            if (runner.getUsableInvariants().indexOf(selected) < 0) {
                setSelectedItem(null);
            }
        }
        
        //
        public void intervalAdded(ListDataEvent e) {
            invariantListModelChanged();
        }
        
        //
        public void intervalRemoved(ListDataEvent e) {
            invariantListModelChanged();
        }
        
        //
        public void contentsChanged(ListDataEvent e) {
            invariantListModelChanged();
        }
        
    }
    
}
