/* GraphFactory.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.factories.graphs;

import java.net.URI;

import org.grinvin.GraphBundle;
import org.grinvin.factories.Factory;
import org.grinvin.factories.FactoryException;
import org.grinvin.factories.FactoryParameterException;

/**
 * An object that satisfies this interface can be used to create a graph
 * and store it into a graph bundle. To have a factory create a graph, first
 * set its parameters using {@link #setParameterValues} and
 * then call {@link #createGraph}.
 */
public interface GraphFactory extends Factory {
    
    /**
     * Create a graph and store it into the given graph bundle.
     * @return an URI which represents the graph being created.
     * @throws FactoryParameterException when the parameters for this graph
     * were not set in an appropriate manner before this method was called.
     */
    public URI createGraph(GraphBundle graphBundle) throws FactoryParameterException, FactoryException;
    
    /**
     * Return the URI of the graph which will be created
     * from this factory when using the current parameter values.
     */
    public URI getURI() throws FactoryParameterException;
    
    /**
     * Return the URI of the factory icon when the factory is not selected.
     */
    public String getIconPath();
    
    /**
     * Return the URI of the factory icon when the factory is selected.
     */
    public String getSelectedIconPath();
    
    /**
     * Return the string that uniquely identifies this factory. This id
     * is typically used for retreiving documentation pages for this factory
     * and for building URIs for the graphs created by this factory. 
     */
    public String getId ();
    
    /**
     * Return the internationalized name of the factory.
     */
    public String getName();
    
    /**
     * Return the internationalized description of the factory.
     */
    public String getDescription();

    /**
     * Return the internationalized name of the factory.
     */
    public String getLongName();
    
}
