/* AbstractGraphFactory.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.factories.graphs;

import java.io.IOException;
import java.lang.reflect.Method;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.MessageFormat;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.Icon;

import org.grinvin.Embedding;
import org.grinvin.Graph;
import org.grinvin.GraphBundle;
import org.grinvin.GraphURI;
import org.grinvin.Vertex;
import org.grinvin.factories.AbstractFactory;
import org.grinvin.factories.FactoryException;
import org.grinvin.factories.FactoryParameterException;
import org.grinvin.gui.icons.DefaultGraphIconFactory;
import org.grinvin.gui.icons.GraphIconFactory;
import org.grinvin.gui.icons.SvgIconManager;
import org.grinvin.params.ParameterList;
import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.Preference;
import org.grinvin.util.InternationalizedProperties;
import org.grinvin.xml.XMLUtils;
import org.jdom.Element;

/**
 * Common superclass of various graph factory classes. Provides support for
 * internationalization and some helper methods.
 */
abstract public class AbstractGraphFactory extends AbstractFactory implements VisualGraphFactory {
    
    /**
     * Default constructor.
     */
    public AbstractGraphFactory() {
        initId ();
        load(getClass().getName().replace('.', '/') + ".xml");
    }
    
    /**
     * Initialize name and description for this graph from the resource bundle.
     * @param iprops Properties object into which the results will be stored
     */
    protected void initProperties(InternationalizedProperties iprops) {
        
        // name
        String nameFormat = factoryProps.getProperty("graph.name");
        String name = MessageFormat.format(nameFormat, (Object[])valueStrings);
        iprops.setProperty("graph.name", name);
        
        // description
        String descFormat = factoryProps.getProperty("graph.description");
        String desc = MessageFormat.format(descFormat, (Object[])valueStrings);
        if (desc != null)
            iprops.setProperty("graph.description", desc);
        
    }
    
    /**
     * Initialize the graph icon factory.
     */
    private void initGraphIconFactory(GraphBundle bundle, String iconFactory) {
        GraphIconFactory gif = DefaultGraphIconFactory.getInstance();
        if (iconFactory != null) {
            try {
                Class<?> clazz = Class.forName(iconFactory);
                Method method = clazz.getMethod("getInstance");
                gif = (GraphIconFactory)method.invoke(null); // static method
                bundle.setGraphIconFactory(gif);
            } catch (Exception ex) {
                LOGGER.log(Level.WARNING, "unable to instantiate graph icon factory", ex);
            }
        }
        bundle.setGraphIconFactory(gif);
    }
    
    //
    private static final Logger LOGGER = Logger.getLogger
            ("org.grinvin.io", "org.grinvin.grinvin.resources");
    
    //
    private InternationalizedProperties factoryProps;
    
    //
    private String factoryDescription;
    
    //
    private String factoryName;
    
    //
    private String factoryLongName;
    
    //
    private String iconPath;
    
    //
    private String selectedIconPath;
    
    /**
     * Load factory information from the description in the given XML-resource.
     */
    private void load(String resource) {
        try {
            Element root = XMLUtils.loadFromClassPath(resource);
            
            // resources
            Element el = root.getChild("i18n");
            if (el == null)
                throw new RuntimeException("Invalid input format: cannot find 'i18n' element");
            factoryProps = new InternationalizedProperties();
            factoryProps.fromElement(el);
            
            // parameters
            list = new ParameterList();
            list.fromElement(root);
            
            // cell properties
            factoryName = factoryProps.getProperty("factory.text");
            if (factoryName == null)
                factoryName = factoryProps.getProperty("factory.name");
            factoryLongName = factoryProps.getProperty("factory.name");
            factoryDescription = factoryProps.getProperty("factory.description");
            if (factoryDescription == null)
                factoryDescription = factoryName;
            
            el = root.getChild("icon");
            if (el != null)
                iconPath = el.getText();
            el = root.getChild("selectedIcon");
            if (el != null)
                selectedIconPath = el.getText();
            if (selectedIconPath == null)
                selectedIconPath = iconPath;
        } catch (IOException ex) {
            throw new RuntimeException("Invalid input format", ex);
        }
    }
    
    // implements GraphFactory
    public URI createGraph(GraphBundle graphBundle) throws FactoryParameterException, FactoryException {
        InternationalizedProperties iprops = new InternationalizedProperties();
        graphBundle.setProperties(iprops);
        initProperties(iprops);
        initGraphIconFactory(graphBundle, factoryProps.getProperty("graphIconFactory"));
        createGraph(graphBundle.createGraph(), graphBundle.createEmbedding());
        return getURI();
    }
    
    //
    public URI getURI() {
        String ssp = getId() + getParametersAsString();
        try {
            return GraphURI.createFactory(ssp);
        } catch (URISyntaxException ex) {
            throw new RuntimeException("Factory generates incorrect URI", ex);
        }
    }
    
    /**
     * Should be overridden to initialize graph and embedding.
     */
    protected abstract void createGraph(Graph graph, Embedding embedding) throws FactoryException;
    
    /**
     * Create a graph and embedding with vertices spaced
     * equally along the circumference of the unit circle.
     * @param graph graph to which the given vertices will be assigned
     * @param embedding embedding which will hold the coordinates
     * @param nr number of vertices to create
     * @return The array of vertices created
     */
    protected static Vertex[] createCycle(Graph graph, Embedding embedding, int nr) {
        embedding.setDimension(2);
        Vertex[] result = new Vertex[nr];
        for(int i = 0; i < nr; i++) {
            double angle = 2 * i * Math.PI / nr;
            double[] coords = new double[2];
            coords[0] = Math.cos(angle);
            coords[1] = Math.sin(angle);
            result[i] = graph.addNewVertex(null);
            embedding.setCoordinates(result[i], coords);
        }
        return result;
    }
    
    /**
     * Create a graph amd embedding with vertices spaced equally along the
     * interval -1..1.
     * @param graph graph to which the given vertices will be assigned
     * @param embedding embedding which will hold the coordinates
     * @param length number of vertices to create
     * @return The array of vertices created
     */
    protected static Vertex[] createPath(Graph graph, Embedding embedding, int length) {
        embedding.setDimension(2);
        Vertex[] result = new Vertex[length+1];
        for (int i = 0; i <= length; i++) {
            double[] coords = new double[2];
            coords[0] = 2.0*i/length - 1.0;
            coords[1] = coords[0]; //0.0;
            result[i] = graph.addNewVertex(null);
            embedding.setCoordinates(result[i], coords);
        }
        return result;
    }
    
    // implements VisualFactory
    public Icon getIcon() {
        int size = Math.min(GrinvinPreferences.INSTANCE.getIntPreference(Preference.GRAPHICON_WIDTH),
                GrinvinPreferences.INSTANCE.getIntPreference(Preference.GRAPHICON_HEIGHT) - 15) - 10;
        return SvgIconManager.getInstance().getIcon(getIconPath(), size);
    }
    
    public String getIconPath() {
        return iconPath;
    }
    
    public String getSelectedIconPath() {
        return selectedIconPath;
    }
    
    public String getName() {
        return factoryName;
    }
    
    public String getLongName() {
        return factoryLongName;
    }
    
    public String getDescription() {
        return factoryDescription;
    }
    
    //
    private String factoryId;
    
    /**
     * The id of a graph factory is usually the same as its fully qualified class name.
     * For reasons of backwards compatibility however, ids of a class in one of the subpackages
     * <code>org.grinvin.factories.graphs.chemical</code> or 
     * <code>org.grinvin.factories.graphs.standard</code> are given an idea as if they resided
     * in the package <code>org.grinvin.factories</code>.
     */
    public String getId() {
        return factoryId;
    }
    
    // 
    private void initId() {
        String id = getClass().getName ();
        int pos = id.lastIndexOf('.');
        if (pos >= 0) {
            String pkg = id.substring(0, pos);
            if ("org.grinvin.factories.graphs.chemical".equals(pkg) ||
                    "org.grinvin.factories.graphs.standard".equals(pkg)) {
                this.factoryId = "org.grinvin.factories" + id.substring(pos);
                return;
            }
        }
        this.factoryId = id;
    }
    
}
