/****************************************************************************
 **
 ** Copyright (C) 1999-2005 The LinCVS development team.
 **    Tilo Riemer <riemer@lincvs.org>
 **    Falk Brettschneider <gigafalk@yahoo.com>
 **    Frank Hemer <frank@hemer.org>
 **    Wim Delvaux <wim.delvaux@chello.be>
 **    Jose Hernandez <joseh@tesco.net>
 **    Helmut Koll <HelmutKoll@web.de>
 **    Tom Mishima <tmishima@mail.at-m.or.jp>
 **    Joerg Preiss <auba@auba.de>
 **    Sven Trogisch <trogisch@iapp.de>
 **
 **
 **----------------------------------------------------------------------------
 **
 **----------------------------------------------------------------------------
 **
 ** LinCVS is available under two different licenses:
 **
 ** If LinCVS is linked against the GPLed version of Qt 
 ** LinCVS is released under the terms of GPL also.
 **
 ** If LinCVS is linked against a nonGPLed version of Qt 
 ** LinCVS is released under the terms of the 
 ** LinCVS License for non-Unix platforms (LLNU)
 **
 **
 ** LinCVS License for non-Unix platforms (LLNU):
 **
 ** Redistribution and use in binary form, without modification, 
 ** are permitted provided that the following conditions are met:
 **
 ** 1. Redistributions in binary form must reproduce the above copyright
 **    notice, this list of conditions and the following disclaimer in the
 **    documentation and/or other materials provided with the distribution.
 ** 2. It is not permitted to distribute the binary package under a name
 **    different than LinCVS.
 ** 3. The name of the authors may not be used to endorse or promote
 **    products derived from this software without specific prior written
 **    permission.
 ** 4. The source code is the creative property of the authors.
 **    Extensions and development under the terms of the Gnu Public License
 **    are limited to the Unix platform. Any distribution or compilation of 
 **    the source code against libraries licensed other than gpl requires 
 **    the written permission of the authors.
 **
 **
 ** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
 ** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 ** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 ** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
 ** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 ** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
 ** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 ** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 ** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 ** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **
 **
 **
 ** LinCVS License for Unix platforms:
 **
 ** This program is free software; you can redistribute it and/or modify 
 ** it under the terms of the GNU General Public License as published by 
 ** the Free Software Foundation; either version 2 of the License, or 
 ** (at your  option) any later version.  This program is distributed in 
 ** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
 ** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 ** PARTICULAR PURPOSE.
 **
 ** See the GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software Foundation,
 ** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **
 *****************************************************************************/

#include "config.h"

//----------------------------------------------------------------------------

#include <assert.h>
#include <qaction.h>
#include <qtextcodec.h>
#include <qtabwidget.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qtextedit.h>
#include <qinputdialog.h>
#include <qkeycode.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qaccel.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qstringlist.h>
#include <qstatusbar.h>
#include <qregexp.h> 
#include <qtooltip.h>
#include <qdragobject.h>
#include <qclipboard.h>
#include <qcheckbox.h>
#include <qheader.h>

//----------------------------------------------------------------------------

#ifndef Q_WS_WIN
#include <unistd.h>
#else
#include <io.h>
#endif

//----------------------------------------------------------------------------

#include "cvsconfig.h"
#include "lincvs.h"
#include "flatsplitter_impl.h"
#include "pixmapcache.h"
#include "PixmapTimer.h"
#include "colortab.h"
#include "cvslistview.h"
#include "noncvslistview.h"
#include "cvsignorelistview.h"
#include "directory.h"
#include "Patch.h"
#include "Edit.h"
#include "Watch.h"
#include "Merge.h"
#include "Diff.h"
#include "Annotate.h"
#include "History.h"
#include "Log.h"
#include "Update.h"
#include "Checkout.h"
#include "Import.h"
#include "Status.h"
#include "Tag.h"
#include "Commit.h"
#include "AutoCheck.h"
#include "Release.h"
#include "Add.h"
#include "Remove.h"
#include "Rename.h"
#include "Move.h"

//----------------------------------------------------------------------------

#include "aboutdialogimpl.h"
#include "ProfilesDialogImpl.h"
#include "PropertiesDialogImpl.h"
#include "settingsdialogimpl.h"
#include "login.h"
#include "WrappersIgnoreDialogImpl.h"
#include "ResolvDialogImpl.h"
#include "FileListViewItem.h"
#include "CustomizeShortcutsDialogImpl.h"
#include "FileMapperDialogImpl.h"
#include "GuiEventBlocker.h"
#include "VerifyDialog.h"

//----------------------------------------------------------------------------

ApplicationWindow::ApplicationWindow( QString startUpDir)
   : CvsControl( startUpDir),m_AutoRefreshEnabled(false), m_validClipboardData(false), m_onDrag(false), m_started(false)
{
   if (!LookAndFeel::g_bTopLevel) {//Adapt this setup in settingsdialogimpl.cpp too
      LookAndFeel::g_nonModalF |= Qt::WStyle_Dialog;
   }
   if (bUseSmallIcons) setIcon (findEmbeddedPixmap ("lincvs-16"));
   else setIcon (findEmbeddedPixmap ("lincvs-32"));
   
   //store the default font
   Font::g_default = QApplication::font().toString();
   
   //set desired font
   if (!Font::g_general.isEmpty()) {
      QFont f;
      if (f.fromString(Font::g_general)) QApplication::setFont(f, true);
      else Font::g_general = QString::null;
   }
   m_timer.setName("statusTimer");
   m_updateTimer.setName("autoUpdateTimer");
   
   GuiEventBlocker::setBlocking(true);
   initMenuAndToolbar();
   GuiEventBlocker::setBlocking(false);
   blockGUI(true);
   initComponents();
   setStatusText( tr("Scanning Project Directories ..."));
   
   connect( qApp, SIGNAL(lastWindowClosed()), this, SLOT(slot_closeApp()));
   
#ifndef NO_CLIPBOARD
   connect (QApplication::clipboard(), SIGNAL(dataChanged()), this, SLOT(clipboardDataChanged()));
#endif
   
   if (bStartedInDir) {
      Timer::singleShot(0, this, SLOT(startInSingleDir()));
   }
   Timer::singleShot(100, this, SLOT(initialDirScanning()));
   
}

//----------------------------------------------------------------------------

ApplicationWindow::~ApplicationWindow()
{
   writeSettings();
}

//----------------------------------------------------------------------------

void ApplicationWindow::writeSetup() {

   GEOM_X = geometry().x();
   GEOM_Y = geometry().y();
   GEOM_WIDTH  = geometry().width();
   GEOM_HEIGHT = geometry().height();

   SPLITTER_POS_H = m_pWorkBenchTree->width();
   SPLITTER_POS_V = m_pFileViews->height();

   WORKBENCH_COLUMN_WIDTH = m_pWorkBenchTree->columnWidth(0);

   TOOLBARPOS = "";
   QTextStream stream(TOOLBARPOS,IO_WriteOnly);
   stream << *this;

   assert(m_pFileViews->page(0));
   SizeProperties::g_fileHeader = static_cast<CvsListView*>(m_pFileViews->page(0))->getHeaderSetup();
}

//----------------------------------------------------------------------------

void ApplicationWindow::saveSettings() {
   writeSettings();//calls writeSetup()
   readSettings();
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_settings()
{
   stopTimers();
   SettingsDialogImpl *sd = new SettingsDialogImpl(m_pTools->whatsThisIconSet(),
	 this, "SettingsDlg", true,
	 LookAndFeel::g_modalF);
   //set font slot
   connect( sd, SIGNAL(setFonts()), this, SLOT(slot_setFonts()));
   connect( sd, SIGNAL(refreshEntries()), this, SLOT(refreshAllCvsEntries()));
   connect( sd, SIGNAL( autoUpdateStateChanged()), this, SLOT( slot_applyAutoUpdate()));
   
   sd->exec();
   int actions = Noop;
   if (sd->isOnTheFlyScanningModeChanged() && (!ONTHEFLYSCANNING)) {// scan and initialize all projects
      actions |= Analyze;
   }
   if (sd->isAgentSetupChanged()) {// kill own running agent
      killSshAgent();
   }
   delete sd;

   slot_updateSettings(actions);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_updateSettings(int mode) {

   static int actions = Noop;
   if (mode != -1) {
      actions = mode;
      QTimer::singleShot(10,this,SLOT(slot_updateSettings()));//is there another way to make sure dlg is hidden??
      return;
   }

   checkInProgress(true);
   blockGUI(true);
   if (actions & Analyze) {
      Directory *item = (Directory *)m_pWorkBenchTree->firstChild();
      while (item) {
         setStatusText("Scanning: "+item->fullName());
         item->analyzeAll();
         item = (Directory *)item->nextSibling();
      }
      setStatusText("");
   }

   slot_setFonts();
   setSettings();
   saveSettings();
   slot_checkStatus();
   startTimers();
   blockGUI(false);
   checkInProgress(false);
   actions = Noop;
}

//----------------------------------------------------------------------------

void ApplicationWindow::refreshAllCvsEntries() {

   qApp->setOverrideCursor(waitCursor); // takes some time on huge dirs
   Directory * myChild = (Directory*)(m_pWorkBenchTree->firstChild());
   while( myChild) {
      myChild->validateControlledFilesStatus(TRUE,TRUE);
      myChild = (Directory*)(myChild->nextSibling());
   }
   DirBase * dir;
   if ( (dir = getSelectedDir()) ) {
      dir->activateItem(FALSE);
   }
   qApp->restoreOverrideCursor();
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_poll() {
   slot_checkStatus();
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_setFonts()
{
   QString fontStr;
   QFont f;

   //font of app
   if (Font::g_general.isEmpty()) fontStr = Font::g_default;
   else fontStr = Font::g_general;
   f.fromString(fontStr);
   QApplication::setFont(f, true);

   //font of message widgets   
   if (!Font::g_message.isEmpty()) fontStr = Font::g_message;
   f.fromString(fontStr);
   m_pMessages->setFont(f);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_profiles()
{
   CProfilesDialogImpl *profileDlg =
      new CProfilesDialogImpl(&cvsUsers, &cvsServers,
	    &cvsRepositories,
	    cvsProfileNameList,
	    cvsProfileContentList,
	    m_pTools->whatsThisIconSet(),
	    this);
   if (profileDlg->exec() == QDialog::Accepted) {
      cvsProfileNameList = profileDlg->profileNameList();
      cvsProfileContentList = profileDlg->profileContentList();
   }

   delete profileDlg;
   saveSettings();
}

//----------------------------------------------------------------------------

void ApplicationWindow::showDirProperties()
{
   Directory * dir;
   if (!(dir = getSelectedDir() ) ) {
      return;
   }

   PropertiesDialogImpl Properties (m_pTools->whatsThisIconSet(),
	 this, dir);
   Properties.exec();
   applyProjectSettings();//do it allways, not only on ok
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_initLocalRep() {
  
   QString initDir;
   //   QFileDialog* fd = new QFileDialog( LookAndFeel::g_b0AsParent ? 0 : this, "fd-cvs-init", TRUE );
   QFileDialog* fd = new QFileDialog( this, "fd-cvs-init", TRUE );
   fd->setCaption(tr("Create local repository in:"));
   fd->setMode( QFileDialog::Directory );
   if ( fd->exec() == QDialog::Accepted) {
      initDir = fd->selectedFile();
      if ( initDir.isEmpty()) return;
   } else {
      return;
   }

   QDir dir(initDir+"/CVSROOT");
   if (dir.exists()) {
      showWarning(tr("Warning"),
	    tr("The repository:") + " "
	    + initDir
	    + tr(" already exists!"));
      return;
   }

   QApplication::setOverrideCursor(waitCursor);

   initLocalRepository(initDir);

   QApplication::restoreOverrideCursor();
}

//----------------------------------------------------------------------------

void ApplicationWindow::resolveFile() {
   DirBase * dir;
   if (!(dir = getSelectedDir())) {
      return;
   }
   QString fname =  dir->getFirstSelectedFile();
   if (fname.isEmpty()) {
      return;
   }

   QString FFN;
   FFN = dir->fullName() + "/" + fname;

   qApp->processEvents();//close popup menu
   new ResolvDialogImpl(FFN, m_pTools->whatsThisIconSet(),
	 this, LookAndFeel::g_nonModalF | Qt::WDestructiveClose,
	 dir->getRev(&fname));
}

//----------------------------------------------------------------------------

void ApplicationWindow::test() {
   CvsControl::test();
}

//----------------------------------------------------------------------------

//#define BLOCKGUIDEBUG
void ApplicationWindow::blockGUI(bool block, int flags /* = BLOCKALL */) {

   static bool isLocked = false;
   if (isLocked) {
#ifdef BLOCKGUIDEBUG
      qDebug("locked blockGUI("+QString(block ? "true" : "false")+","+QString::number(flags)+") call, suppressed");
#endif
      return;
   }

   bool doBlock = (GuiEventBlocker::isBlocked() != block);

#ifdef BLOCKGUIDEBUG
   if (doBlock) {
      qDebug("blockGUI("+QString(block ? "true" : "false")+","+QString::number(flags)+")");
   } else {
      qDebug("repeated blockGUI("+QString(block ? "true" : "false")+","+QString::number(flags)+") call");
   }
#endif

   if (!doBlock) {
      if (flags == BLOCKNOCONTROLS) {
#ifdef BLOCKGUIDEBUG
	 qDebug("blockGUI BLOCKNOCONTROLS done");
#endif
	 return;
      }
   }

   if (block) {
      if (doBlock) {
	 lockNotifications(true);
	 setTimersActive(false);
	 GuiEventBlocker::setBlocking(true);
	 isLocked = true;
	 qApp->processEvents();//clear event queue
	 isLocked = false;
	 if (flags & BLOCKNOCURSOR) {
	    m_showStopTimer.start(5000,true);
#ifdef BLOCKGUIDEBUG
	    qDebug("blockGUI stopTimer started");
#endif
	 }
      }
      if (!(flags & BLOCKNOCURSOR)) QApplication::setOverrideCursor(waitCursor);
   }

   if (!(flags & BLOCKNOCONTROLS)) {
      m_pTools->m_pStopAction->setEnabled(block);
      m_pTools->setEnabled(!block);
      menuBar()->setEnabled(!block);
   }

   if (!block) {
      if (!(flags & BLOCKNOCURSOR)) QApplication::restoreOverrideCursor();
      if (doBlock) {
	 if (m_showStopTimer.isActive()) {
	    m_showStopTimer.stop();
#ifdef BLOCKGUIDEBUG
	    qDebug("blockGUI stopTimer stopped");
#endif
	 } else m_pTools->m_pStopAction->setEnabled(false);
	 GuiEventBlocker::setBlocking(false);
	 setTimersActive(true);
	 lockNotifications(false);
      }
   }
#ifdef BLOCKGUIDEBUG
   qDebug("blockGUI done");
#endif
}
#undef BLOCKGUIDEBUG
//----------------------------------------------------------------------------

bool ApplicationWindow::isGuiBlocked() {
   return GuiEventBlocker::isBlocked();
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_blockGUI(bool state) {
   blockGUI(state);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_stopStatusTimer() {
   stopTimers(TSTATUS);
}

//----------------------------------------------------------------------------

void ApplicationWindow::setTimersActive(bool activate) {

   if (activate) {
      m_updateTimer.proceed();
      m_timer.proceed();
   } else {
      m_timer.pause();
      m_updateTimer.pause();
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_enableStopButton() {
   m_pTools->m_pStopAction->setEnabled(true);
#ifdef BLOCKGUIDEBUG
   qDebug("blockGUI stop button enabled");
#endif
}

//----------------------------------------------------------------------------

void ApplicationWindow::lockNotifications(bool lock) {
   if (lock) {
      lockDirWatch();
   } else {
      unlockDirWatch();
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::closeEvent( QCloseEvent* e )
{
   e->accept();
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_closeApp() {
   KILLED = true;//for event loops or sock reads to detect shutdown
   qApp->quit ();
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_closeApplication() {
   qApp->closeAllWindows ();
}

//----------------------------------------------------------------------------

void ApplicationWindow::initDone() {

   setStatusText(tr("Analyzing dirs, please wait ..."));

   /** Enable automatic status refresh */
   qApp->setOverrideCursor(waitCursor); // takes some time on huge dirs
   m_started = true;
   DirBase * dir = m_pWorkBenchTree->selectedItem();
   if (dir) {
      slot_dirClicked(dir);
      setDirMenu();
   }
   slot_checkStatus();//starts timer if required
   qApp->restoreOverrideCursor();

   setStatusText(tr("Ready"));

   blockGUI(false);

   //start updateTimer if required
   slot_toggleAutoUpdate();

}

//----------------------------------------------------------------------------

void ApplicationWindow::setStatusText( const QString &txt, int ms) {

   if (ms == 0) {
      m_statusMsg = txt;
      statusBar()->message( txt);
   } else {
      statusBar()->message( txt, ms);
   }

}

//----------------------------------------------------------------------------

void ApplicationWindow::statusMessageChanged( const QString & txt) {

   if ( txt == QString::null) {
      statusBar()->message( m_statusMsg);
      m_statusMsg = QString::null;
   } else if (m_statusMsg.isNull()) {
      m_statusMsg = txt;
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::checkInProgress(bool state) {
   m_pProgressWidget->setStatus(state);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_refreshAll() {
   slot_checkStatus(true);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_applyAutoUpdate() {
   m_pTools->autoUpdateChanged();
   applyProjectSettings();
}

//----------------------------------------------------------------------------

/**
 * activates or stops update timer
 * responsible for automatic running the cvs update cmd
 * if enabled in Directory/Properties
 */
void ApplicationWindow::slot_toggleAutoUpdate() {
   slot_applyAutoUpdate();
   if (AUTOUPDATE) {
      startTimers(TUPDATE);
   } else {
      stopTimers(TUPDATE);
   }
}

//----------------------------------------------------------------------------

/**
 * Initialises the application's menu and toolbar.  Note that some partial
 * initialisation may have already been done in the designer generated stub.
 */
void ApplicationWindow::initMenuAndToolbar()
{
   //main menu items
   m_pProjectMenu = new Menu( this );
   connect (m_pProjectMenu,SIGNAL(aboutToShow()),this,SLOT(fillProjectMenu()));
   m_pMenubar->insertItem( QString(tr("&Project")), m_pProjectMenu, MENU_PROJECT );

   m_pDirMenu = new Menu( this );
   m_pDirProjectSubMenu = new Menu(m_pDirMenu);//create submenu 'Project'
   m_pDirStateSubMenu = new Menu(m_pDirMenu);//create submenu 'State'
   m_pDirWatchSubMenu = new Menu(m_pDirMenu);//create submenu 'Watch'
   m_pDirImportSubMenu = new Menu(m_pDirMenu);//create submenu 'Import/Export'
   m_pDirUpdateSubMenu = new Menu(m_pDirMenu);//create submenu 'Update'
   m_pDirAnnotateSubMenu = new Menu(m_pDirMenu);//create submenu 'Annotate'
   m_pDirDiffSubMenu = new Menu(m_pDirMenu);//create submenu 'Diff/Patch'
   m_pDirTreeSubMenu = new Menu(m_pDirMenu);//create submenu 'Tree'
   m_pDirEditSubMenu = new Menu(m_pDirMenu);//create submenu 'Edit/Unedig'
   m_pDirLockSubMenu = new Menu(m_pDirMenu);//create submenu 'Lock/Unlock'
   m_pMenubar->insertItem( QString(tr("&Directory")), m_pDirMenu, MENU_DIR );


   m_pFileMenu = new Menu( this );
   m_pFileStateSubMenu = new Menu(m_pFileMenu);//create submenu 'State'
   m_pFileWatchSubMenu = new Menu(m_pFileMenu);//create submenu 'Watch'
   m_pFileSubMenu = new Menu(m_pFileMenu);//create submenu 'Update'
   m_pFileAnnotateSubMenu = new Menu(m_pFileMenu);//create submenu 'Annotate'
   m_pFileDiffSubMenu = new Menu(m_pFileMenu);//create submenu 'Diff/Patch'
   m_pFileLockSubMenu = new Menu(m_pFileMenu);//create submenu 'Lock/Unlock'
   m_pMenubar->insertItem( QString(tr("&File")), m_pFileMenu, MENU_FILE );

   m_pConfigMenu = new Menu( this );
   m_pMenubar->insertItem( QString(tr("&Config")), m_pConfigMenu, MENU_OPTIONS );

   // Makes the Help menu right-align in Motif style
   m_pMenubar->insertSeparator();

   m_pHelpMenu = new Menu( this );
   m_pMenubar->insertItem( QString(tr("&Help")), m_pHelpMenu, MENU_HELP );

   
   // toolbar
   m_pTools = new cvsToolbar (this);
   addDockWindow( m_pTools, tr( "Toolbar" ), Top, FALSE );
   QMenuBar *mb = new QMenuBar( m_pTools );
   mb->setFrameStyle( QFrame::NoFrame );
   m_pTools->setStretchableWidget( mb );

   QTextStream stream(TOOLBARPOS,IO_ReadOnly);
   stream >> *this;
  
   connect( m_pTools, SIGNAL( autoUpdateStateChanged()),
	 this, SLOT( slot_toggleAutoUpdate()));

   // StatusBar
   m_pProgressWidget = new StatusWidget(statusBar());
   statusBar()->addWidget(m_pProgressWidget, 0, true);
   connect( statusBar(), SIGNAL(messageChanged(const QString &)),
	 this, SLOT( statusMessageChanged( const QString &)));
  
   /* Create the pixmap timer and connect the timeout signal
    * that will reset the timer image to its idle position
    * whenever an operation finishes.
    */
   m_pPixmapTimer = new CPixmapTimer(this, "progress icon indicator");
   connect(m_pPixmapTimer, SIGNAL(signalTimeout(const QPixmap &)),
	 m_pProgressWidget, SLOT(setPixmap(const QPixmap &)));

   // force update of the progress icon
   m_pPixmapTimer->slotStop();
  
   //init menu entries
   //while instanciating the following MenuItems, each is attached an increasing instanceNum(),
   //according to its position in the following list.
   //I started putting these in alphabetical order but:
   //For not to change the meaning of the .lincvsrc PopupMenuDir and file values, additional
   //entries !have! to be put to the end of this list, disregarding the alphabetical order.
  
   M_COMMIT = new MenuItem( TMenu::DIR,findEmbeddedPixmap ("commit_16x16"),
	 tr("Commit ..."), SLOT(slot_commitDir()) );
  
   M_COMMITFILE = new MenuItem(TMenu::FILE,findEmbeddedPixmap("commit_16x16"),
	 tr("Commit ..."), SLOT(slot_commitFile()));
  
   M_CUSTOMIZE = new MenuItem(TMenu::DIR,tr("Customize"), SLOT(customizeDirPopupMenu()),
	 0, DP_Customize);
  
   M_CUSTOMIZEFILE = new MenuItem(TMenu::FILE,tr("Customize"), SLOT(customizeFilePopupMenu()),
	 0,DP_Customize);
  
   M_DIFFSIDEBYSIDEFILE = new MenuItem(TMenu::FILE,tr("Diff side-by-side"), SLOT(slot_diffFileSideBySide()));
  
   M_DIFFTOCONSOLEFILE = new MenuItem(TMenu::FILE,tr("Diff to console"), SLOT(slot_diffFile()));
  
   M_DISABLEDIR = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("Disable16x16"),
	 tr("Disable Directory"), SLOT(disableProjectSlot()) );

   M_EDITFILE = new MenuItem(TMenu::FILE,findEmbeddedPixmap ("FileWriteable16x16"),
	 tr("Edit"), SLOT(slot_editFile()));

   M_EDITDIR = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("FileWriteable16x16"),
	 tr("Edit"), SLOT(slot_editDir()));

   M_ENABLEDIR = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("Enable16x16"),
	 tr("Enable Directory"), SLOT(enableProjectSlot()) );

   M_EXPANDDIR = new MenuItem(TMenu::DIR,tr("Expand all"), SLOT(expandDir()));

   M_HISTORY = new MenuItem(TMenu::DIR,tr("History"), SLOT(slot_historyDir()) );

   M_HISTORYFILE = new MenuItem(TMenu::FILE,tr("History"), SLOT(slot_historyFile()));

   M_LOGIN = new MenuItem(TMenu::DIR,tr("Login"), SLOT(loginDir()) );

   M_LOGOUT = new MenuItem(TMenu::DIR,tr("Logout"), SLOT(logoutDir()) );

   M_LOGTREEFILE = new MenuItem(TMenu::FILE,findEmbeddedPixmap ("LogTree16x16"),
	 tr("Log tree"), SLOT(slot_logFile()));

   M_PROPERTIES = new MenuItem(TMenu::DIR,tr("Properties ..."), SLOT(showDirProperties()) );

   M_QUERYUPDATE = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("QueryUpdate16x16"),
	 tr("Query Update"), SLOT(slot_queryUpdateDir()) );

   M_QUERYUPDATEFILE = new MenuItem(TMenu::FILE,findEmbeddedPixmap ("QueryUpdate16x16"),
	 tr("Query Update"), SLOT(slot_queryUpdateFile()));

   M_RELEASE = new MenuItem(TMenu::DIR, tr("Release"), SLOT(slot_release()) );

   M_RELEASEANDREMOVE = new MenuItem(TMenu::DIR, tr("Release and remove"), SLOT(slot_releaseAndRemove()) );

   M_REMOVE = new MenuItem(TMenu::DIR,tr("Remove") );

   M_REMOVEFILE = new MenuItem(TMenu::FILE,tr("Remove") );

   M_REMOVEFROMWORKBENCH = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("Delete16x16"),
	 tr("Remove From Workbench"), SLOT(slot_removeProjectSlot()) );

   M_RENAMEFILE = new MenuItem(TMenu::FILE,tr("Rename file ..."), SLOT(slot_renameFile()));

   M_REREADDIR = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("refresh_16x16"),
	 tr("Reread tree"), SLOT(slot_rereadProject()) );

   M_SEPARATOR = new MenuItem( TMenu::NONE,tr("------------------"), 0 );

   M_SHOWEDITORS = new MenuItem(TMenu::DIR,tr("Show Editors"), SLOT(slot_showEditorsDir()));

   M_SHOWEDITORSFILE = new MenuItem(TMenu::FILE,tr("Show Editors"), SLOT(slot_showEditorsFile()));

   M_STATUS = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("FolderStatus16x16"),
	 tr("Status"), SLOT(slot_statusDir()) );

   M_STATUSFILE = new MenuItem(TMenu::FILE,findEmbeddedPixmap ("FileStatus16x16"),
	 tr("Status"), SLOT(slot_statusFile()));

   M_TAG = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("Tag16x16"),
	 tr("Tag ..."), SLOT(slot_tagDir()) );

   M_TAGFILE = new MenuItem(TMenu::FILE,findEmbeddedPixmap ("Tag16x16"),
	 tr("Tag ..."), SLOT(slot_tagFile()));

   M_UNEDITDIR = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("FileUnchanged16x16"),
	 tr("Unedit"), SLOT(slot_uneditDir()));

   M_UNEDITFILE = new MenuItem(TMenu::FILE,findEmbeddedPixmap ("FileUnchanged16x16"),
	 tr("Unedit"), SLOT(slot_uneditFile()));

   M_UPDATE = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("update_16x16"),
	 tr("Update (patch)"), SLOT(slot_updateDir()) );

   M_UPDATEFILE = new MenuItem(TMenu::FILE,findEmbeddedPixmap ("update_16x16"),
	 tr("Update (patch)"), SLOT(slot_updateFile()));

   M_UPDATEOTHER = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("Tag16x16"),
	 tr("Update Other Revision ..."), SLOT(slot_updateOtherRevisionDir()) );

   M_UPDATEOTHERFILE = new MenuItem(TMenu::FILE,findEmbeddedPixmap ("Tag16x16"),
	 tr("Update Other Revision ..."), SLOT(slot_updateOtherRevisionFile()) );

   M_TEST = new MenuItem(TMenu::NONE, tr("test"), SLOT(test()));


   M_LOCKFILE = new MenuItem(TMenu::FILE,tr("Lock File"), SLOT(test()));
   M_LOCKDIR = new MenuItem(TMenu::DIR,tr("Lock Files"), SLOT(test()));

   M_UNLOCKFILE = new MenuItem(TMenu::FILE,tr("Unlock File"), SLOT(test()));
   M_UNLOCKDIR = new MenuItem(TMenu::DIR,tr("Unlock Files"), SLOT(test()));

   M_SHOWWATCHERSFILE = new MenuItem(TMenu::FILE,tr("Show Watchers"), SLOT(slot_showWatchersFile()));
   M_SHOWWATCHERSDIR = new MenuItem(TMenu::DIR,tr("Show Watchers"), SLOT(slot_showWatchersDir()));

   M_OPENFILE = new MenuItem(TMenu::FILE | TMenu::NONCVSFILE | TMenu::IGNOREFILE,tr("Open File"), SLOT(openFile()));

   M_BROWSEDIR = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("FolderClosed16x16"),
	 tr("Browse Dir"), SLOT(browseDir()));

   M_OPENSHELLDIR = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("FolderClosed16x16"),
	 tr("Open Shell"), SLOT(openShellDir()));

   M_UPDATEDIRBRANCHFILE = new MenuItem(TMenu::FILE,findEmbeddedPixmap ("Tag16x16"),
	 tr("Update (using dir branch)"), SLOT(slot_updateToDirBranchFile()));

   M_UPDATEDIRBRANCHDIR = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("Tag16x16"),
	 tr("Update (using dir branch)"), SLOT(slot_updateToDirBranchDir()));

   M_REMOVELOCALOPTIONSFILE = new MenuItem(TMenu::FILE, tr("Remove local (-kx) options"), SLOT(slot_removeLocalOptionsFile()));

   M_REMOVELOCALOPTIONSDIR = new MenuItem(TMenu::DIR, tr("Remove local (-kx) options"), SLOT(slot_removeLocalOptionsDir()));

   M_EXPORTDIR = new MenuItem(TMenu::DIR, tr("Export dir tree ..."), SLOT(exportFromModuleDir()));

   M_COLLAPSDIR = new MenuItem(TMenu::DIR,tr("Collapse all"), SLOT(collapsDir()));

   M_VIEWFILE = new MenuItem(TMenu::FILE | TMenu::NONCVSFILE | TMenu::IGNOREFILE,tr("View file"), SLOT(viewFile()));

   M_REPLACEFILE = new MenuItem(TMenu::FILE,findEmbeddedPixmap ("update_16x16"),
	 tr("Update (replace)"), SLOT(slot_updateReplaceFile()));

   M_REPLACEDIR = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("update_16x16"),
	 tr("Update (replace)"), SLOT(slot_updateReplaceDir()));

   M_RESOLVE = new MenuItem(TMenu::FILE, findEmbeddedPixmap ("FileConflict16x16"),
	 tr("Resolve ..."), SLOT(resolveFile()));

   M_DIFFTOCONSOLEDIR = new MenuItem(TMenu::DIR,tr("Diff to console"), SLOT(slot_diffDir()));

   M_ADVDIFFTOCONSOLEFILE = new MenuItem(TMenu::FILE, tr("Diff (tag/s) to console ..."), SLOT(slot_diffAdvancedFile()));

   M_ADVDIFFTOCONSOLEDIR = new MenuItem(TMenu::DIR, tr("Diff (tag/s) to console ..."), SLOT(slot_diffAdvancedDir()));

   M_QUERYUPDATEALL = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("QueryUpdate16x16"),
	 tr("Query Update all"), SLOT(slot_queryUpdateAllDir()));

   M_REPLACEALLDIR = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("update_16x16"),
	 tr("Update all (replace)"), SLOT(slot_updateReplaceAllDir()));

   M_UPDATEALL = new MenuItem(TMenu::DIR,findEmbeddedPixmap ("update_16x16"),
	 tr("Update all (patch)"), SLOT(slot_updateAllDir()));

   M_UPDATEVIRTUALDIR = new MenuItem(TMenu::NONE,findEmbeddedPixmap("update_16x16"),
	 tr("Update unknown dir"),SLOT(slot_updateVirtualDir()));

   M_REMOVEFROMDISK = new MenuItem(TMenu::FILE, tr("From disk"),SLOT(slot_removeFileFromDisk()));

   M_REMOVEFROMREP = new MenuItem(TMenu::NONE, tr("From repository"),SLOT(test()));//don't remove to keep old lincvsrc compatible

   M_REMOVEFROMREPANDDISK = new MenuItem(TMenu::FILE,findEmbeddedPixmap("Delete16x16"),
	 tr("From repository and disk"),SLOT(slot_removeFile()));

   M_REMOVEFROMDISKDIR = new MenuItem(TMenu::DIR,  tr("From disk"),SLOT(removeDirIfNoCvsRegisteredFiles()));

   M_REMOVEFROMREPANDDISKDIR = new MenuItem(TMenu::DIR,findEmbeddedPixmap("Delete16x16"),
	 tr("From repository and disk"),SLOT(slot_removeDir()));

   M_ADDFILE = new MenuItem(TMenu::NONE, tr("Add (resurrect) file"),SLOT(slot_resurrectFile()));

   M_ADDFOLDER = new MenuItem(TMenu::NONCVSFILE | TMenu::IGNOREFILE,findEmbeddedPixmap("FolderClosed16x16"),
	 tr("Add folder"),
	 SLOT(slot_addDir()));

   M_ADDASCIIFILE = new MenuItem(TMenu::NONCVSFILE | TMenu::IGNOREFILE,findEmbeddedPixmap("FileASCII16x16"),
	 tr("Add ASCII file"),
	 SLOT(slot_addFileAscii()));
  
   M_ADDBINARYFILE = new MenuItem(TMenu::NONCVSFILE | TMenu::IGNOREFILE,findEmbeddedPixmap ("FileBinary16x16"),
	 tr("Add binary file"),
	 SLOT(slot_addFileBinary()));
  
   M_CREATENEWFILE = new MenuItem(TMenu::NONCVSFILE,findEmbeddedPixmap("FileUnknown16x16"),
	 tr("Create new file"),
	 SLOT(createNewFile()));
  
   M_CREATENEWDIR = new MenuItem(TMenu::NONCVSFILE,findEmbeddedPixmap("FolderClosed16x16"),
	 tr("Create new dir"),
	 SLOT(createNewDir()));

   M_DELETENONCVSFROMDISKFILE = new MenuItem(TMenu::NONCVSFILE,findEmbeddedPixmap ("Delete16x16"),
	 tr("Delete from disk"),
	 SLOT(removeNonCvsFileFromDisk()));

   M_DELETEIGNOREFROMDISKFILE = new MenuItem(TMenu::IGNOREFILE,findEmbeddedPixmap ("Delete16x16"),
	 tr("Delete from disk"),
	 SLOT(removeCvsIgnoreFileFromDisk()));

   M_ADDPROJECT = new MenuItem(TMenu::PROJECT, tr("Add project to workbench ..."),SLOT( slot_addProject()) );

   M_CHECKOUTPROJECT = new MenuItem(TMenu::PROJECT, tr("Checkout ..."),SLOT( slot_checkoutProject()) );

   M_IMPORTPROJECT = new MenuItem(TMenu::PROJECT, tr("Import ..."),SLOT( slot_importProject()) );

   M_EXPORTPROJECT = new MenuItem(TMenu::PROJECT, tr("Export ..."),SLOT( slot_exportProject()) );

   M_EXITPROJECT = new MenuItem(TMenu::PROJECT, tr("Exit"),SLOT( slot_closeApplication()) );

   M_SETTINGSOPTIONS = new MenuItem(TMenu::OPTIONS, tr("Settings ..."),SLOT( slot_settings()) );

   M_PROFILESOPTIONS = new MenuItem(TMenu::OPTIONS, tr("Profiles ..."),SLOT( slot_profiles()) );

   M_KEYSHORTCUTSOPTIONS = new MenuItem(TMenu::OPTIONS, tr("Keyboard shortcuts ..."),SLOT( slot_customizeKeyShortcuts()) );

   M_OPENFILEMAPPINGOPTIONS = new MenuItem(TMenu::OPTIONS, tr("Open file mapping ..."),SLOT( slot_customizeOpenFileMapping()) );

   M_VIEWFILEMAPPINGOPTIONS = new MenuItem(TMenu::OPTIONS, tr("View file mapping ..."),SLOT( slot_customizeViewFileMapping()) );

   M_REFRESHOPTIONS = new MenuItem(TMenu::PROJECT, tr("Refresh"),SLOT( slot_refreshAll()) );

   M_HELP = new MenuItem(TMenu::HELP, tr("Help ..."),SLOT( slot_help()) );

   M_ONLINEHELP = new MenuItem(TMenu::HELP, tr("Online cvs help"),SLOT( slot_onlineHelp()) );

   M_DEBUGLOGHELP = new MenuItem(TMenu::HELP, tr("Write bugreport ..."),SLOT( slot_debug()) );

   M_ABOUTHELP = new MenuItem(TMenu::HELP, tr("About ..."),SLOT( slot_about()) );

   M_ABOUTQTHELP = new MenuItem(TMenu::HELP, tr("About Qt ..."),SLOT( slot_aboutQt()) );

   M_RENAMEDIR = new MenuItem(TMenu::DIR, tr("Rename project (solely local)"),SLOT( slot_renameLocalDir()) );

   M_COPYFILE = new MenuItem(TMenu::FILE, tr("Copy"),SLOT(slot_copyFile()));

   M_PASTE = new MenuItem(TMenu::DIR, tr("Paste"),SLOT(slot_paste()) );

   M_MERGEDIR = new MenuItem(TMenu::DIR, findEmbeddedPixmap("merge_16x16"), tr("Merge ..."), SLOT( slot_mergeDir()));

   M_MERGEFILE = new MenuItem(TMenu::FILE, findEmbeddedPixmap("merge_16x16"), tr("Merge ..."), SLOT( slot_mergeFile()));

   M_COPYOUTPUT = new MenuItem(TMenu::OUTPUT, tr("Copy"), SLOT(copyOutput()));

   M_SELECTALLOUTPUT = new MenuItem(TMenu::OUTPUT, tr("Select All"), SLOT(selectAllOutput()));

   M_TOGGLESHOWALLOUTPUT = new MenuItem(TMenu::OUTPUT, tr("Toggle show all"), SLOT(toggleShowAllOutput()));

   M_CREATEPATCHDIR = new MenuItem(TMenu::DIR, tr("Create patch ..."), SLOT(slot_createPatchDir()));

   M_LOGTREEDIR = new MenuItem(TMenu::DIR, findEmbeddedPixmap("LogTree16x16"), tr("Log tree"), SLOT( slot_logDir()));

   M_IMPORTVENDORDIR = new MenuItem(TMenu::DIR, tr("Import Vendor ..."), SLOT(test()));

   M_RENAMENONCVSFILE = new MenuItem(TMenu::NONCVSFILE | TMenu::IGNOREFILE, tr("Rename non-cvs ..."), SLOT(slot_renameNonCvsFile()));

   M_REMOVECVSBACKED = new MenuItem(TMenu::DIR, tr("Remove .#xxx files"), SLOT(slot_removeCvsBackedFiles()));

   M_INITLOCALCVS = new MenuItem(TMenu::PROJECT, tr("Create local repository ..."), SLOT(slot_initLocalRep()));

   M_ANNOTATEDIR = new MenuItem(TMenu::DIR, tr("Annotate"), SLOT(slot_annotateDir()));

   M_ANNOTATELINEDIR = new MenuItem(TMenu::DIR, tr("Annotate (search line)"), SLOT(slot_annotateGrepLineDir()));

   M_ANNOTATELINESDIR = new MenuItem(TMenu::DIR, tr("Annotate (search lines)"), SLOT(slot_annotateGrepLinesDir()));

   M_ANNOTATEFILE = new MenuItem(TMenu::FILE, tr("Annotate"), SLOT(slot_annotateFile()));

   M_ANNOTATELINEFILE = new MenuItem(TMenu::FILE, tr("Annotate (search line)"), SLOT(slot_annotateGrepLineFile()));

   M_ANNOTATELINESFILE = new MenuItem(TMenu::FILE, tr("Annotate (search lines)"), SLOT(slot_annotateGrepLinesFile()));

   M_WATCHONDIR = new MenuItem(TMenu::DIR, tr("Watch on"), SLOT(slot_watchOnDir()));

   M_WATCHOFFDIR = new MenuItem(TMenu::DIR, tr("Watch off"), SLOT(slot_watchOffDir()));

   M_WATCHADDDIR = new MenuItem(TMenu::DIR, tr("Watch add"), SLOT(slot_watchAddDir()));

   M_WATCHREMOVEDIR = new MenuItem(TMenu::DIR, tr("Watch remove"), SLOT(slot_watchRemoveDir()));

   M_WATCHONFILE = new MenuItem(TMenu::FILE, tr("Watch on"), SLOT(slot_watchOnFile()));

   M_WATCHOFFFILE = new MenuItem(TMenu::FILE, tr("Watch off"), SLOT(slot_watchOffFile()));

   M_WATCHADDFILE = new MenuItem(TMenu::FILE, tr("Watch add"), SLOT(slot_watchAddFile()));

   M_WATCHREMOVEFILE = new MenuItem(TMenu::FILE, tr("Watch remove"), SLOT(slot_watchRemoveFile()));

   M_ENABLETREE = new MenuItem(TMenu::DIR, tr("Enable tree"), SLOT(enableTree()));

   M_DISABLEREVERTTREE = new MenuItem(TMenu::DIR, tr("Disable others"), SLOT(disableOthers()));

   M_CREATEPATCHFILE = new MenuItem(TMenu::FILE, tr("Create Patch ..."), SLOT(slot_createPatchFile()));

   M_CONDITIONALEDITFILE = new MenuItem(TMenu::FILE, tr("Conditional edit"), SLOT(test()));

   M_CONDITIONALEDITDIR = new MenuItem(TMenu::DIR, tr("Conditional edit"), SLOT(test()));

   M_REFRESHEDITORS = new MenuItem(TMenu::NONE, "", SLOT(test()) );

   M_ADDSYMLINK = new MenuItem(TMenu::NONE, "", SLOT(test()));

   M_DIFFFILEMAPPINGOPTIONS = new MenuItem(TMenu::NONE, "", SLOT(test()) );

   M_MODIFYIGNORE = new MenuItem(TMenu::NONE, "", SLOT(test()));

   M_MODIFYIGNOREFILE = new MenuItem(TMenu::NONE, "", SLOT(test()));

   M_MODELOKALDIR = new MenuItem(TMenu::NONE, "", SLOT(test()));

   M_MODELOKALFILE = new MenuItem(TMenu::NONE, "", SLOT(test()));

   M_MODEREMOTEDIR = new MenuItem(TMenu::NONE, "", SLOT(test()));

   M_MODEREMOTEFILE = new MenuItem(TMenu::NONE, "", SLOT(test()));

   M_ADDOTHERFILE = new MenuItem(TMenu::NONE, "",SLOT(test()));

   M_CHANGESETDIR = new MenuItem(TMenu::NONE, "",SLOT(test()));

   M_BROWSEREMOTEMODULES = new MenuItem(TMenu::NONE, "",SLOT(test()));

   M_BROWSEREMOTETAGS = new MenuItem(TMenu::NONE, "",SLOT(test()));

   M_CLEAROUTPUT = new MenuItem(TMenu::OUTPUT, tr("Clear"), SLOT(slot_clearOutput()));

   //put additional MenuItems here:
   //M_NEWITEM = new MenuItem( tr("the_new_item_txt"), SLOT(newslot()));

   //create parent<->child dependencies here
   M_REMOVEFROMREPANDDISKDIR->setParent(M_REMOVE);
   M_REMOVEFROMDISKDIR->setParent(M_REMOVE);

   M_REMOVEFROMDISK->setParent(M_REMOVEFILE);
   M_REMOVEFROMREPANDDISK->setParent(M_REMOVEFILE);

   //create default choice for cvsdirlistview
   //use alphabetical order here
   allDirPopupMenuItems = new QVector<MenuItem>(1);
   allDirPopupMenuItems->insert(0,M_SEPARATOR);

   MenuItem * item = MenuItem::firstSorted();
   do {
      if ( item->menu() & TMenu::DIR ) {
	 if (item->getParent()) continue;
	 unsigned int pos = allDirPopupMenuItems->count();
	 allDirPopupMenuItems->resize(pos+1);
	 allDirPopupMenuItems->insert(pos,item);
      }
   } while ( (item = item->nextSortedItem()) );

   //read menu for cvsdirlistview from cfg
   dirPopupMenu = new Menu(this);
   connect(dirPopupMenu,SIGNAL(aboutToShow ()),this,SLOT(fillDirPopupMenu()));
   QString menuContent = DIRPOPUPMENU;

   int count = menuContent.contains(':');
   if (count==0) {//fallback to default - should never happen
      menuContent = DEFDIRPOPUPMENU; //"29:17:27:11:26:35:0:31:37:26:13:14:26:8:33:10:26:9:6:25:23:26:19:21:20:26:16:26:2:";
      DIRPOPUPMENU = menuContent;// otherwise no entries are written to cfg
      count = menuContent.contains(':');
   }

   menuContent.truncate(menuContent.length());
   dirPopupMenuItems = new QVector<MenuItem>(count);
  
   unsigned int size = 0;
   unsigned int i;
   for (i = 0; i<(unsigned int)count; i++) {
      int z = menuContent.find(':');
      int id = menuContent.left(z).toInt();
      MenuItem *tmp = MenuItem::getInstance(id);
      if (tmp) {
	 bool hasParent = FALSE;
	 MenuItem * parent = tmp->getParent();//in case new parent<->child relations are set up, this is needed to keep old user configuration working
	 while (parent) {
	    hasParent = TRUE;
	    tmp = parent;
	    parent = parent->getParent();
	 }
	 if  ( (!hasParent) || (dirPopupMenuItems->find(tmp) == -1) ) {
	    dirPopupMenuItems->insert(size++,tmp);
	 }
      }
      menuContent=menuContent.mid(z+1);
   }
   dirPopupMenuItems->resize(size);
  
   //create default choice for cvsfilelistview
   allFilePopupMenuItems = new QVector<MenuItem>(1);
   allFilePopupMenuItems->insert(0,M_SEPARATOR);

   item = MenuItem::firstSorted();
   do {
      if ( item->menu() & TMenu::FILE ) {
	 if (item->getParent()) continue;
	 unsigned int pos = allFilePopupMenuItems->count();
	 allFilePopupMenuItems->resize(pos+1);
	 allFilePopupMenuItems->insert(pos,item);
      }
   } while ( (item = item->nextSortedItem()) );
  
   //read menu for cvsfilelistview from cfg
   filePopupMenu = new Menu(this);
   connect(filePopupMenu,SIGNAL(aboutToShow ()),this,SLOT(fillFilePopupMenu()));
   menuContent = FILEPOPUPMENU;
  
   count = menuContent.contains(':');
  
   if (count==0) {//fallback to default - should never happen
      menuContent = DEFFILEPOPUPMENU; //"30:18:28:12:26:15:4:5:26:36:1:32:38:26:7:34:26:24:22:26:3:";
      FILEPOPUPMENU = menuContent;// otherwise no entries are written to cfg
      count = menuContent.contains(':');
   }
  
   menuContent.truncate(menuContent.length());
   filePopupMenuItems = new QVector<MenuItem>(count);
   size = 0;
   for (i = 0; i<(unsigned int)count; i++) {
      int z = menuContent.find(':');
      int id = menuContent.left(z).toInt();
      MenuItem *tmp = MenuItem::getInstance(id);
      if (tmp) {
	 bool hasParent = FALSE;
	 MenuItem * parent = tmp->getParent();//in case new parent<->child relations are set up, this is needed to keep old user configuration working
	 while (parent) {
	    hasParent = TRUE;
	    tmp = parent;
	    parent = parent->getParent();
	 }
	 if ( (!hasParent) || (filePopupMenuItems->find(tmp) == -1) ) {
	    filePopupMenuItems->insert(size++,tmp);
	 }
      }
      menuContent = menuContent.mid(z+1);
   }
   filePopupMenuItems->resize(size);
  
   //setup key-shortcuts
   menuContent = KEYBOARDSHORTCUTS;
   count = menuContent.contains(':');
   for (i = 0; i<(unsigned int)count; i++) {
      int z = menuContent.find(':');
      QString h = menuContent.left(z);
      int sep = h.find(',');
      int id = h.left(sep).toInt();
      int accel = h.mid(sep+1).toInt();
      MenuItem *tmp = MenuItem::getInstance(id);
      if (tmp) MenuItem::setAccel(tmp,accel);
      menuContent = menuContent.mid(z+1);
   }
  
}

//----------------------------------------------------------------------------

//never set a new m_pCurCvsDir here since this gets called when a dir 
//is deleted and the selection changes
void ApplicationWindow::setMenuItemState(bool show) {

   //dir
   bool enabled = FALSE;
   bool dirSelected = FALSE;
   bool cvsControlled = FALSE;
   bool topDir = FALSE;
   bool topCvsDir = FALSE;
   bool hasClipboardData = FALSE;
   bool isPserver = FALSE;

   //files
   bool cvsItemSelected = FALSE;
   bool nonCvsItemSelected = FALSE;
   bool ignoreItemSelected = FALSE;
   bool onlyRemovedFilesSelected = FALSE;
   bool isDir = FALSE;
   bool isFilesAndDirs = FALSE;
   bool multiFileSelection = FALSE;

   if (show) {

      hasClipboardData = m_validClipboardData;

      DirBase * dir = m_pWorkBenchTree->selectedItem();
      if (dir) {

	 //------------ dir
	 dirSelected = TRUE;
	 if (!dir->isDisabled()) {
	    enabled = TRUE;
	 }
	 if (dir->isControlled()) {
	    cvsControlled = TRUE;
	 }
	 if (dir == dir->topDir()) {
	    topDir = TRUE;
	 }
	 if (dir == dir->topControlledDir()) {
	    topCvsDir = TRUE;
	 }
	 if (dir->connectMethod().find("pserver")==0) {
	    isPserver = TRUE;
	 }

	 //----------- file
	 int count = selectCount(2);
	 switch( m_pFileViews->currentPageIndex()) {

	    case 0: {// cvs-fileview
	       if (count > 0) {
		  cvsItemSelected = TRUE;
		  if (count > 1) {
		     multiFileSelection = TRUE;
		  }
		  if (checkOnlyRemovedFilesSelected()) {
		     onlyRemovedFilesSelected = TRUE;
		  }
	       }
	       break;
	    }
	    case 1: {// non-cvsview
	       if (count > 0) {
		  nonCvsItemSelected = TRUE;
		  if (count > 1) {
		     multiFileSelection = TRUE;
		  }
		  if (isDirsSelected()) {
		     isDir = TRUE;
		     if ( multiFileSelection && isFilesAndDirsSelected()) {
			isFilesAndDirs = TRUE;
		     }
		  }
	       }
	       break;
	    }
	    case 2: {// cvs-ignoreview
	       if (count > 0) {
		  ignoreItemSelected = TRUE;
		  if (isDirsSelected()) {
		     isDir = TRUE;
		     if ( (count > 1) && isFilesAndDirsSelected()) {
			isFilesAndDirs = TRUE;
		     }
		  }
	       }
	       break;
	    }
	    default: {
	       qDebug("There is no such page index: "+QString::number( m_pFileViews->currentPageIndex())+" implemented");
	       return;
	    }
	 }
      }
   }

   //now set states

   //dir
   M_PASTE->setEnabled(hasClipboardData);
   M_STATUS->setEnabled(enabled && cvsControlled);
   M_QUERYUPDATE->setEnabled(enabled && cvsControlled);
   M_QUERYUPDATEALL->setEnabled(enabled && cvsControlled);
   M_SHOWEDITORS->setEnabled(enabled && cvsControlled);
   M_SHOWWATCHERSDIR->setEnabled(enabled && cvsControlled);
   M_HISTORY->setEnabled(enabled && cvsControlled);

   M_UPDATE->setEnabled(enabled && cvsControlled);
   M_UPDATEALL->setEnabled(enabled && cvsControlled);
   M_REPLACEDIR->setEnabled(enabled && cvsControlled);
   M_REPLACEALLDIR->setEnabled(enabled && cvsControlled);
   M_COMMIT->setEnabled(enabled && cvsControlled);
   M_MERGEDIR->setEnabled(enabled && cvsControlled);
   M_LOGTREEDIR->setEnabled(enabled && cvsControlled);
   M_TAG->setEnabled(enabled && cvsControlled);
   M_DIFFTOCONSOLEDIR->setEnabled(enabled && cvsControlled);
   M_ADVDIFFTOCONSOLEDIR->setEnabled(enabled && cvsControlled);
   M_CREATEPATCHDIR->setEnabled(enabled && cvsControlled);
   M_UPDATEOTHER->setEnabled(enabled && cvsControlled);
   M_UPDATEDIRBRANCHDIR->setEnabled(enabled && cvsControlled);
   M_REMOVECVSBACKED->setEnabled(enabled);
   M_REMOVELOCALOPTIONSDIR->setEnabled(enabled && cvsControlled);

   M_WATCHONDIR->setEnabled(enabled && cvsControlled);
   M_WATCHOFFDIR->setEnabled(enabled && cvsControlled);
   M_WATCHADDDIR->setEnabled(enabled && cvsControlled);
   M_WATCHREMOVEDIR->setEnabled(enabled && cvsControlled);
   M_ANNOTATEDIR->setEnabled(enabled && cvsControlled);
   M_ANNOTATELINEDIR->setEnabled(enabled && cvsControlled);
   M_ANNOTATELINESDIR->setEnabled(enabled && cvsControlled);

   M_LOGIN->setEnabled(enabled && cvsControlled && isPserver);
   M_LOGOUT->setEnabled(enabled && cvsControlled && isPserver);

   M_BROWSEDIR->setEnabled(enabled);
   M_OPENSHELLDIR->setEnabled(enabled);
   M_EXPANDDIR->setEnabled(enabled);
   M_COLLAPSDIR->setEnabled(enabled);

   M_EDITDIR->setEnabled(enabled && cvsControlled);
   M_UNEDITDIR->setEnabled(enabled && cvsControlled);
   M_CONDITIONALEDITDIR->setEnabled(false);

   M_LOCKDIR->setEnabled(false);
   M_UNLOCKDIR->setEnabled(false);

   M_ENABLEDIR->setEnabled(!enabled && dirSelected);
   M_DISABLEDIR->setEnabled(enabled);
   M_ENABLETREE->setEnabled(dirSelected);
   M_DISABLEREVERTTREE->setEnabled(enabled && dirSelected);
   M_EXPORTDIR->setEnabled(enabled);
   M_IMPORTVENDORDIR->setEnabled(false);
   M_REREADDIR->setEnabled(enabled);
   M_REMOVEFROMWORKBENCH->setEnabled(dirSelected && topDir);
   M_RELEASE->setEnabled(enabled && dirSelected && cvsControlled && topCvsDir);
   M_REMOVEFROMDISKDIR->setEnabled(enabled && cvsControlled);
   M_REMOVEFROMREPANDDISKDIR->setEnabled(enabled && cvsControlled);
   M_RELEASEANDREMOVE->setEnabled(enabled && cvsControlled);

   M_RENAMEDIR->setEnabled(dirSelected && topDir);

   M_PROPERTIES->setEnabled(dirSelected);

   switch( m_pFileViews->currentPageIndex()) {//cvs-controled files

      case 0: {
	 M_OPENFILE->setEnabled( cvsItemSelected);
	 M_VIEWFILE->setEnabled( cvsItemSelected);
	 M_RESOLVE->setEnabled( !multiFileSelection && cvsItemSelected);

	 M_COPYFILE->setEnabled( cvsItemSelected);

	 M_STATUSFILE->setEnabled( cvsItemSelected);
	 M_QUERYUPDATEFILE->setEnabled( cvsItemSelected);
	 M_SHOWEDITORSFILE->setEnabled( cvsItemSelected);
	 M_SHOWWATCHERSFILE->setEnabled( cvsItemSelected);
	 M_HISTORYFILE->setEnabled( cvsItemSelected);
    
	 M_LOGTREEFILE->setEnabled( cvsItemSelected);
	 M_DIFFSIDEBYSIDEFILE->setEnabled( !multiFileSelection && cvsItemSelected);
	 M_DIFFTOCONSOLEFILE->setEnabled( cvsItemSelected);
	 M_ADVDIFFTOCONSOLEFILE->setEnabled( cvsItemSelected);

	 M_ANNOTATEFILE->setEnabled( !multiFileSelection && cvsItemSelected);
	 M_ANNOTATELINEFILE->setEnabled( !multiFileSelection && cvsItemSelected);
	 M_ANNOTATELINESFILE->setEnabled( !multiFileSelection && cvsItemSelected);
	 M_WATCHONFILE->setEnabled(cvsItemSelected);
	 M_WATCHOFFFILE->setEnabled(cvsItemSelected);
	 M_WATCHADDFILE->setEnabled(cvsItemSelected);
	 M_WATCHREMOVEFILE->setEnabled(cvsItemSelected);

    
	 M_ADDFILE->setEnabled( cvsItemSelected && onlyRemovedFilesSelected);
	 M_UPDATEFILE->setEnabled( cvsItemSelected);
	 M_REPLACEFILE->setEnabled( cvsItemSelected);
	 M_COMMITFILE->setEnabled( cvsItemSelected);
	 M_MERGEFILE->setEnabled( cvsItemSelected);
	 M_TAGFILE->setEnabled( cvsItemSelected);
	 M_UPDATEOTHERFILE->setEnabled( cvsItemSelected);
	 M_UPDATEDIRBRANCHFILE->setEnabled( cvsItemSelected);
	 M_REMOVELOCALOPTIONSFILE->setEnabled( cvsItemSelected);
	 M_CREATEPATCHFILE->setEnabled( cvsItemSelected);

	 M_EDITFILE->setEnabled( cvsItemSelected);
	 M_UNEDITFILE->setEnabled( cvsItemSelected);
	 M_CONDITIONALEDITFILE->setEnabled(false);
    
	 M_LOCKFILE->setEnabled(false);
	 M_UNLOCKFILE->setEnabled(false);
    
	 M_RENAMEFILE->setEnabled( !multiFileSelection && cvsItemSelected);

	 M_REMOVEFROMDISK->setEnabled( cvsItemSelected);
	 M_REMOVEFROMREPANDDISK->setEnabled( cvsItemSelected);
      
	 break;
      }
      case 1: {//non-cvs files
	 M_OPENFILE->setEnabled( !isDir && !isFilesAndDirs);
	 M_VIEWFILE->setEnabled( !isDir && !isFilesAndDirs);

	 M_COPYFILE->setEnabled( nonCvsItemSelected && !isDir && !isFilesAndDirs);

	 M_ADDFOLDER->setEnabled( cvsControlled && isDir && !isFilesAndDirs);
	 M_ADDASCIIFILE->setEnabled( cvsControlled && !isDir && !isFilesAndDirs);
	 M_ADDBINARYFILE->setEnabled( cvsControlled && !isDir && !isFilesAndDirs);

	 M_RENAMENONCVSFILE->setEnabled( !multiFileSelection && nonCvsItemSelected);
	 M_DELETENONCVSFROMDISKFILE->setEnabled( true);

	 M_CREATENEWFILE->setEnabled( true);
	 M_CREATENEWDIR->setEnabled( true);

	 break;
      }
      case 2: {//ignored files
	 M_OPENFILE->setEnabled( ignoreItemSelected && !isDir && !isFilesAndDirs);
	 M_VIEWFILE->setEnabled( ignoreItemSelected && !isDir && !isFilesAndDirs);

	 M_COPYFILE->setEnabled( ignoreItemSelected && !isDir && !isFilesAndDirs);
      
	 M_ADDFOLDER->setEnabled( cvsControlled && ignoreItemSelected && isDir && !isFilesAndDirs);
	 M_ADDASCIIFILE->setEnabled( cvsControlled && ignoreItemSelected && !isDir && !isFilesAndDirs);
	 M_ADDBINARYFILE->setEnabled( cvsControlled && ignoreItemSelected && !isDir && !isFilesAndDirs);
      
	 M_DELETEIGNOREFROMDISKFILE->setEnabled( ignoreItemSelected);

	 break;
      }
      default: {}
   };

}

//----------------------------------------------------------------------------

void ApplicationWindow::fillProjectMenu() {

   m_pProjectMenu->clear();
   insertMenuItem(m_pProjectMenu, M_INITLOCALCVS);
   insertMenuItem(m_pProjectMenu, M_IMPORTPROJECT);
   insertMenuItem(m_pProjectMenu, M_SEPARATOR);
   insertMenuItem(m_pProjectMenu, M_ADDPROJECT);
   insertMenuItem(m_pProjectMenu, M_SEPARATOR);
   insertMenuItem(m_pProjectMenu, M_CHECKOUTPROJECT);
   insertMenuItem(m_pProjectMenu, M_EXPORTPROJECT);
   insertMenuItem(m_pProjectMenu, M_SEPARATOR);
   insertMenuItem(m_pProjectMenu, M_REFRESHOPTIONS);
   insertMenuItem(m_pProjectMenu, M_SEPARATOR);
   insertMenuItem(m_pProjectMenu, M_EXITPROJECT);

}

//----------------------------------------------------------------------------

//never set a new m_pCurCvsDir here since this gets called when a dir 
//is deleted and the selection changes
void ApplicationWindow::fillDirMenu(bool show) {

   m_pDirMenu->clear();
   m_pDirProjectSubMenu->clear();
   m_pDirStateSubMenu->clear();
   m_pDirWatchSubMenu->clear();
   m_pDirImportSubMenu->clear();
   m_pDirUpdateSubMenu->clear();
   m_pDirAnnotateSubMenu->clear();
   m_pDirDiffSubMenu->clear();
   m_pDirTreeSubMenu->clear();
   m_pDirEditSubMenu->clear();
   m_pDirLockSubMenu->clear();

   if (show) {
      DirBase * dir = m_pWorkBenchTree->selectedItem();
      if ( dir && dir->isVirtual()) {
	 insertMenuItem( m_pDirMenu, M_UPDATEVIRTUALDIR);
	 return;
      } else {

	 insertMenuItem( m_pDirMenu, M_PASTE);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirProjectSubMenu, M_REMOVEFROMWORKBENCH);
	 insertMenuItem( m_pDirProjectSubMenu, M_RENAMEDIR);
	 insertMenuItem( m_pDirProjectSubMenu, M_SEPARATOR);
	 insertMenuItem( m_pDirProjectSubMenu, M_LOGIN);
	 insertMenuItem( m_pDirProjectSubMenu, M_LOGOUT);
	 insertMenuItem( m_pDirProjectSubMenu, M_SEPARATOR);
	 insertMenuItem( m_pDirProjectSubMenu, M_PROPERTIES);
	 m_pDirMenu->insertItem(tr("Project"),m_pDirProjectSubMenu);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirStateSubMenu, M_STATUS);
	 insertMenuItem( m_pDirStateSubMenu, M_HISTORY);
	 insertMenuItem( m_pDirStateSubMenu, M_SEPARATOR);
	 insertMenuItem( m_pDirStateSubMenu, M_LOGTREEDIR);
	 insertMenuItem( m_pDirStateSubMenu, M_SEPARATOR);
	 insertMenuItem( m_pDirStateSubMenu, M_SHOWEDITORS);
	 insertMenuItem( m_pDirStateSubMenu, M_SHOWWATCHERSDIR);
	 m_pDirMenu->insertItem(tr("State"),m_pDirStateSubMenu);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirWatchSubMenu, M_WATCHONDIR);
	 insertMenuItem( m_pDirWatchSubMenu, M_WATCHOFFDIR);
	 insertMenuItem( m_pDirWatchSubMenu, M_WATCHADDDIR);
	 insertMenuItem( m_pDirWatchSubMenu, M_WATCHREMOVEDIR);
	 m_pDirMenu->insertItem(tr("Watch"),m_pDirWatchSubMenu);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirImportSubMenu, M_IMPORTVENDORDIR);
	 insertMenuItem( m_pDirImportSubMenu, M_EXPORTDIR);
	 m_pDirMenu->insertItem(tr("Import/Export"),m_pDirImportSubMenu);

	 insertMenuItem( m_pDirUpdateSubMenu, M_QUERYUPDATE);
	 insertMenuItem( m_pDirUpdateSubMenu, M_QUERYUPDATEALL);
	 insertMenuItem( m_pDirUpdateSubMenu, M_SEPARATOR);
	 insertMenuItem( m_pDirUpdateSubMenu, M_UPDATE);
	 insertMenuItem( m_pDirUpdateSubMenu, M_UPDATEALL);
	 insertMenuItem( m_pDirUpdateSubMenu, M_SEPARATOR);
	 insertMenuItem( m_pDirUpdateSubMenu, M_REPLACEDIR);
	 insertMenuItem( m_pDirUpdateSubMenu, M_REPLACEALLDIR);
	 insertMenuItem( m_pDirUpdateSubMenu, M_SEPARATOR);
	 insertMenuItem( m_pDirUpdateSubMenu, M_UPDATEDIRBRANCHDIR);
	 insertMenuItem( m_pDirUpdateSubMenu, M_UPDATEOTHER);
	 m_pDirMenu->insertItem(tr("Update"),m_pDirUpdateSubMenu);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirAnnotateSubMenu, M_ANNOTATEDIR);
	 insertMenuItem( m_pDirAnnotateSubMenu, M_ANNOTATELINEDIR);
	 insertMenuItem( m_pDirAnnotateSubMenu, M_ANNOTATELINESDIR);
	 m_pDirMenu->insertItem(tr("Annotate"),m_pDirAnnotateSubMenu);
	 insertMenuItem( m_pDirMenu, M_COMMIT);
	 insertMenuItem( m_pDirMenu, M_MERGEDIR);
	 insertMenuItem( m_pDirMenu, M_TAG);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirDiffSubMenu, M_DIFFTOCONSOLEDIR);
	 insertMenuItem( m_pDirDiffSubMenu, M_ADVDIFFTOCONSOLEDIR);
	 insertMenuItem( m_pDirDiffSubMenu, M_CREATEPATCHDIR);
	 m_pDirMenu->insertItem(tr("Diff/Patch"),m_pDirDiffSubMenu);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirMenu, M_BROWSEDIR);
	 insertMenuItem( m_pDirMenu, M_OPENSHELLDIR);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirTreeSubMenu, M_REREADDIR);
	 insertMenuItem( m_pDirTreeSubMenu, M_REMOVECVSBACKED);
	 insertMenuItem( m_pDirTreeSubMenu, M_REMOVELOCALOPTIONSDIR);
	 insertMenuItem( m_pDirTreeSubMenu, M_SEPARATOR);
	 insertMenuItem( m_pDirTreeSubMenu, M_ENABLEDIR);
	 insertMenuItem( m_pDirTreeSubMenu, M_DISABLEDIR);
	 insertMenuItem( m_pDirTreeSubMenu, M_ENABLETREE);
	 insertMenuItem( m_pDirTreeSubMenu, M_DISABLEREVERTTREE);
	 m_pDirMenu->insertItem(tr("Tree"),m_pDirTreeSubMenu);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirMenu, M_EXPANDDIR);
	 insertMenuItem( m_pDirMenu, M_COLLAPSDIR);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirEditSubMenu, M_EDITDIR);
	 insertMenuItem( m_pDirEditSubMenu, M_UNEDITDIR);
	 insertMenuItem( m_pDirEditSubMenu, M_CONDITIONALEDITDIR);
	 m_pDirMenu->insertItem(tr("Edit/Unedit"),m_pDirEditSubMenu);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirMenu, M_RELEASE);
	 insertMenuItem( m_pDirMenu, M_RELEASEANDREMOVE);
	 insertMenuItem( m_pDirMenu, M_REMOVE);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirLockSubMenu, M_LOCKDIR);
	 insertMenuItem( m_pDirLockSubMenu, M_UNLOCKDIR);
	 m_pDirMenu->insertItem(tr("Lock/Unlock"),m_pDirLockSubMenu);

	 insertMenuItem( m_pDirMenu, M_SEPARATOR);

	 insertMenuItem( m_pDirMenu, M_CUSTOMIZE);
      }
   }

   if (Debug::g_bDebug) {//testing only, do !not! remove
      insertMenuItem( m_pDirMenu, M_SEPARATOR);
      insertMenuItem( m_pDirMenu, M_TEST);
   }
}

//----------------------------------------------------------------------------

//never set a new m_pCurCvsDir here since this gets called when a dir 
//is deleted and the selection changes
void ApplicationWindow::fillFileMenu(bool show) {

   m_pFileMenu->clear();
   m_pFileStateSubMenu->clear();
   m_pFileWatchSubMenu->clear();
   m_pFileSubMenu->clear();
   m_pFileAnnotateSubMenu->clear();
   m_pFileDiffSubMenu->clear();
   m_pFileLockSubMenu->clear();

   if (show) {
      switch( m_pFileViews->currentPageIndex()) {

	 case 0: {// cvsview

	    if ( checkOnlyVirtualDirFileItemsSelected() ) {
	       insertMenuItem( m_pFileMenu, M_UPDATEVIRTUALDIR);
	       break;
	    } else {

	       insertMenuItem( m_pFileMenu, M_COPYFILE);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileMenu, M_OPENFILE);
	       insertMenuItem( m_pFileMenu, M_VIEWFILE);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileStateSubMenu, M_STATUSFILE);
	       insertMenuItem( m_pFileStateSubMenu, M_HISTORYFILE);
	       insertMenuItem( m_pFileStateSubMenu, M_SEPARATOR);
	       insertMenuItem( m_pFileStateSubMenu, M_SHOWEDITORSFILE);
	       insertMenuItem( m_pFileStateSubMenu, M_SHOWWATCHERSFILE);
	       insertMenuItem( m_pFileStateSubMenu, M_SEPARATOR);
	       insertMenuItem( m_pFileStateSubMenu, M_LOGTREEFILE);
	       m_pFileMenu->insertItem(tr("State"),m_pFileStateSubMenu);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileWatchSubMenu, M_WATCHONFILE);
	       insertMenuItem( m_pFileWatchSubMenu, M_WATCHOFFFILE);
	       insertMenuItem( m_pFileWatchSubMenu, M_WATCHADDFILE);
	       insertMenuItem( m_pFileWatchSubMenu, M_WATCHREMOVEFILE);
	       m_pFileMenu->insertItem(tr("Watch"),m_pFileWatchSubMenu);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileSubMenu, M_QUERYUPDATEFILE);
	       insertMenuItem( m_pFileSubMenu, M_SEPARATOR);
	       insertMenuItem( m_pFileSubMenu, M_UPDATEFILE);
	       insertMenuItem( m_pFileSubMenu, M_REPLACEFILE);
	       insertMenuItem( m_pFileSubMenu, M_SEPARATOR);
	       insertMenuItem( m_pFileSubMenu, M_UPDATEDIRBRANCHFILE);
	       insertMenuItem( m_pFileSubMenu, M_UPDATEOTHERFILE);
	       m_pFileMenu->insertItem(tr("Update"),m_pFileSubMenu);

	       insertMenuItem( m_pFileMenu, M_ADDFILE);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileAnnotateSubMenu, M_ANNOTATEFILE);
	       insertMenuItem( m_pFileAnnotateSubMenu, M_ANNOTATELINEFILE);
	       insertMenuItem( m_pFileAnnotateSubMenu, M_ANNOTATELINESFILE);
	       m_pFileMenu->insertItem(tr("Annotate"),m_pFileAnnotateSubMenu);
	       insertMenuItem( m_pFileMenu, M_COMMITFILE);
	       insertMenuItem( m_pFileMenu, M_MERGEFILE);
	       insertMenuItem( m_pFileMenu, M_TAGFILE);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileDiffSubMenu, M_DIFFSIDEBYSIDEFILE);
	       insertMenuItem( m_pFileDiffSubMenu, M_DIFFTOCONSOLEFILE);
	       insertMenuItem( m_pFileDiffSubMenu, M_ADVDIFFTOCONSOLEFILE);
	       insertMenuItem( m_pFileDiffSubMenu, M_CREATEPATCHFILE);
	       m_pFileMenu->insertItem(tr("Diff/Patch"),m_pFileDiffSubMenu);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileMenu, M_EDITFILE);
	       insertMenuItem( m_pFileMenu, M_UNEDITFILE);
	       insertMenuItem( m_pFileMenu, M_CONDITIONALEDITFILE);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileMenu, M_RENAMEFILE);
	       insertMenuItem( m_pFileMenu, M_REMOVEFILE);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileLockSubMenu, M_LOCKFILE);
	       insertMenuItem( m_pFileLockSubMenu, M_UNLOCKFILE);
	       m_pFileMenu->insertItem(tr("Lock/Unlock"),m_pFileLockSubMenu);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileMenu, M_RESOLVE);
	       insertMenuItem( m_pFileMenu, M_REMOVELOCALOPTIONSFILE);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileMenu, M_CUSTOMIZEFILE);

	    }
	    break;
	 }

	 case 1: {// non-cvsview

	    if ( selectCount(1) > 0) {

	       insertMenuItem( m_pFileMenu, M_COPYFILE);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileMenu, M_OPENFILE);
	       insertMenuItem( m_pFileMenu, M_VIEWFILE);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileMenu, M_ADDFOLDER);
	       insertMenuItem( m_pFileMenu, M_ADDASCIIFILE);
	       insertMenuItem( m_pFileMenu, M_ADDBINARYFILE);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	       insertMenuItem( m_pFileMenu, M_RENAMENONCVSFILE);
	       insertMenuItem( m_pFileMenu, M_DELETENONCVSFROMDISKFILE);

	       insertMenuItem( m_pFileMenu, M_SEPARATOR);

	    } else {

	       insertMenuItem( m_pFileMenu, M_CREATENEWFILE);
	       insertMenuItem( m_pFileMenu, M_CREATENEWDIR);
	    }
	    break;
	 }

	 case 2: {// cvs-ignoreview

	    insertMenuItem( m_pFileMenu, M_COPYFILE);

	    insertMenuItem( m_pFileMenu, M_SEPARATOR);

	    insertMenuItem( m_pFileMenu, M_OPENFILE);
	    insertMenuItem( m_pFileMenu, M_VIEWFILE);
      
	    insertMenuItem( m_pFileMenu, M_SEPARATOR);
      
	    insertMenuItem( m_pFileMenu, M_ADDFOLDER);
	    insertMenuItem( m_pFileMenu, M_ADDASCIIFILE);
	    insertMenuItem( m_pFileMenu, M_ADDBINARYFILE);

	    insertMenuItem( m_pFileMenu, M_SEPARATOR);

	    insertMenuItem( m_pFileMenu, M_DELETEIGNOREFROMDISKFILE);

	    break;
	 }

	 default: {
	    return;
	 }
      }
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::fillConfigMenu() {

   m_pConfigMenu->clear();
   insertMenuItem(m_pConfigMenu,M_SETTINGSOPTIONS);
   insertMenuItem(m_pConfigMenu,M_PROFILESOPTIONS);
   insertMenuItem(m_pConfigMenu,M_SEPARATOR);
   insertMenuItem(m_pConfigMenu,M_KEYSHORTCUTSOPTIONS);
   insertMenuItem(m_pConfigMenu,M_SEPARATOR);
   insertMenuItem(m_pConfigMenu,M_OPENFILEMAPPINGOPTIONS);
   insertMenuItem(m_pConfigMenu,M_VIEWFILEMAPPINGOPTIONS);
}

//----------------------------------------------------------------------------

void ApplicationWindow::fillHelpMenu() {

   m_pHelpMenu->clear();
   MenuItem * item = MenuItem::first();
   do {
      if (item->menu() == TMenu::HELP ) {
	 insertMenuItem(m_pHelpMenu,item);
      }
      if (item == M_DEBUGLOGHELP ) {
	 insertMenuItem(m_pHelpMenu,M_SEPARATOR);
      }
   } while ( (item = item->nextItem()) );
}

//----------------------------------------------------------------------------

void ApplicationWindow::fillDirPopupMenu() {
  
   dirPopupMenu->clear();

   DirBase * dir = m_pWorkBenchTree->selectedItem();
   if (dir) {
      if (dir->isVirtual() ) {
	 insertMenuItem(dirPopupMenu,M_UPDATEVIRTUALDIR);
      } else {
	 unsigned int i;
	 for( i = 0; i < dirPopupMenuItems->size(); i++ ) {
	    MenuItem * item = dirPopupMenuItems->at(i);
	    insertMenuItem(dirPopupMenu,item);
	 }
      }
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::fillFilePopupMenu() {

   filePopupMenu->clear();
   DirBase * dir = m_pWorkBenchTree->selectedItem();

   if (dir) {

      switch( m_pFileViews->currentPageIndex()) {

	 case 0: {// cvsview

	    if ( selectCount(1)) {       
	       if ( checkOnlyVirtualDirFileItemsSelected() ) {
		  insertMenuItem(filePopupMenu,M_UPDATEVIRTUALDIR);
	       } else {
		  unsigned int i;
		  for( i = 0; i<filePopupMenuItems->size(); i++ ) {
		     insertMenuItem(filePopupMenu,filePopupMenuItems->at(i));
		  }
	       }
	    }
	    break;
	 }
	 case 1: {

	    if ( selectCount(1) > 0) {
	       insertMenuItem( filePopupMenu, M_OPENFILE);
	       insertMenuItem( filePopupMenu, M_VIEWFILE);
	  
	       insertMenuItem( filePopupMenu, M_SEPARATOR);
	  
	       insertMenuItem( filePopupMenu, M_ADDFOLDER);
	       insertMenuItem( filePopupMenu, M_ADDASCIIFILE);
	       insertMenuItem( filePopupMenu, M_ADDBINARYFILE);
	  
	       insertMenuItem( filePopupMenu, M_SEPARATOR);
	  
	       insertMenuItem( filePopupMenu, M_RENAMENONCVSFILE);
	       insertMenuItem( filePopupMenu, M_DELETENONCVSFROMDISKFILE);
	  
	       insertMenuItem( filePopupMenu, M_SEPARATOR);
	  
	    } else {
	       insertMenuItem( filePopupMenu, M_CREATENEWFILE);
	       insertMenuItem( filePopupMenu, M_CREATENEWDIR);
	    }
	    break;
	 }
	 case 2: {

	    insertMenuItem( filePopupMenu, M_OPENFILE);
	    insertMenuItem( filePopupMenu, M_VIEWFILE);
	
	    insertMenuItem( filePopupMenu, M_SEPARATOR);
	
	    insertMenuItem( filePopupMenu, M_ADDFOLDER);
	    insertMenuItem( filePopupMenu, M_ADDASCIIFILE);
	    insertMenuItem( filePopupMenu, M_ADDBINARYFILE);
	
	    insertMenuItem( filePopupMenu, M_SEPARATOR);
	
	    insertMenuItem( filePopupMenu, M_DELETEIGNOREFROMDISKFILE);
	    break;
	 }
	 default: {}
      }
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::fillOutputPopupMenu(int mode) {

   bool hasText = !m_pMessages->text().isEmpty();
   M_COPYOUTPUT->setEnabled(hasText);
   M_SELECTALLOUTPUT->setEnabled(hasText);
   M_TOGGLESHOWALLOUTPUT->setEnabled(mode == CCvsOutput::QUIET);
   M_CLEAROUTPUT->setEnabled(hasText);

   m_pOutputMenu = new Menu(this);
  
   MenuItem * item = MenuItem::first();
   do {
      if (item->menu() == TMenu::OUTPUT ) {
	 insertMenuItem(m_pOutputMenu,item);
      }
   } while ( (item = item->nextItem()) );
}

//----------------------------------------------------------------------------

void ApplicationWindow::copyOutput() {
   m_pMessages->copy();
}

//----------------------------------------------------------------------------

void ApplicationWindow::selectAllOutput() {
   m_pMessages->selectAll(TRUE);
}

//----------------------------------------------------------------------------

void ApplicationWindow::toggleShowAllOutput() {
   m_pMessages->toggleShowAll();
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_clearOutput() {
   m_pMessages->reset();
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_renameNonCvsFile() {
   DirBase * dir;
   if (!(dir = getSelectedDir())) {
      return;
   }

   QStringList list = dir->getSelectedFiles();
   if (list.isEmpty()) return;

   QString fname = list[0];
   bool ok;
   QString newName = QInputDialog::getText(tr("Rename"), tr("New name: "),
	 QLineEdit::Normal,
	 fname, &ok, this ).stripWhiteSpace();
  
   if( !ok || fname.isEmpty() || (fname == newName) ) {
      return;
   }
   QString path = dir->fullName()+"/";
   if (QFileInfo(path+newName).exists()) {
      if (!showOkCancelQuestion( tr("Rename"),
		  tr("Cannot rename file") + "\n"
		  + tr("File exists:") + "\n"
		  + path+newName + "\n\n"
		  + tr("Overwrite?") )) {
	 return;
      }
   }
  
   if (!rename( path+fname, path+newName)) {
      showWarning(tr("Rename"), tr("Cannot rename:") + "\n" + path + fname + ",\n" + tr("check permissions"));
   } else dir->activateItem(FALSE);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_renameLocalDir() {

   Directory * dir;
   if (!(dir = getSelectedDir())) {
      return;
   }

   QString fname = dir->shortName();
   bool ok;
   QString newName = QInputDialog::getText(tr("Rename"), tr("New name: "),
	 QLineEdit::Normal,
	 fname, &ok, this ).stripWhiteSpace();
  
   if( !ok || fname.isEmpty() || (fname == newName) ) {
      return;
   }
  
   blockGUI(true);
   QString path = dir->fullName().left(dir->fullName().findRev("/")+1);
   renameDir(dir, path+newName);
   blockGUI(false);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_createPatchDir() {
   Patch::patchDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, getCurrentTagList());
}

void ApplicationWindow::slot_createPatchFile() {
   Patch::patchFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, getCurrentTagList());
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_statusFile() {
   Status::statusFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_statusDir() {
   Status::statusDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_editFile() {
   Edit::editFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_uneditFile() {
   Edit::uneditFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_showEditorsFile() {
   Edit::showEditorsFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_editDir() {
   Edit::editDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_uneditDir() {
   Edit::uneditDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_showEditorsDir() {
   Edit::showEditorsDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_watchOnAny() {
   if (checkFilesSelected()) slot_watchOnFile();
   else slot_watchOnDir();
}

void ApplicationWindow::slot_watchOffAny() {
   if (checkFilesSelected()) slot_watchOffFile();
   else slot_watchOffDir();
}

void ApplicationWindow::slot_watchAddAny() {
   if (checkFilesSelected()) slot_watchAddFile();
   else slot_watchAddDir();
}

void ApplicationWindow::slot_watchRemoveAny() {
   if (checkFilesSelected()) slot_watchRemoveFile();
   else slot_watchRemoveDir();
}

void ApplicationWindow::slot_watchOnFile() {
   Watch::watchOnFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_watchOffFile() {
   Watch::watchOffFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_watchAddFile() {
   Watch::watchAddFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_watchRemoveFile() {
   Watch::watchRemoveFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_showWatchersFile() {
   Watch::showWatchersFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_watchOnDir() {
   Watch::watchOnDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_watchOffDir() {
   Watch::watchOffDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_watchAddDir() {
   Watch::watchAddDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_watchRemoveDir() {
   Watch::watchRemoveDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_showWatchersDir() {
   Watch::showWatchersDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_mergeAny() {
   if (checkFilesSelected()) slot_mergeFile();
   else slot_mergeDir();
}

void ApplicationWindow::slot_mergeFile() {
   Merge::mergeFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, getCurrentTagList());
}

void ApplicationWindow::slot_mergeDir() {
   Merge::mergeDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, getCurrentTagList());
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_diffFile() {
   Diff::diffFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_diffAdvancedFile() {
   Diff::diffAFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, getCurrentTagList());
}

void ApplicationWindow::slot_diffDir() {
   Diff::diffDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_diffAdvancedDir() {
   Diff::diffADir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, getCurrentTagList());
}

void ApplicationWindow::slot_diffFileSideBySide() {
   Diff::diffFileSideBySide( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_annotateAny() {
   if (checkFilesSelected()) slot_annotateFile();
   else slot_annotateDir();
}

void ApplicationWindow::slot_annotateFile() {
   Annotate::annotateFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_annotateDir() {
   Annotate::annotateDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_annotateGrepLineAny() {
   if (checkFilesSelected()) slot_annotateGrepLineFile();
   else slot_annotateGrepLineDir();
}

void ApplicationWindow::slot_annotateGrepLineFile() {
   Annotate::annotateGrepLineFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_annotateGrepLineDir() {
   Annotate::annotateGrepLineDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_annotateGrepLinesAny() {
   if (checkFilesSelected()) slot_annotateGrepLinesFile();
   else slot_annotateGrepLinesDir();
}

void ApplicationWindow::slot_annotateGrepLinesFile() {
   Annotate::annotateGrepLinesFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_annotateGrepLinesDir() {
   Annotate::annotateGrepLinesDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_historyFile() {
   History::historyFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_historyDir() {
   History::historyDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_logFile() {
   Log::logFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_logDir() {
   Log::logDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_updateAny() {
   if (checkFilesSelected()) slot_updateFile();
   else slot_updateDir();
}

void ApplicationWindow::slot_queryUpdateFile() {
   Update::queryUpdateFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_updateFile() {
   Update::updateFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_updateReplaceFile() {
   Update::updateReplaceFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_updateToDirBranchFile() {
   Update::updateToDirBranchFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_updateOtherRevisionFile() {
   Update::updateOtherRevisionFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, getCurrentTagList());
}

void ApplicationWindow::slot_removeLocalOptionsFile() {
   Update::removeLocalOptionsFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_queryUpdateDir() {
   Update::queryUpdateDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_queryUpdateAllDir() {
   Update::queryUpdateAllDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_updateDir() {
   Update::updateDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_updateAllDir() {
   Update::updateAllDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_updateReplaceDir() {
   Update::updateReplaceDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_updateReplaceAllDir() {
   Update::updateReplaceAllDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_updateToDirBranchDir() {
   Update::updateToDirBranchDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_updateOtherRevisionDir() {
   Update::updateOtherRevisionDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, getCurrentTagList());
}

void ApplicationWindow::slot_updateVirtualDir() {
   Update::updateVirtualDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_removeLocalOptionsDir() {
   Update::removeLocalOptionsDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_tagFile() {
   Tag::tagFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, getCurrentTagList());
}

void ApplicationWindow::slot_tagDir() {
   Tag::tagDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, getCurrentTagList());
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_checkoutProject() {
   Checkout::checkoutProject( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_exportProject() {
   Checkout::exportProject( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_importProject() {
   Import::importProject( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_commitAny() {
   if (checkFilesSelected()) slot_commitFile();
   else slot_commitDir();
}

void ApplicationWindow::slot_commitFile() {
   Commit::commitFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_commitDir() {
   Commit::commitDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_release() {
   Release::release( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_releaseAndRemove() {
   Release::releaseAndRemove( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_addFileAscii() {
   Add::addFileAscii( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_addFileBinary() {
   Add::addFileBinary( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

void ApplicationWindow::slot_addDir() {
   Add::addDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_resurrectFile() {
   Add::addResurrect( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_removeFile() {
   if (showVerifyDialog(tr("Warning"),tr("This command will remove file(s) from disk!"))) {
      Remove::removeFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
   }
}

void ApplicationWindow::slot_removeDir() {
   if (showVerifyDialog(tr("Warning"),tr("This command will remove file(s) from disk!"))) {
      Remove::removeDir( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_autoUpdate() {
   AutoCheck::autoCheck( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_renameFile() {
   Rename::renameFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_addFiles(const QStringList list, bool binary) {
   QStringList fileList;
   DirBase * dir = NULL;
   int pos = -1;
   QStringList::ConstIterator it;
   for (it = list.begin(); it != list.end(); it++) {
      if (!dir) {
	 pos = (*it).findRev("/");
	 dir = m_pWorkBenchTree->find( (*it).left(pos) );
	 if (!dir) continue;
	 ++pos;
      }
      fileList.append( (*it).mid(pos) );
   }
   if (!fileList.isEmpty()) {
      if (binary) {
	 Add::addSelectionBinary( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, dir, fileList, NULL);
      } else {
	 Add::addSelectionAscii( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, dir, fileList, NULL);
      }
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_moveFiles(const QString dropDir, const QStringList list) {
   DirBase * dir = m_pWorkBenchTree->find(dropDir);
   assert(dir);
   Move::moveSelectionFile( m_pTools->whatsThisIconSet(), this, m_pWorkBenchTree, dir, list, NULL);
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_copyFile() {
   m_clipboardFileList = m_pFileViews->currentPage()->copyToClipboard();
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_paste() {
   m_pWorkBenchTree->paste(m_clipboardFileList);
}

//----------------------------------------------------------------------------

void ApplicationWindow::showFileMenu( bool) {//bool param true if there are selected items
   //fill and set states on every selectionChanged(), since this affects the key-shortcuts
   setMenuItemState(TRUE);
   fillFileMenu(TRUE);
}

//----------------------------------------------------------------------------

void ApplicationWindow::setDirMenu( bool) {//bool param true if there are selected items
   //set states on every selectionChanged(), since this affects the key-shortcuts
   if (!m_onDrag && m_started) {
      setMenuItemState(TRUE);
      fillProjectMenu();
      fillDirMenu(TRUE);
      fillFileMenu(TRUE);
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::clipboardDataChanged() {

   bool prevState = m_validClipboardData;

#ifndef NO_CLIPBOARD
   if (QApplication::clipboard()
	 && QApplication::clipboard()->data(QClipboard::Clipboard)
	 && QUriDrag::canDecode(QApplication::clipboard()->data(QClipboard::Clipboard) ) ) {
      m_validClipboardData = TRUE;
   }
#endif

   if (prevState != m_validClipboardData) {
      setMenuItemState(TRUE);
      fillProjectMenu();
      fillDirMenu(TRUE);
      fillFileMenu(TRUE);
   }
}

//----------------------------------------------------------------------------

bool ApplicationWindow::checkOnlyRemovedFilesSelected() {
   bool retval = true;
    
   QListViewItem *myChild = static_cast<QListView*>(m_pFileViews->page(0))->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
	 if (!myChild->text(4).stripWhiteSpace().startsWith(tr("commit to remove"))) {
	    retval = false;
	    break;
	 }
      }
      myChild = myChild->nextSibling();
   }
   return retval;
}

//----------------------------------------------------------------------------

bool ApplicationWindow::checkOnlyVirtualDirFileItemsSelected() {
    
   bool retval = false;
   DirBase * dir = m_pWorkBenchTree->selectedItem();
   if (dir) {
      QListViewItem *myChild = static_cast<QListView*>(m_pFileViews->page(0))->firstChild();
      while(myChild) {
         if(myChild->isSelected()) {
	    if (dir->getEntryState(myChild->text(0)) != ES_missing_dir) {
	       return false;
	    } else {
	       retval = true;
	    }
	 }
	 myChild = myChild->nextSibling();
      }
   }
   return retval;
}

//----------------------------------------------------------------------------

int ApplicationWindow::selectCount(const int & max) {
   int retval = 0;
    
   FileListViewItem * myChild = m_pFileViews->currentFirstChild();
   while(myChild) {
      if(myChild->isSelected()) {
	 ++retval;
	 if (retval >= max) {
	    return retval;
	 }
      }
      myChild = myChild->nextSibling();
   }
   return retval;
}

//----------------------------------------------------------------------------

bool ApplicationWindow::isDirsSelected() {

   FileListViewItem * item = m_pFileViews->currentFirstChild();
   while (item) {
      if (item->isSelected() && item->isDir()) {
	 return TRUE;
      }
      item = item->nextSibling();
   }
   return FALSE;
}

//----------------------------------------------------------------------------

bool ApplicationWindow::isFilesAndDirsSelected() {
   bool dirSelected = FALSE;
   bool fileSelected = FALSE;
   FileListViewItem * item = m_pFileViews->currentFirstChild();

   while (item && !( dirSelected && fileSelected)) {
      if (item->isSelected()) {
	 if (item->isDir()) {
	    dirSelected = TRUE;
	 } else {
	    fileSelected = TRUE;
	 }
      }
      item = item->nextSibling();
   }
   return dirSelected && fileSelected;
}

//----------------------------------------------------------------------------

void ApplicationWindow::initComponents()
{
   resize( 600, 400 );

   
   /** Add workbench pane to the main window */
   CFlatSplitter *m_pHorzSplitter = new CFlatSplitter(
						      CFlatSplitter::Horizontal, this);
   m_pHorzSplitter->setMargin(2);
   m_pHorzSplitter->setOpaqueResize(true);
   m_pWorkBenchTree = new CvsDirListView(dirPopupMenu, m_pHorzSplitter);
   m_pWorkBenchTree->setFocusPolicy(QWidget::StrongFocus);
   m_pWorkBenchTree->setRootIsDecorated(true);
   m_pWorkBenchTree->addColumn(tr("Workbench"));
   m_pWorkBenchTree->header()->hide();
   m_pWorkBenchTree->setColumnWidth(0, WORKBENCH_COLUMN_WIDTH);
   m_pTools->addWhatsThisText(m_pWorkBenchTree, m_pTools->wBench);
   setCentralWidget(m_pHorzSplitter);

   /** Restore horizontal widget positions */
   QValueList<int> HorzSplitterPos;
   HorzSplitterPos.append(SPLITTER_POS_H);
   m_pHorzSplitter->setResizeMode(m_pWorkBenchTree, CFlatSplitter::KeepSize);
   m_pHorzSplitter->setSizes(HorzSplitterPos);

   connect(m_pWorkBenchTree, SIGNAL(showDirMenu(bool)),
	 SLOT(setDirMenu(bool)));
   connect(m_pWorkBenchTree, SIGNAL(setStatusText( const QString &, int)),
	 this, SLOT(setStatusText( const QString &, int)));
   /** drag'n drop */
   connect(m_pWorkBenchTree, SIGNAL(folderDropped(const QString)),
	 SLOT(slot_dropProject(const QString)));
   connect(m_pWorkBenchTree, SIGNAL(addCvsFiles(const QStringList,bool)),
	 SLOT(slot_addFiles(const QStringList,bool)));
   connect(m_pWorkBenchTree, SIGNAL(moveCvsFiles(const QString,const QStringList)),
	 SLOT(slot_moveFiles(const QString,const QStringList)));
   connect(m_pWorkBenchTree, SIGNAL(dirClicked(DirBase *)),SLOT(slot_dirClicked(DirBase *)));
 
   /** Create the work file pane and tab widget on the main window */
   CFlatSplitter *m_pVertSplitter = new CFlatSplitter(
						      CFlatSplitter::Vertical, m_pHorzSplitter);
   m_pFileViews = new ColorTabWidget(m_pVertSplitter);
   m_pFileViews->setMargin(2);
   m_pTools->addWhatsThisText(m_pFileViews, m_pTools->tabFileList);

   /** Restore vertical widget positions */
   QValueList<int> VertSplitterPos;
   VertSplitterPos.append(SPLITTER_POS_V);
   m_pVertSplitter->setResizeMode(m_pFileViews, CFlatSplitter::KeepSize);
   m_pVertSplitter->setSizes(VertSplitterPos);
   m_pVertSplitter->setOpaqueResize(true);

   /** Add cvs file view page to the main window */
   CvsListView * controledFileList = new CvsListView(filePopupMenu, m_pFileViews);
   controledFileList->setFocusPolicy(QWidget::StrongFocus);
   controledFileList->setHeaderSetup(SizeProperties::g_fileHeader);
   m_pFileViews->addTab(controledFileList, new ColorTab(tr("Controlled files")));
   m_pTools->addWhatsThisText(controledFileList, m_pTools->cvsFile);

   connect(controledFileList, SIGNAL(showFileMenu(bool)),
	 SLOT(showFileMenu(bool)));
   connect(controledFileList, SIGNAL(doubleClickedFile (QString)),
	 SLOT(editFile (QString)));
   connect(controledFileList, SIGNAL(dragStarted()),
	 SLOT(dragStarted()));
   connect(controledFileList, SIGNAL(dragDone()),
	 SLOT(dragDone()));

   /** Add non-cvs file view page to the main window */
   NonCvsListView * nonControledFileList = new NonCvsListView(filePopupMenu, m_pFileViews);
   m_pFileViews->addTab(nonControledFileList, new ColorTab(tr("Non-controlled files")));
   m_pTools->addWhatsThisText(nonControledFileList, m_pTools->nonCvsFile);

   connect(nonControledFileList, SIGNAL(showFileMenu(bool)),
	 SLOT(showFileMenu(bool)));
   connect(nonControledFileList, SIGNAL(itemCount(int)),
	 SLOT(updateColor(int)));
   connect(nonControledFileList, SIGNAL(doubleClickedFile(QString)),
	 SLOT(editFile(QString)));
   connect(nonControledFileList, SIGNAL(dragStarted()),
	 SLOT(dragStarted()));
   connect(nonControledFileList, SIGNAL(dragDone()),
	 SLOT(dragDone()));

   /** Add cvsignore file view page to the main window */
   CvsIgnoreListView * ignoreFileList = new CvsIgnoreListView(filePopupMenu, m_pFileViews);
   ignoreFileList->setFocusPolicy(QWidget::StrongFocus);
   m_pFileViews->addTab(ignoreFileList,
	 new ColorTab(tr("Ignored files")));
   m_pTools->addWhatsThisText(ignoreFileList, m_pTools->cvsIgnoreFile);

   connect(ignoreFileList, SIGNAL(showFileMenu(bool)),
	 SLOT(showFileMenu(bool)));
   connect(ignoreFileList, SIGNAL(doubleClickedFile(QString)),
	 SLOT(editFile(QString)));
   connect(ignoreFileList, SIGNAL(dragStarted()),
	 SLOT(dragStarted()));
   connect(ignoreFileList, SIGNAL(dragDone()),
	 SLOT(dragDone()));

   /**
    * Connect change of selected view to state setting of menuBar() after
    * all views are created, or pointers might be NULL
    */
   connect(m_pFileViews, SIGNAL(currentChanged(QWidget *)),
	 m_pFileViews->currentPage(), SLOT(selectionChanged()));


   //initialize static Directory values
   Directory::initialize(this,m_pWorkBenchTree,m_pFileViews,filePopupMenu,controledFileList,nonControledFileList,ignoreFileList);

   /** Create the message pane widget on the main window */
   m_pMessages = new CCvsOutput(&m_pOutputMenu, m_pVertSplitter, "message log");
   m_pMessages->setFocusPolicy(QWidget::StrongFocus);
   m_pMessages->setReadOnly(true);
   m_pMessages->setWordWrap(QTextEdit::NoWrap);
   m_pTools->addWhatsThisText(m_pMessages, m_pTools->status);
   connect(m_pMessages, SIGNAL(fillOutputMenu(int)), SLOT(fillOutputPopupMenu(int)));

   if (!Font::g_message.isEmpty()) {
      QFont f;
      if (f.fromString(Font::g_message)) m_pMessages->setFont(f);
      else Font::g_message = QString::null;
   }

   connect(&m_timer, SIGNAL(timeout()), SLOT(slot_poll()));
   connect(&m_updateTimer, SIGNAL(timeout()), SLOT(slot_autoUpdate()));

   /** preset initial menu state on startup */
   m_pTools->setEnabled(false);
   m_pTools->m_pStopAction->setEnabled(true);
   connect(&m_showStopTimer, SIGNAL(timeout()), SLOT(slot_enableStopButton()) );

   //set state
   setMenuItemState(FALSE);

   //fill static menues
   fillProjectMenu();
   fillDirMenu(TRUE);
   fillFileMenu(TRUE);
   fillConfigMenu();
   fillHelpMenu();
}

//----------------------------------------------------------------------------

void ApplicationWindow::customizeDirPopupMenu() {

   CustomizeDialogImpl *dlg = new CustomizeDialogImpl(allDirPopupMenuItems, dirPopupMenuItems, 
	 m_pTools->whatsThisIconSet(),
	 this, "Customize Menu", true,
	 LookAndFeel::g_modalF);

   if (dlg->exec()) {
      dirPopupMenu->clear();
      DIRPOPUPMENU="";
 
      unsigned int i;
      for( i = 0; i<dirPopupMenuItems->size(); i++ ) {
	 insertMenuItem(dirPopupMenu,dirPopupMenuItems->at(i));
	 QString hlp;
	 hlp.setNum(dirPopupMenuItems->at(i)->getInstanceNum());
	 DIRPOPUPMENU += hlp+":";
      }
   }
   delete dlg;
}

//----------------------------------------------------------------------------

void ApplicationWindow::customizeFilePopupMenu() {
   CustomizeDialogImpl *dlg = new CustomizeDialogImpl(allFilePopupMenuItems, 
	 filePopupMenuItems, 
	 m_pTools->whatsThisIconSet(),	
	 this, "Customize Menu", true,
	 LookAndFeel::g_modalF);

   if (dlg->exec()) {
      filePopupMenu->clear();
      FILEPOPUPMENU="";
        
      unsigned int i;
      for( i = 0; i<filePopupMenuItems->size(); i++ ) {
	 insertMenuItem(filePopupMenu,filePopupMenuItems->at(i));
	 QString hlp;
	 hlp.setNum(filePopupMenuItems->at(i)->getInstanceNum());
	 FILEPOPUPMENU += hlp+":";
      }
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_customizeKeyShortcuts() {

   CustomizeShortcutsDialogImpl * dlg = new CustomizeShortcutsDialogImpl(
									 m_pTools->whatsThisIconSet(),
									 this);
   if (dlg->exec() ) {
    
   }
   if (m_pWorkBenchTree->currentItem()) setDirMenu(TRUE);
   fillProjectMenu();
   fillConfigMenu();
   fillHelpMenu();
   delete dlg;
}

//----------------------------------------------------------------------------

int ApplicationWindow::insertMenuItem(Menu *menu, MenuItem *item) {
   MenuItem * child = item->getFirstChild();
   if (child) {
      int id;
      Menu * subMenu = menu->addSubMenu(item->caption(),id);
      while (child) {
	 insertMenuItem(subMenu, child);
	 child = item->getNextChild();
      }
      return id;

   } else {
      return insertMenuItem( menu, item, item->getEnabled() );
   }
}

//----------------------------------------------------------------------------

int ApplicationWindow::insertMenuItem(Menu *menu, const MenuItem *item, bool enabled) {
   //printf("\ninsert: %s, slot: %s, accel: %i, id: %i, index: %i\n",item->caption().ascii(), item->slot(), item->accel(), item->id(), item->index());
 
   if (!item) {
      //it should never happen, but the release version under w98
      //put zero item into this function
      qDebug("item == 0 should never happen, check it!!!");    
      return -1;
   }

   if (item == M_SEPARATOR) {
      int id = menu->insertSeparator();
      if (!enabled) menu->setItemEnabled(id,false);
      return id;
   }
   else {
      int id = item->insertInto(menu, this);
      if (!enabled) menu->setItemEnabled(id,false);
      return id;
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_customizeOpenFileMapping() {
   customizeFileMapping(Mapping::openFileAppList,tr("Open file mapping"));
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_customizeViewFileMapping() {
   customizeFileMapping(Mapping::viewFileAppList,tr("View file mapping"));
}

//----------------------------------------------------------------------------

void ApplicationWindow::customizeFileMapping(Mapping::FileAppList &list, QString caption) {

   Mapping::FileAppList dlgList = list;
   FileMapperDialogImpl * dlg = new FileMapperDialogImpl(dlgList, caption, 
	 m_pTools->whatsThisIconSet(),
	 this, "FileMapperDlg", true,
	 LookAndFeel::g_modalF | Qt::WDestructiveClose);

   if (dlg->exec()) {
      list = dlgList;
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::updateColor( int ct ) {
   ColorTab * ctab = (ColorTab *)m_pFileViews->getColorTabBar()->tabAt(1);
   if( ct == 0 ) {
      if (ctab->hasColor()) {
	 ctab->resetColor();
	 m_pFileViews->getColorTabBar()->update();
      }
   } else {
      if (!ctab->hasColor()) {
	 QColor C( 255, 0, 0 );
	 ctab->setColor( C );
	 m_pFileViews->getColorTabBar()->update();
      }
   }
}

//----------------------------------------------------------------------------

/**
 * Add a new project to the workbench
 */
void ApplicationWindow::slot_addProject()
{
   blockGUI(true);
   addProject();
   blockGUI(false);
}

//----------------------------------------------------------------------------

/**
 * Add a new project to the workbench via drag'n drop
 */
void ApplicationWindow::slot_dropProject(const QString file) {
   bool res = showYesNoQuestion( tr("Checkout Info"),
	 tr("Add this project to workbench:")+"\n"+
	 file.mid(file.findRev("/")+1)+"?" );
   if (res) {
      blockGUI(true);
      addNewProject(file,TRUE);
      blockGUI(false);
   }
}

//----------------------------------------------------------------------------

Directory* ApplicationWindow::getSelectedDir(const bool bShowWarning /*= true*/)
{
   Directory * dir;
   if ( (dir = static_cast<Directory *>(m_pWorkBenchTree->selectedItem())) ) {
      return dir;
   } else {
     if (bShowWarning) showWarning(tr("Warning"), tr("There is no directory selected"));
      return NULL;
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_removeCvsBackedFiles() {
   DirBase * dir;
   if (!(dir = getSelectedDir())) {
      return;
   }
   QString filter = ".#*";
   if (!dir->recRemoveNonControlledFiles(filter)) {
      showWarning( tr("Remove .#xxx files"), tr("File:")+"\n"+filter+"\n"+tr("can not be removed, check permissions")+"\n"+tr("exiting"));
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_about()
{
   QString versionString(tr("LinCVS ") + LC_VERSION);
   AboutDialogImpl AboutDlg(this, versionString);   
   AboutDlg.exec();
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_help()
{
   QDialog*dlg = new QDialog(this, 0, true,   //modal
	 LookAndFeel::g_modalF);

   dlg->setCaption(tr("Help"));
   
   QBoxLayout *layer = new QVBoxLayout(dlg, 5);
   QTextEdit *e = new QTextEdit(dlg);
   e->setReadOnly(true);
   layer->addWidget(e);

   QWidget *backGround = new QWidget(dlg);
   layer->addWidget(backGround);

   QBoxLayout *buttonLayer = new QHBoxLayout(backGround, 3);
   buttonLayer->addStretch();
   
   QPushButton *ok = new QPushButton(backGround);
   ok->setText(tr("Close"));
   buttonLayer->addWidget(ok);
   buttonLayer->addStretch();
   ok->setFocus();
   
   connect(ok, SIGNAL(clicked()), dlg, SLOT(accept()));
   dlg->resize(600, 300);

   QFile f;
   QString line, filename, helpDir, helpDirDef;
   
   QString lang = QString(QTextCodec::locale()).left(2);  //maybe it's useful without left(2)

   helpDir    = APPDIR + "/" + LC_HELP_DIR + "/" + LC_TRANSLATIONS_SUBDIR + "/" + lang;
   helpDirDef = APPDIR + "/" + LC_HELP_DIR;

   filename = helpDir + tr("/README.html");
   f.setName(filename);
   e->setTextFormat(RichText);
   
   //try to open the translated html file
   if(!f.open(IO_ReadOnly)) {//file is NOT readable
      filename = helpDir + tr("/README");
      f.setName(filename);
      e->setTextFormat(PlainText);
      
      //try to open the translated plain text file
      if(!f.open(IO_ReadOnly)) {//file is NOT readable
         filename = helpDirDef + "/README.html";
         f.setName(filename);
         e->setTextFormat(RichText);

         //try to open the default html file
         if(!f.open(IO_ReadOnly)) {//file is NOT readable
            filename = helpDirDef + "/README";
            f.setName(filename);
            e->setTextFormat(PlainText);

            //try to open the default plain text file
            if(!f.open(IO_ReadOnly)) {//file is NOT readable
               QString error = tr("File not found: %1");
               showWarning(tr("Error"), error.arg(filename));
               return;
            }
         }
      }
   }
       
       
   QTextStream stream( &f );
   e->setText( stream.read() );
   f.close();

   dlg->exec();
   delete dlg;
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_aboutQt()
{
   QMessageBox::aboutQt( this, "Qt..." );
}

//----------------------------------------------------------------------------

void ApplicationWindow::showInfo( const QString &caption, const QString &txt) {
   bool block = GuiEventBlocker::isBlocked();
   if (block) GuiEventBlocker::setBlocking(false);

   QWidget * parentWidget = qApp->activeWindow();
   if (!parentWidget && sender() && sender()->isWidgetType()) parentWidget = (QWidget *)sender();
   if (!parentWidget) parentWidget = this;

   QString cap = LC_APPNAME;
   cap += " - " + caption;
   QMessageBox::information (parentWidget, cap, txt,
	 QMessageBox::Ok,
	 QMessageBox::NoButton);
   if (block) GuiEventBlocker::setBlocking(true);
}

//----------------------------------------------------------------------------

void ApplicationWindow::showWarning( const QString &caption, const QString &txt) {
   bool block = GuiEventBlocker::isBlocked();
   if (block) GuiEventBlocker::setBlocking(false);

   QWidget * parentWidget = qApp->activeWindow();
   if (!parentWidget && sender() && sender()->isWidgetType()) parentWidget = (QWidget *)sender();
   if (!parentWidget) parentWidget = this;

   QString cap = LC_APPNAME;
   cap += " - " + caption;
   QMessageBox::warning (parentWidget, cap, txt,
	 QMessageBox::Ok,
	 QMessageBox::NoButton);
   if (block) GuiEventBlocker::setBlocking(true);
}

//----------------------------------------------------------------------------

bool ApplicationWindow::showYesNoQuestion( const QString &caption, const QString &txt) {
   bool block = GuiEventBlocker::isBlocked();
   if (block) GuiEventBlocker::setBlocking(false);

   QWidget * parentWidget = qApp->activeWindow();
   if (!parentWidget) parentWidget = this;

   QString cap = LC_APPNAME;
   cap += " - " + caption;
   bool retval = (QMessageBox::warning (parentWidget, cap, txt,
	       QMessageBox::Yes,
	       QMessageBox::No)
	 == QMessageBox::Yes);

   if (block) GuiEventBlocker::setBlocking(true);
   return retval;
}

//----------------------------------------------------------------------------

bool ApplicationWindow::showOkCancelQuestion( const QString &caption, const QString &txt) {
   bool block = GuiEventBlocker::isBlocked();
   if (block) GuiEventBlocker::setBlocking(false);

   QWidget * parentWidget = qApp->activeWindow();
   if (!parentWidget) parentWidget = this;

   QString cap = LC_APPNAME;
   cap += " - " + caption;
   bool retval = (QMessageBox::warning (parentWidget, cap, txt,
	       QMessageBox::Ok,
	       QMessageBox::Cancel)
	 == QMessageBox::Ok);
   if (block) GuiEventBlocker::setBlocking(true);
   return retval;
}

//----------------------------------------------------------------------------

bool ApplicationWindow::showVerifyDialog(const QString &caption, const QString &txt) {
   bool retVal = true;
   if (Dialog::g_showWarning) {
      QWidget * parentWidget = qApp->activeWindow();
      if (!parentWidget) parentWidget = this;
      QString cap = LC_APPNAME;
      cap += " - " + caption;
      VerifyDialog * dlg = new VerifyDialog(parentWidget);
      dlg->setCaption(cap);
      dlg->TextLabel->setText(txt);
      dlg->adjustSize();
      retVal = (dlg->exec() == QDialog::Accepted);
      if (!dlg->VerifyCheckBox->isChecked()) Dialog::g_showWarning = false;
   }
   return retVal;
}

//----------------------------------------------------------------------------

bool ApplicationWindow::checkFilesSelected() {
   Directory * dir;
   if ( (dir = getSelectedDir()) && !dir->getSelectedFiles().isEmpty()) {
      return true;
   }
   return false;
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_dirClicked(DirBase * dir) {
   if (!m_started) return;
   checkInProgress(true);
   dir->activateItem(false);
   checkInProgress(false);
}

//----------------------------------------------------------------------------

void ApplicationWindow::dragStarted() {
   slot_stopStatusTimer();
   m_onDrag = true;
}

//----------------------------------------------------------------------------

void ApplicationWindow::dragDone() {
   m_onDrag = false;
   slot_checkStatus();
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_deleteObject(QObject * obj) {
   m_delObjStack.push(obj);
   QTimer::singleShot(0,this,SLOT(slot_deletePendingObjects()));
}

//----------------------------------------------------------------------------

void ApplicationWindow::slot_deletePendingObjects() {
   QObject * obj;
   while (!m_delObjStack.isEmpty()) {
      obj = m_delObjStack.pop();
      assert(obj);
      delete obj;
   }
}

//----------------------------------------------------------------------------
