/****************************************************************************
 **
 ** Copyright (C) 2001-2005 Tilo Riemer <riemer@lincvs.org> and
 **                         Frank Hemer <frank@hemer.org>
 **
 **
 **----------------------------------------------------------------------------
 **
 **----------------------------------------------------------------------------
 **
 ** LinCVS is available under two different licenses:
 **
 ** If LinCVS is linked against the GPLed version of Qt 
 ** LinCVS is released under the terms of GPL also.
 **
 ** If LinCVS is linked against a nonGPLed version of Qt 
 ** LinCVS is released under the terms of the 
 ** LinCVS License for non-Unix platforms (LLNU)
 **
 **
 ** LinCVS License for non-Unix platforms (LLNU):
 **
 ** Redistribution and use in binary form, without modification, 
 ** are permitted provided that the following conditions are met:
 **
 ** 1. Redistributions in binary form must reproduce the above copyright
 **    notice, this list of conditions and the following disclaimer in the
 **    documentation and/or other materials provided with the distribution.
 ** 2. It is not permitted to distribute the binary package under a name
 **    different than LinCVS.
 ** 3. The name of the authors may not be used to endorse or promote
 **    products derived from this software without specific prior written
 **    permission.
 ** 4. The source code is the creative property of the authors.
 **    Extensions and development under the terms of the Gnu Public License
 **    are limited to the Unix platform. Any distribution or compilation of 
 **    the source code against libraries licensed other than gpl requires 
 **    the written permission of the authors.
 **
 **
 ** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
 ** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 ** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 ** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
 ** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 ** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
 ** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 ** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 ** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 ** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 ** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **
 **
 **
 ** LinCVS License for Unix platforms:
 **
 ** This program is free software; you can redistribute it and/or modify 
 ** it under the terms of the GNU General Public License as published by 
 ** the Free Software Foundation; either version 2 of the License, or 
 ** (at your  option) any later version.  This program is distributed in 
 ** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
 ** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 ** PARTICULAR PURPOSE.
 **
 ** See the GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software Foundation,
 ** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **
 *****************************************************************************/

#include "Merge.h"

#include <qapplication.h>
#include <assert.h>

#include "globals.h"
#include "MergeDialogImpl.h"

Merge * Merge::mergeFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, QStringList * currentTagList)
{
   return setup(whatsThisIconSet, parent, workBench, false, currentTagList);
}

Merge * Merge::mergeDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, QStringList * currentTagList)
{
   return setup(whatsThisIconSet, parent, workBench, true, currentTagList);
}

Merge * Merge::setup(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, bool isDir, QStringList * currentTagList) {

   DirBase * dir = workBench->selectedItem();
   if (!dir) return NULL;

   if (dir->getType() == DirBase::Cvs) {

      Merge * p = new Merge(whatsThisIconSet, parent, workBench, dir, isDir);

      if (!isDir) {
	 p->m_filesList = p->m_dir->getSelectedFiles();
	 if (p->m_filesList.isEmpty()) {
	    delete p;
	    return NULL;
	 }
      }

      MergeDialogImpl * dlg = new MergeDialogImpl(workBench, whatsThisIconSet,
	    parent, QString(isDir ? QString("") : masqWs(p->m_filesList.join("\" \""))), 
	    currentTagList,
	    dir->topControlledDir()->relativeName());
      if (!isDir) {
	 dlg->hideGetNewDirs();
      }
      p->m_dlg = dlg;
      connect(dlg,SIGNAL(accept()),p,SLOT(acceptCvs()));
      connect(dlg,SIGNAL(reject()),p,SLOT(reject()));
      dlg->show();

      return p;
      
   } else {
      qDebug("Merge: Method not implemented");
   }
   return NULL;
}

Merge::Merge(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, DirBase * dir, bool isDir)
   :  m_whatsThisIconSet(whatsThisIconSet),
      m_parent(parent),
      m_workBench(workBench),
      m_dir(dir),
      m_dlg(NULL),
      m_isDir(isDir)
{
   connect(this,SIGNAL(deleteObject(QObject *)),parent,SLOT(slot_deleteObject(QObject *)));
   connect(this,SIGNAL(checkInProgress(bool)),parent,SLOT(checkInProgress(bool)));
   connect(this,SIGNAL(showWarning(const QString &,const QString &)),parent,SLOT(showWarning(const QString &,const QString &)));
}

Merge::~Merge() {
   if (m_dlg) delete m_dlg;
   m_dlg = NULL;
}

void Merge::acceptCvs() {

   if (!m_workBench->validate(m_dir)) {
      emit showWarning(tr("Warning"),tr("The former selected dir has\nbeen modified or removed,\naborting"));
      reject();
      return; //show warning
   }

   MergeDialogImpl * dlg = static_cast<MergeDialogImpl*>(m_dlg);
   assert(dlg);
   m_query = dlg->isQuery();
   bool getNewDirs = dlg->isGetNewDirs();
   QString firstTag = dlg->getFirstTag();
   QString secondTag = dlg->getSecondTag();

   delete m_dlg;
   m_dlg = NULL;

   int cmd = -1;
   QString command = "";
   if (!bRWPermission) command += "-r ";
   command += CvsOptions::cmprStr();
   if (m_query) command += " -n";
   command += " update ";
   if (!m_query && bKeywordSuppression) command += "-kk ";
   if (m_isDir) {
      if (CvsOptions::g_bPruneDirs) {
	 command += "-P ";
	 cmd = CVS_MERGE_DIR_CMD;
      } else {
	 cmd = CVS_MERGE_DIR_NO_PRUNE_CMD;
      }
      command += getNewDirs ? "-d " : "";
   } else {
      cmd = CVS_MERGE_FILES_CMD;
   }
   command += "-j ";
   command += firstTag;
   if (!secondTag.isEmpty()) command += " -j " + secondTag;

   QString files;
   if (!m_filesList.isEmpty()) {
      QStringList list = m_filesList;
      masqQuoteMarks(&list);
      files = masqWs(list.join("\" \""));
   }
   QString dir = m_dir->fullName();
   QString topModule = m_dir->topControlledDir()->relativeName();
   
   callInteractive( topModule, dir, command,
	 files, cmd,
	 ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
	 NOROOT);
}

void Merge::reject() {
   emit deleteObject(this);
}

void Merge::cvsCallStarted() {
   QApplication::setOverrideCursor(Qt::waitCursor);
}

void Merge::cvsCallFinished() {
   QApplication::restoreOverrideCursor();
}

void Merge::afterCall(int cmd,CvsBuffer * buf,bool failed) {
   if (failed) {
      cvsCallFinished();
      reject();
      return;
   }
   assert(m_dir);
   switch( cmd) {
      case CVS_MERGE_FILES_CMD: {
	 m_dir->parseCallResult( buf, m_query ? CVS_QUERY_UPDATE_CMD : CVS_UPDATE_FILES_CMD);
	 emit checkInProgress(true);
	 m_dir->postCallCheck(DirBase::Controled|(m_query ? DirBase::Force : 0));
	 emit checkInProgress(false);
	 break;
      }
      case CVS_MERGE_DIR_NO_PRUNE_CMD:
	 m_dir->parseCallResult( buf, m_query ? CVS_QUERY_UPDATE_ALL_CMD : CVS_UPDATE_DIR_CMD);
	 emit checkInProgress(true);
	 m_dir->postCallCheck(DirBase::Controled|DirBase::Tree|(m_query ? DirBase::Force : 0));
	 emit checkInProgress(false);
	 break;
      case CVS_MERGE_DIR_CMD: {
	 m_dir->parseCallResult( buf, m_query ? CVS_QUERY_UPDATE_ALL_CMD : CVS_UPDATE_DIR_CMD);
	 emit checkInProgress(true);
	 m_dir->postCallCheck(DirBase::Controled|DirBase::Tree|(m_query ? DirBase::Force : 0));
	 emit checkInProgress(false);
	 break;
      }
      default: {
	 qDebug("Merge::afterCall: Unknown cmd");
      }
   }
   cvsCallFinished();
   reject();
}
